/*
 * Decompiled with CFR 0.152.
 */
package marytts.util.math;

import Jama.Matrix;
import marytts.util.math.MathUtils;

public class Polynomial {
    public final double[] coeffs;

    public Polynomial(int order) {
        this.coeffs = new double[order + 1];
    }

    public Polynomial(double[] coeffs) {
        this.coeffs = coeffs;
    }

    public int getOrder() {
        return this.coeffs.length - 1;
    }

    public void copyCoeffs(Polynomial other) {
        if (other.coeffs.length != this.coeffs.length) {
            throw new IllegalArgumentException("Polynomial orders differ: I have " + this.getOrder() + ", other has " + other.getOrder());
        }
        System.arraycopy(other.coeffs, 0, this.coeffs, 0, this.coeffs.length);
    }

    public double[] generatePolynomialValues(int numSamples, double a, double b) {
        return Polynomial.generatePolynomialValues(this.coeffs, numSamples, a, b);
    }

    public double getValueAt(double x) {
        return Polynomial.getValueAt(this.coeffs, x);
    }

    public double polynomialDistance(Polynomial other) {
        return Polynomial.polynomialDistance(this.coeffs, other.coeffs);
    }

    public double polynomialSquaredDistance(Polynomial other) {
        return Polynomial.polynomialSquaredDistance(this.coeffs, other.coeffs);
    }

    public static double[] fitPolynomial(double[] data, int order) {
        if (data == null) {
            throw new NullPointerException("Null data");
        }
        if (order < 0) {
            throw new IllegalArgumentException("Polynomial order < 0 not supported");
        }
        double[][] A = new double[data.length][order + 1];
        double[][] b = new double[data.length][1];
        for (int i = 0; i < A.length; ++i) {
            if (Double.isNaN(data[i])) {
                b[i][0] = 0.0;
                for (int j = 0; j <= order; ++j) {
                    A[i][j] = 0.0;
                }
                continue;
            }
            b[i][0] = data[i];
            double t = (double)i / (double)data.length;
            for (int j = 0; j <= order; ++j) {
                A[i][j] = Math.pow(t, order - j);
            }
        }
        try {
            Matrix x = new Matrix(A).solve(new Matrix(b));
            double[] coeffs = new double[order + 1];
            for (int j = 0; j <= order; ++j) {
                coeffs[j] = x.get(j, 0);
            }
            return coeffs;
        }
        catch (RuntimeException re) {
            return null;
        }
    }

    public static double[] generatePolynomialValues(double[] coeffs, int numSamples, double a, double b) {
        if (numSamples <= 0) {
            throw new IllegalArgumentException("Need positive number of samples");
        }
        if (a >= b) {
            throw new IllegalArgumentException("Not a valid interval: [" + a + "," + b + "[");
        }
        double[] pred = new double[numSamples];
        double step2 = (b - a) / (double)numSamples;
        double t = a;
        for (int i = 0; i < numSamples; ++i) {
            pred[i] = Polynomial.getValueAt(coeffs, t);
            t += step2;
        }
        return pred;
    }

    public static double getValueAt(double[] coeffs, double x) {
        if (coeffs == null) {
            throw new NullPointerException("Received null coeffs");
        }
        if (coeffs.length == 0) {
            throw new IllegalArgumentException("Received empty coeffs");
        }
        double val = 0.0;
        int order = coeffs.length - 1;
        for (int j = 0; j <= order; ++j) {
            val += coeffs[j] * Math.pow(x, order - j);
        }
        return val;
    }

    public static Polynomial mean(Polynomial[] p) {
        int order = p[0].getOrder();
        double[] meanCoeffs = new double[order + 1];
        int k = 0;
        while (k <= order) {
            for (int i = 0; i < p.length; ++i) {
                int n = k;
                meanCoeffs[n] = meanCoeffs[n] + p[i].coeffs[k];
            }
            int n = k++;
            meanCoeffs[n] = meanCoeffs[n] / (double)p.length;
        }
        return new Polynomial(meanCoeffs);
    }

    public static double[] mean(double[][] p) {
        int order = p[0].length - 1;
        double[] meanCoeffs = new double[order + 1];
        int k = 0;
        while (k <= order) {
            for (int i = 0; i < p.length; ++i) {
                int n = k;
                meanCoeffs[n] = meanCoeffs[n] + p[i][k];
            }
            int n = k++;
            meanCoeffs[n] = meanCoeffs[n] / (double)p.length;
        }
        return meanCoeffs;
    }

    public static float[] mean(float[][] p) {
        int order = p[0].length - 1;
        float[] meanCoeffs = new float[order + 1];
        int k = 0;
        while (k <= order) {
            for (int i = 0; i < p.length; ++i) {
                int n = k;
                meanCoeffs[n] = meanCoeffs[n] + p[i][k];
            }
            int n = k++;
            meanCoeffs[n] = meanCoeffs[n] / (float)p.length;
        }
        return meanCoeffs;
    }

    public static double variance(Polynomial[] p, Polynomial mean) {
        if (p.length <= 1) {
            return 0.0;
        }
        double variance = 0.0;
        for (int i = 0; i < p.length; ++i) {
            variance += Polynomial.polynomialSquaredDistance(mean.coeffs, p[i].coeffs);
        }
        return variance / (double)(p.length - 1);
    }

    public static double variance(double[][] p, double[] mean) {
        if (p.length <= 1) {
            return 0.0;
        }
        double variance = 0.0;
        for (int i = 0; i < p.length; ++i) {
            variance += Polynomial.polynomialSquaredDistance(mean, p[i]);
        }
        return variance / (double)(p.length - 1);
    }

    public static double variance(float[][] p, float[] mean) {
        if (p.length <= 1) {
            return 0.0;
        }
        double variance = 0.0;
        for (int i = 0; i < p.length; ++i) {
            variance += Polynomial.polynomialSquaredDistance(mean, p[i]);
        }
        return variance / (double)(p.length - 1);
    }

    public static double polynomialDistance(double[] coeffs1, double[] coeffs2) {
        if (coeffs1 == null || coeffs2 == null) {
            throw new NullPointerException("Received null argument");
        }
        if (coeffs1.length != coeffs2.length) {
            throw new IllegalArgumentException("Can only compare polynomials with same order");
        }
        double dist = 0.0;
        int order = coeffs1.length - 1;
        for (int i = 0; i <= order; ++i) {
            dist += (coeffs1[order - i] - coeffs2[order - i]) / (double)(i + 1);
        }
        return Math.abs(dist);
    }

    public static double polynomialDistance(float[] coeffs1, float[] coeffs2) {
        if (coeffs1 == null || coeffs2 == null) {
            throw new NullPointerException("Received null argument");
        }
        if (coeffs1.length != coeffs2.length) {
            throw new IllegalArgumentException("Can only compare polynomials with same order");
        }
        double dist = 0.0;
        int order = coeffs1.length - 1;
        for (int i = 0; i <= order; ++i) {
            dist += ((double)coeffs1[order - i] - (double)coeffs2[order - i]) / (double)(i + 1);
        }
        return Math.abs(dist);
    }

    public static double polynomialSquaredDistance(double[] coeffs1, double[] coeffs2) {
        if (coeffs1 == null || coeffs2 == null) {
            throw new NullPointerException("Received null argument");
        }
        if (coeffs1.length != coeffs2.length) {
            throw new IllegalArgumentException("Can only compare polynomials with same order");
        }
        int order = coeffs1.length - 1;
        double[] a = new double[coeffs1.length];
        for (int i = 0; i < a.length; ++i) {
            a[i] = coeffs1[order - i] - coeffs2[order - i];
        }
        return Polynomial.integrateSquared(order, a);
    }

    public static double polynomialSquaredDistance(float[] coeffs1, float[] coeffs2) {
        if (coeffs1 == null || coeffs2 == null) {
            throw new NullPointerException("Received null argument");
        }
        if (coeffs1.length != coeffs2.length) {
            throw new IllegalArgumentException("Can only compare polynomials with same order");
        }
        int order = coeffs1.length - 1;
        double[] a = new double[coeffs1.length];
        for (int i = 0; i < a.length; ++i) {
            a[i] = coeffs1[order - i] - coeffs2[order - i];
        }
        return Polynomial.integrateSquared(order, a);
    }

    private static double integrateSquared(int order, double[] a) {
        double dist = 0.0;
        dist += a[0] * a[0];
        if (order == 0) {
            return dist;
        }
        dist += a[0] * a[1] + a[1] * a[1] / 3.0;
        if (order == 1) {
            return dist;
        }
        dist += 0.6666666666666666 * a[0] * a[2] + a[1] * a[2] / 2.0 + a[2] * a[2] / 5.0;
        if (order == 2) {
            return dist;
        }
        dist += a[0] * a[3] / 2.0 + 0.4 * a[1] * a[3] + a[2] * a[3] / 3.0 + a[3] * a[3] / 7.0;
        if (order == 3) {
            return dist;
        }
        dist += 0.4 * a[0] * a[4] + a[1] * a[4] / 3.0 + 0.2857142857142857 * a[2] * a[4] + a[3] * a[4] / 4.0 + a[4] * a[4] / 9.0;
        if (order == 4) {
            return dist;
        }
        throw new IllegalArgumentException("Order greater than 4 not supported");
    }

    public static double polynomialPearsonProductMomentCorr(double[] coeffs1, double[] coeffs2) {
        if (coeffs1 == null || coeffs2 == null) {
            throw new NullPointerException("Received null argument");
        }
        if (coeffs1.length != coeffs2.length) {
            throw new IllegalArgumentException("Can only compare polynomials with same order");
        }
        double[] contour1 = Polynomial.generatePolynomialValues(coeffs1, 25, 0.0, 1.0);
        double[] contour2 = Polynomial.generatePolynomialValues(coeffs2, 25, 0.0, 1.0);
        double meanF01 = MathUtils.mean(contour1);
        double meanF02 = MathUtils.mean(contour2);
        double diffF01Sum = 0.0;
        double diffF02Sum = 0.0;
        double diffProductSum = 0.0;
        for (int i = 0; i < contour1.length; ++i) {
            double diffF01 = contour1[i] - meanF01;
            double diffF02 = contour2[i] - meanF02;
            double diffProduct = diffF01 * diffF02;
            diffF01Sum += diffF01 * diffF01;
            diffF02Sum += diffF02 * diffF02;
            diffProductSum += diffProduct;
        }
        return 1.0 - diffProductSum / Math.sqrt(diffF01Sum * diffF02Sum);
    }
}

