package org.gcube.common.homelibrary.jcr.repository;

import static org.gcube.resources.discovery.icclient.ICFactory.clientFor;
import static org.gcube.resources.discovery.icclient.ICFactory.queryFor;

import java.security.Principal;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;
import javax.jcr.Value;
import javax.jcr.security.AccessControlList;
import javax.jcr.security.AccessControlManager;
import javax.jcr.security.AccessControlPolicy;
import javax.jcr.security.AccessControlPolicyIterator;
import javax.jcr.security.Privilege;

import org.apache.jackrabbit.commons.JcrUtils;
import org.apache.jackrabbit.rmi.repository.URLRemoteRepository;
import org.gcube.common.encryption.StringEncrypter;
import org.gcube.common.homelibrary.home.User;
import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.jcr.workspace.usermanager.JCRUserManager;
import org.gcube.common.homelibrary.jcr.workspace.util.Utils;
import org.gcube.common.resources.gcore.ServiceEndpoint;
import org.gcube.common.resources.gcore.ServiceEndpoint.AccessPoint;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.resources.discovery.client.api.DiscoveryClient;
import org.gcube.resources.discovery.client.queries.api.SimpleQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;



public class JCRRepository {

	public static final String HL_NAMESPACE					= "hl:";
	public static final String JCR_NAMESPACE				= "jcr:";
	public static final String REP_NAMESPACE				= "rep:";

	public static final String PATH_SEPARATOR 				= "/";
	public static final String ROOT_WEBDAV					= "/repository/default/";

	//root level
	public static final String HOME_FOLDER 					= "Home";
	public static final String SHARED_FOLDER				= "Share";
	private static final String GCUBE_FOLDER 				= "GCube";

	//home level
	private static final String DOWNLOADS					= "Downloads";
	private static final String SMART_FOLDER 				= "Folders";
	private static final String IN_BOX_FOLDER 				= "InBox";
	private static final String OUT_BOX_FOLDER				= "OutBox";
	private static final String HIDDEN_FOLDER				= "HiddenFolder";


	private static final String NT_FOLDER 					= "nt:folder";
	private static final String NT_HOME						= "nthl:home";
	public static final String 	NT_WORKSPACE_FOLDER 		= "nthl:workspaceItem";
	private static final String NT_ROOT_ITEM_SENT  			= "nthl:rootItemSentRequest";
	private static final String NT_ROOT_FOLDER_BULK_CREATOR = "nthl:rootFolderBulkCreator";

	private static final String SCOPES						= "hl:scopes";

	private static final String nameResource 				= "HomeLibraryRepository";

	private static Repository repository;
	private String portalLogin;

	private static String user;
	private static String pass;
	public static String url;
	private static String webdavUrl;
	public static String serviceName;

	//HL release version
	public static String HLversion;
	private static String version;
	private static String minorVersion;
	private static String revisionVersion;


	private static Logger logger = LoggerFactory.getLogger(JCRRepository.class);


	private static synchronized void initializeRepository() throws InternalErrorException {

		if(repository != null)
			return;


		String callerScope = ScopeProvider.instance.get();

		try {

			if (callerScope==null) throw new IllegalArgumentException("scope is null");

			String rootScope = Utils.getRootScope(callerScope);

			logger.debug("scope for repository creation is "+rootScope+" caller scope is "+callerScope);

			ScopeProvider.instance.set(rootScope);

			SimpleQuery query = queryFor(ServiceEndpoint.class);

			query.addCondition("$resource/Profile/Category/text() eq 'Database' and $resource/Profile/Name eq '"+ nameResource + "' ");

			DiscoveryClient<ServiceEndpoint> client = clientFor(ServiceEndpoint.class);

			List<ServiceEndpoint> resources = client.submit(query);

			if(resources.size() != 0) {	   
				try {
					ServiceEndpoint resource = resources.get(0);

					for (AccessPoint ap:resource.profile().accessPoints()) {

						if (ap.name().equals("JCR")) {

							url = ap.address();
							//							url = "http://node11.d.d4science.research-infrastructures.eu:8080/jackrabbit-webapp-patched-2.4.3";

							user = ap.username();						
							pass = StringEncrypter.getEncrypter().decrypt(ap.password());

							Iterator<org.gcube.common.resources.gcore.ServiceEndpoint.Property> properties = ap.properties().iterator();
							while(properties.hasNext()) {
								org.gcube.common.resources.gcore.ServiceEndpoint.Property p = properties.next();		
								if (p.name().equals("version")){							
									version = p.value();
								} else if (p.name().equals("minorVersion")){							
									minorVersion = p.value();
								} else if (p.name().equals("revisionVersion")){							
									revisionVersion = p.value();
								}
							}	
							HLversion =  version + "." + minorVersion + "." + revisionVersion;
							//							System.out.println("HL VERSION: " + version + "." + minorVersion + "." + revisionVersion);					
						}
						else if (ap.name().equals("WebDav")) {								
							webdavUrl = ap.address();				
						}else if (ap.name().equals("ServiceName")) {								
							serviceName = ap.address();				
						}
					}
				} catch (Throwable e) {
					logger.error("error decrypting resource",e);
				}
			}


			if (user==null || pass==null) throw new InternalErrorException("cannot discover password and username in scope "+callerScope);

			repository = new URLRemoteRepository(url + "/rmi");
			//			repository = JcrUtils.getRepository(url + "/server");

			logger.debug("user is "+user+" password is null?"+(pass==null)+" and repository is null?"+(repository==null));

		} catch (Exception e) {
			throw new InternalErrorException(e);
		}finally{
			if (callerScope!=null)
				ScopeProvider.instance.set(callerScope);
		}

	}


	public static List<String> getHomeNames() throws RepositoryException{
		List<String> homes = new ArrayList<String>();
		Session session = null;
		try{
			try {
				session = getSession();
			} catch (InternalErrorException e) {
				logger.error("error gettin session",e);
				throw new RepositoryException(e);
			}
			Node home = (Node)session.getItem(PATH_SEPARATOR+HOME_FOLDER);
			NodeIterator it = home.getNodes();
			while (it.hasNext()){
				Node node = it.nextNode();
				if (node.getPrimaryNodeType().getName().equals(NT_HOME))
					homes.add(node.getName());
			}
		}finally{
			if (session != null)
				session.logout();
		}
		return homes;
	}



	//	private static void addUserToJCRUserManager(String userId, String userHome) {
	//
	//		GetMethod getMethod = null;
	//		try {
	//
	//			HttpClient httpClient = new HttpClient();            
	//			//			 System.out.println(url);
	//			getMethod =  new GetMethod(url + "/PortalUserManager?userId=" + userId + "&userHome=" +userHome);
	//			httpClient.executeMethod(getMethod);
	//
	//			logger.debug("User set with status code " + getMethod.getResponseBodyAsString());
	//
	//		} catch (Exception e) {
	//			logger.error("User set with error ", e);
	//		} finally {
	//			if(getMethod != null)
	//				getMethod.releaseConnection();
	//		}
	//
	//	}



	public static Session getSession(String user) throws InternalErrorException  {

		initializeRepository();
		//		synchronized (repository) {
		try {
			logger.debug("session of " + user);
			Session session = repository.login( 
					new SimpleCredentials(user, JCRUserManager.getSecurePassword(user).toCharArray()));
			return session;
		} catch (Exception e) {
			throw new InternalErrorException(e);
		}
		//		}
	}

	public static Session getSession() throws InternalErrorException  {
		initializeRepository();
		synchronized (repository) {
			try {
				Session session = repository.login( 
						new SimpleCredentials(user, pass.toCharArray()));
				return session;
			} catch (Exception e) {
				throw new InternalErrorException(e);
			}
		}
	}

	public synchronized static void initialize() throws InternalErrorException {
		logger.debug("Initialize repository");
		initializeRepository();

	}

	public JCRRepository(final User user) throws InternalErrorException {

		portalLogin = user.getPortalLogin();
		logger.info("getHome " + portalLogin);
		try {
			init();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		}

	}

	public static void removeUser(User user) throws Exception {

		Session session = getSession();
		Node node = session.getNode(PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR + user.getPortalLogin());
		node.remove();
		session.save();
	}


	/**
	 * Create folder in /Home/xxx/
	 * @param user
	 * @throws PathNotFoundException
	 * @throws RepositoryException
	 * @throws InternalErrorException
	 */
	public void init() throws PathNotFoundException, RepositoryException, InternalErrorException{

		JCRUserManager um = new JCRUserManager();
		String userVersion = getUserVersion(portalLogin, um);

		logger.info(portalLogin + " - USER VERSION: " + userVersion + " - HL VERSION: " + HLversion);

		if(!HLversion.equals(userVersion)){

			Session session = null;
			try {
				session = getSession();
				Node home = session.getNode("/" + HOME_FOLDER);
				//create user folder es. Home/test.test
				Node userHome;
				if (!exist(home, portalLogin)) {
					userHome = home.addNode(portalLogin, NT_HOME);				
					session.save();

					um.createUser(portalLogin, version+"");
				}
				else
					userHome = home.getNode(portalLogin);	

				//				System.out.println(userHome.getPath());

				//create folder es. Home/test.test/Folders
				if (!exist(userHome, SMART_FOLDER)) 
					userHome.addNode(SMART_FOLDER, NT_FOLDER);					

				if (!exist(userHome, IN_BOX_FOLDER)) 
					userHome.addNode(IN_BOX_FOLDER, NT_ROOT_ITEM_SENT);

				if (!exist(userHome, OUT_BOX_FOLDER)) 
					userHome.addNode(OUT_BOX_FOLDER, NT_ROOT_ITEM_SENT);		

				if (!exist(userHome, HIDDEN_FOLDER))
					userHome.addNode(HIDDEN_FOLDER, NT_FOLDER);					

				if (!exist(userHome, DOWNLOADS))
					userHome.addNode(DOWNLOADS, NT_ROOT_FOLDER_BULK_CREATOR);

				session.save();


				//Workspace folder created in JCRWorkspace

				//				if (!exist(userHome, WORKSPACE_FOLDER)) {
				//					Node wsNode = userHome.addNode(WORKSPACE_FOLDER, NT_WORKSPACE_FOLDER);
				//
				//					this.root = new JCRWorkspaceFolder(this, wsNode, WORKSPACE_FOLDER, "The root");
				//					this.root.save(wsNode);
				//					session.save();
				//				
				//					String wsRootPath = "/" + HOME_FOLDER + "/" +  portalLogin + "/" + WORKSPACE_FOLDER;
				//					System.out.println(wsRootPath);
				//					JCRRepository.setACL(portalLogin, wsRootPath);
				//					System.out.println("ws");
				//				}


			}catch (InternalErrorException e) {
				throw new InternalErrorException(e);
			} finally {
				if (session!=null)
					session.logout();
			}
		}else
			logger.info("skip init in JCRRepository");


	}

	public static void setACL(final String portalLogin , String userHome)
			throws InternalErrorException {


		Session session = null;
		try {

			Repository RMIrepository = new URLRemoteRepository(url + "/rmi");
			session = RMIrepository.login( 
					new SimpleCredentials(user, pass.toCharArray()));

			AccessControlManager accessControlManager = session.getAccessControlManager();
			AccessControlPolicyIterator acpIterator = accessControlManager.getApplicablePolicies(userHome);
			if (acpIterator.hasNext()) {
				logger.debug(" ------------ ACL Present ");
				AccessControlPolicy policy = acpIterator.nextAccessControlPolicy();
				AccessControlList list = (AccessControlList)policy;
				list.addAccessControlEntry(new Principal() {

					@Override
					public String getName() {
						return portalLogin;
					}
				}, new Privilege[]{accessControlManager.privilegeFromName(Privilege.JCR_READ)});

				accessControlManager.setPolicy(userHome, list);
			}

			session.save();
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			if(session != null)
				session.logout();
		}
	}


	//	public void setScope(Session session, String scope) throws RepositoryException {
	//
	//		Node userHome = getUserHome(session);
	//
	//		Property scopes;
	//		try {
	//			scopes = userHome.getProperty(SCOPES);
	//		} catch (PathNotFoundException e) {
	//			String[] values = {scope};
	//			userHome.setProperty(SCOPES, values);
	//			return;
	//		}
	//
	//		for (Value value : scopes.getValues()) {
	//			if(value.getString().equals(scope))
	//				return;
	//		}
	//
	//		String[] newScopes = new String[scopes.getValues().length + 1];
	//		newScopes[0] = scope;
	//		for (int i = 1; i < newScopes.length; i++) {
	//			newScopes[i] = scopes.getValues()[i - 1].getString();
	//		}
	//		userHome.setProperty(SCOPES, newScopes);
	//	}


	public boolean exist(Node parent, String childName) throws RepositoryException {
		try {
			parent.getNode(childName);
		} catch (PathNotFoundException e) {
			logger.info(childName + " does not exist");
			return false;
		} 
		return true;
	}

	/*
	public boolean existChild(Node father, String child){

		try {
			father.getNode(child);
		} catch (RepositoryException e) {
			return false;
		}
		return true;
	}*/

	public List<String> listScopes() throws RepositoryException, InternalErrorException {

		List<String> list = new LinkedList<String>();

		Session session = getSession();
		try {
			Node userHome = session.getNode(PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR + portalLogin);
			Property scopes = userHome.getProperty(SCOPES);

			for (Value value  : scopes.getValues()) {
				list.add(value.getString());
			}
			return list;

		} catch (PathNotFoundException e) {
			return new LinkedList<String>();
		} finally {
			if (session != null)
				session.logout();
		}
	}


	/**
	 * Get gCube root
	 * @param session
	 * @return gCube root
	 * @throws RepositoryException
	 */
	public static Node getGCubeRoot(Session session) throws RepositoryException {

		//		logger.info("getGCubeRoot");
		Node gcubeRoot = null;
		try{
			gcubeRoot = session.getNode(PATH_SEPARATOR + GCUBE_FOLDER);
		}catch (Exception e) {
			session.getRootNode().addNode(GCUBE_FOLDER, NT_FOLDER);
			session.save();
		}

		return gcubeRoot;
	}

	/**
	 * get SharedRoot
	 * @param session
	 * @return SharedRoot
	 * @throws RepositoryException
	 */
	public static Node getSharedRoot(Session session) throws RepositoryException {
		logger.info("getSharedRoot");
		Node sharedNode = null;
		try{
			sharedNode = session.getNode(PATH_SEPARATOR + SHARED_FOLDER);
		}catch (PathNotFoundException e) {
			sharedNode = session.getRootNode().addNode(SHARED_FOLDER, NT_FOLDER);
			session.save();
		}
		return sharedNode;
	}

	/**
	 * get User Home
	 * @param session
	 * @return
	 * @throws RepositoryException
	 */
	public Node getUserHome(Session session) throws RepositoryException {		
		logger.info("getUserHome: " + PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR + portalLogin);
		return  session.getNode(PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR + portalLogin); 
	}


	/**
	 * Get Smart Folders root
	 * @param session
	 * @return Smart Folders root
	 * @throws RepositoryException
	 */
	public Node getRootSmartFolders(Session session) throws RepositoryException{

		logger.info("getRootSmartFolders: " + PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR 
				+ portalLogin + PATH_SEPARATOR + SMART_FOLDER);
		Node smartFolders = null;
		if (smartFolders==null){
			try {
				//create applicationFolder
				try {				
					smartFolders = session.getNode(PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR 
							+ portalLogin + PATH_SEPARATOR + SMART_FOLDER);	
				} catch (PathNotFoundException e) {
					smartFolders = session.getNode(PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR 
							+ portalLogin).addNode(SMART_FOLDER, NT_FOLDER);
					session.save();
				}
			}catch (Exception e) {
				throw new RepositoryException(e);
			} 
		}
		return smartFolders;
	}

	/**
	 * get InBoxFolder
	 * @param session
	 * @return InBoxFolder
	 * @throws RepositoryException
	 */
	public Node getOwnInBoxFolder(Session session) throws RepositoryException {

		Node inBoxNode = null;

		if (inBoxNode==null){
			try {
				//create applicationFolder
				try {				
					inBoxNode = session.getNode(PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR 
							+ portalLogin + PATH_SEPARATOR + IN_BOX_FOLDER);	
				} catch (PathNotFoundException e) {
					inBoxNode = session.getNode(PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR 
							+ portalLogin).addNode(IN_BOX_FOLDER, NT_ROOT_ITEM_SENT);
					session.save();
				}
			}catch (Exception e) {
				throw new RepositoryException(e);
			}
		}
		return inBoxNode;

	}

	/**
	 * get OutBoxFolder
	 * @param session
	 * @return OutBoxFolder
	 * @throws RepositoryException
	 */
	public Node getOutBoxFolder(Session session) throws RepositoryException {

		Node outBoxNode = null;

		logger.info("getOutBoxFolder: " + PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR 
				+ portalLogin + PATH_SEPARATOR + OUT_BOX_FOLDER);
		if (outBoxNode==null){
			try {
				//create applicationFolder
				try {				
					outBoxNode = session.getNode(PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR 
							+ portalLogin + PATH_SEPARATOR + OUT_BOX_FOLDER);	
				} catch (PathNotFoundException e) {
					outBoxNode = session.getNode(PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR 
							+ portalLogin).addNode(OUT_BOX_FOLDER, NT_ROOT_ITEM_SENT);
					session.save();
				}
			}catch (Exception e) {
				throw new RepositoryException(e);
			}
		}
		return outBoxNode;
	}

	/**
	 * Get Download folder
	 * @param session
	 * @return Download folder
	 * @throws RepositoryException
	 */
	public Node getRootFolderBulkCreators(Session session) throws RepositoryException {

		Node downloads = null;

		logger.info("Get Download Folder: " + PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR 
				+ portalLogin + PATH_SEPARATOR + DOWNLOADS);
		if (downloads==null){
			try {
				//create applicationFolder
				try {				
					downloads = session.getNode(PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR 
							+ portalLogin + PATH_SEPARATOR + DOWNLOADS);	
				} catch (PathNotFoundException e) {
					downloads = session.getNode(PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR 
							+ portalLogin).addNode(DOWNLOADS, NT_ROOT_FOLDER_BULK_CREATOR);
					session.save();
				}
			}catch (Exception e) {
				throw new RepositoryException(e);
			}
		}
		return downloads;
	}


	/**
	 * Get Hidden folder
	 * @param session
	 * @return Hidden folder
	 * @throws RepositoryException
	 */
	public Node getHiddenFolder(Session session) throws RepositoryException {

		Node hiddenFolder = null;

		logger.info("hiddenFolder: " + PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR 
				+ portalLogin + PATH_SEPARATOR + HIDDEN_FOLDER);
		if (hiddenFolder==null){
			try {
				//create applicationFolder
				try {				
					hiddenFolder = session.getNode(PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR 
							+ portalLogin + PATH_SEPARATOR + HIDDEN_FOLDER);	
				} catch (PathNotFoundException e) {
					hiddenFolder = session.getNode(PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR 
							+ portalLogin).addNode(HIDDEN_FOLDER, NT_ROOT_ITEM_SENT);
					session.save();
				}
			}catch (Exception e) {
				throw new RepositoryException(e);
			}
		}
		return hiddenFolder;
	}


	/**
	 * get InBoxFolder
	 * @param session
	 * @param user
	 * @return InBoxFolder
	 * @throws RepositoryException
	 * @throws InternalErrorException
	 */
	public Node getInBoxFolder(Session session, String user) throws RepositoryException,
	InternalErrorException  {
		return session.getNode(PATH_SEPARATOR + HOME_FOLDER + PATH_SEPARATOR 
				+ user + PATH_SEPARATOR + IN_BOX_FOLDER);
	}

	public String getUserHomeUrl(String portalLogin) {
		return url + ROOT_WEBDAV + HOME_FOLDER + PATH_SEPARATOR + portalLogin;     
	}

	public String getWebDavUrl(String portalLogin) {
		return webdavUrl + PATH_SEPARATOR + portalLogin;      
	}



	public static String getUserVersion(String portalLogin, JCRUserManager um) throws InternalErrorException {
		if (um==null)
			um = new JCRUserManager();
		String userVersion = um.getVersionByUser(portalLogin);
		return userVersion;
	}


}
