package org.gcube.portlets.user.tdwx.datasource.td.filters;

import java.util.ArrayList;
import java.util.Date;

import org.gcube.data.analysis.tabulardata.expression.Expression;
import org.gcube.data.analysis.tabulardata.expression.composite.comparable.Equals;
import org.gcube.data.analysis.tabulardata.expression.composite.comparable.GreaterThan;
import org.gcube.data.analysis.tabulardata.expression.composite.comparable.LessThan;
import org.gcube.data.analysis.tabulardata.expression.composite.text.TextContains;
import org.gcube.data.analysis.tabulardata.expression.logical.And;
import org.gcube.data.analysis.tabulardata.model.column.Column;
import org.gcube.data.analysis.tabulardata.model.column.ColumnReference;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDBoolean;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDDate;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDInteger;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDNumeric;
import org.gcube.data.analysis.tabulardata.model.datatype.value.TDText;
import org.gcube.data.analysis.tabulardata.model.table.Table;
import org.gcube.data.analysis.tabulardata.query.parameters.QueryFilter;
import org.gcube.portlets.user.tdwx.datasource.td.TDXDataSource;
import org.gcube.portlets.user.tdwx.datasource.td.map.DataTypeMap;
import org.gcube.portlets.user.tdwx.server.datasource.DataSourceXException;
import org.gcube.portlets.user.tdwx.shared.FilterInformation;
import org.gcube.portlets.user.tdwx.shared.model.ColumnDefinition;
import org.gcube.portlets.user.tdwx.shared.model.TableDefinition;
import org.gcube.portlets.user.tdwx.shared.model.ValueType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 
 * @author "Giancarlo Panichi"
 * 
 */
public class FiltersBuilder {

	protected ArrayList<FilterInformation> filters;
	protected TableDefinition tableDefinition;

	protected Table serviceTable;
	protected Logger logger = LoggerFactory.getLogger(TDXDataSource.class);

	/**
	 * 
	 * @param filters
	 * @param tableDefinition
	 */
	public FiltersBuilder(ArrayList<FilterInformation> filters,
			TableDefinition tableDefinition, Table serviceTable) {
		this.filters = filters;
		this.tableDefinition = tableDefinition;
		this.serviceTable = serviceTable;
	}

	/**
	 * 
	 * @return
	 * @throws DataSourceXException
	 */
	protected Expression createExpression() throws DataSourceXException {
		Expression exp = null;

		ArrayList<Expression> andExp = new ArrayList<Expression>();
		for (FilterInformation filter : filters) {
			switch (filter.getFilterType()) {
			case "string":
				if (tableDefinition.getColumns().get(filter.getFilterField()) == null) {
					logger.error("The specified filter column \""
							+ filter.getFilterField() + "\" don't exists");

					throw new DataSourceXException(
							"The specified filter column \""
									+ filter.getFilterField()
									+ "\" don't exists");
				} else {
					ColumnDefinition columnDefinition = tableDefinition
							.getColumns().get(filter.getFilterField());
					Column column = serviceTable
							.getColumnByName(columnDefinition.getId());
					TextContains contains = new TextContains(
							new ColumnReference(serviceTable.getId(),
									column.getLocalId()), new TDText(
									filter.getFilterValue()));
					andExp.add(contains);
				}
				break;
			case "numeric":
				if (tableDefinition.getColumns().get(filter.getFilterField()) == null) {
					logger.error("The specified filter column \""
							+ filter.getFilterField() + "\" don't exists");

					throw new DataSourceXException(
							"The specified filter column \""
									+ filter.getFilterField()
									+ "\" don't exists");
				} else {
					ColumnDefinition columnDefinition = tableDefinition
							.getColumns().get(filter.getFilterField());
					Column column = serviceTable
							.getColumnByName(columnDefinition.getId());

					Expression ex = getExpressionNumeric(column, filter);
					if (ex != null) {
						andExp.add(ex);
					}
				}
				break;
			case "boolean":
				if (tableDefinition.getColumns().get(filter.getFilterField()) == null) {
					logger.error("The specified filter column \""
							+ filter.getFilterField() + "\" don't exists");

					throw new DataSourceXException(
							"The specified filter column \""
									+ filter.getFilterField()
									+ "\" don't exists");
				} else {
					ColumnDefinition columnDefinition = tableDefinition
							.getColumns().get(filter.getFilterField());
					Column column = serviceTable
							.getColumnByName(columnDefinition.getId());

					Equals contains = new Equals(new ColumnReference(
							serviceTable.getId(), column.getLocalId()),
							new TDBoolean(new Boolean(filter.getFilterValue())));
					andExp.add(contains);
				}

				break;

			case "date":
				if (tableDefinition.getColumns().get(filter.getFilterField()) == null) {
					logger.error("The specified filter column \""
							+ filter.getFilterField() + "\" don't exists");

					throw new DataSourceXException(
							"The specified filter column \""
									+ filter.getFilterField()
									+ "\" don't exists");
				} else {
					ColumnDefinition columnDefinition = tableDefinition
							.getColumns().get(filter.getFilterField());
					Column column = serviceTable
							.getColumnByName(columnDefinition.getId());
					Expression ex = getExpressionDate(column, filter);
					if (ex != null) {
						andExp.add(ex);
					}
				}

				break;
			default:
				break;

			}
		}

		if (andExp.size() > 0) {
			if (andExp.size() == 1) {
				exp = andExp.get(0);
			} else {
				And andE = new And(andExp);
				exp = andE;
			}
		}

		return exp;
	}

	protected Expression getExpressionDate(Column column,
			FilterInformation filter) throws DataSourceXException {
		// [{ "filterField":"kqljyp" ,"filterType":"date"
		// ,"filterComparison":"on" ,"filterValue": "1399370400000" }]
		Expression exp = null;

		Date dd = new Date();
		Long day = null;
		
		try {
			day = new Long(filter.getFilterValue());
		} catch (NumberFormatException e) {
			logger.error("No valid filterValue for date: "
					+ filter.getFilterValue());
			throw new DataSourceXException("No valid filter value");
		}

		dd.setTime(day);
		TDDate tdDate = new TDDate(dd);

		switch (filter.getFilterComparison()) {
		case "on":
			exp = new Equals(new ColumnReference(serviceTable.getId(),
					column.getLocalId()), tdDate);
			break;
		case "after":
			exp = new GreaterThan(new ColumnReference(serviceTable.getId(),
					column.getLocalId()), tdDate);
			break;	
		case "before":
			exp = new LessThan(new ColumnReference(serviceTable.getId(),
					column.getLocalId()), tdDate);
			break;		
		default:
			break;

		}
		return exp;
	}

	protected Expression getExpressionNumeric(Column column,
			FilterInformation filter) {
		Expression exp = null;

		ValueType vt = DataTypeMap.getValueType(column.getDataType());

		switch (vt) {
		case FLOAT:
			TDNumeric floatnum = new TDNumeric(new Float(
					filter.getFilterValue()));
			if (filter.getFilterComparison().compareTo("eq") == 0) {
				exp = new Equals(new ColumnReference(serviceTable.getId(),
						column.getLocalId()), floatnum);
			} else {
				if (filter.getFilterComparison().compareTo("gt") == 0) {
					exp = new GreaterThan(new ColumnReference(
							serviceTable.getId(), column.getLocalId()),
							floatnum);
				} else {
					if (filter.getFilterComparison().compareTo("lt") == 0) {
						exp = new LessThan(new ColumnReference(
								serviceTable.getId(), column.getLocalId()),
								floatnum);
					} else {

					}

				}

			}
			break;
		case INTEGER:
			TDInteger intnum = new TDInteger(new Integer(
					filter.getFilterValue()));
			if (filter.getFilterComparison().compareTo("eq") == 0) {
				exp = new Equals(new ColumnReference(serviceTable.getId(),
						column.getLocalId()), intnum);
			} else {
				if (filter.getFilterComparison().compareTo("gt") == 0) {
					exp = new GreaterThan(new ColumnReference(
							serviceTable.getId(), column.getLocalId()), intnum);
				} else {
					if (filter.getFilterComparison().compareTo("lt") == 0) {
						exp = new LessThan(new ColumnReference(
								serviceTable.getId(), column.getLocalId()),
								intnum);
					} else {

					}
				}
			}
			break;
		case LONG:
			TDInteger longnum = new TDInteger(new Integer(
					filter.getFilterValue()));
			if (filter.getFilterComparison().compareTo("eq") == 0) {
				exp = new Equals(new ColumnReference(serviceTable.getId(),
						column.getLocalId()), longnum);
			} else {
				if (filter.getFilterComparison().compareTo("gt") == 0) {
					exp = new GreaterThan(new ColumnReference(
							serviceTable.getId(), column.getLocalId()), longnum);
				} else {
					if (filter.getFilterComparison().compareTo("lt") == 0) {
						exp = new LessThan(new ColumnReference(
								serviceTable.getId(), column.getLocalId()),
								longnum);
					} else {

					}
				}
			}
			break;
		default:
			break;

		}
		return exp;
	}

	/**
	 * 
	 * @return
	 * @throws DataSourceXException
	 */
	public QueryFilter createQueryFilter() throws DataSourceXException {
		QueryFilter queryFilter = null;
		Expression exp = createExpression();
		if (exp != null) {
			queryFilter = new QueryFilter(exp);
		}
		return queryFilter;
	}

}
