package org.gcube.opensearch.opensearchdatasource;

import gr.uoa.di.madgik.environment.hint.EnvHint;
import gr.uoa.di.madgik.environment.hint.EnvHintCollection;
import gr.uoa.di.madgik.environment.hint.NamedEnvHint;
import gr.uoa.di.madgik.grs.record.GenericRecordDefinition;
import gr.uoa.di.madgik.grs.record.RecordDefinition;
import gr.uoa.di.madgik.grs.record.field.FieldDefinition;
import gr.uoa.di.madgik.grs.record.field.StringFieldDefinition;
import gr.uoa.di.madgik.grs.utils.Locators;
import gr.uoa.di.madgik.is.InformationSystem;
import gr.uoa.di.madgik.rr.element.search.Field;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.faults.GCUBERetrySameException;
import org.gcube.common.core.informationsystem.client.ISClient;
import org.gcube.common.core.informationsystem.client.QueryParameter;
import org.gcube.common.core.informationsystem.client.XMLResult;
import org.gcube.common.core.informationsystem.client.queries.GCUBEGenericQuery;
import org.gcube.common.core.resources.GCUBEGenericResource;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.state.GCUBEWSResource;
import org.gcube.common.core.state.GCUBEWSResourceKey;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.opensearch.opensearchdatasource.processor.FieldDefinitionInfo;
import org.gcube.opensearch.opensearchdatasource.processor.GCQLNodeAnnotation;
import org.gcube.opensearch.opensearchdatasource.processor.OpenSearchCollectionEnricher;
import org.gcube.opensearch.opensearchdatasource.processor.OpenSearchGcqlAnnotator;
import org.gcube.opensearch.opensearchdatasource.processor.OpenSearchGcqlProcessor;
import org.gcube.opensearch.opensearchdatasource.processor.OpenSearchGcqlQueryContainer;
import org.gcube.opensearch.opensearchdatasource.processor.OpenSearchProjector;
import org.gcube.opensearch.opensearchlibrary.DescriptionDocument;
import org.gcube.opensearch.opensearchlibrary.queryelements.BasicQueryElementFactory;
import org.gcube.opensearch.opensearchlibrary.urlelements.BasicURLElementFactory;
import org.gcube.opensearch.opensearchlibrary.utils.FactoryClassNamePair;
import org.gcube.opensearch.opensearchoperator.OpenSearchOp;
import org.gcube.opensearch.opensearchoperator.OpenSearchOpConfig;
import org.gcube.opensearch.opensearchoperator.resource.ISOpenSearchResource;
import org.gcube.opensearch.opensearchoperator.resource.ISOpenSearchResourceCache;
import org.gcube.opensearch.opensearchoperator.resource.OpenSearchResource;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;

import search.library.util.cql.query.tree.GCQLNode;



/**
 * A class containing stateful information regarding an OpenSearch adaptor.
 *
 * @author gerasimos.farantatos, NKUA
 *
 */
public class OpenSearchDataSourceResource extends GCUBEWSResource {

	/*
	 * Environmental Hints 
	 */
	EnvHintCollection envHints = new EnvHintCollection();
	
	/* Log4j */
	static GCUBELog logger = new GCUBELog(OpenSearchDataSource.class);
	
	/* cached generic OpenSearch resource */
	public ISOpenSearchResource[] openSearchGenericResources = null;
	
	/* cache including OpenSearch generic resource and description document XML representations and XSLT generic resources */
	public ISOpenSearchResourceCache cache = new ISOpenSearchResourceCache();
	
	String[][] fixedParameters;
	
	public List<String> searchableFields = new ArrayList<String>();
	public List<String> presentableFields = new ArrayList<String>();
	
	public List<String> allPresentableNames = new ArrayList<String>();
	
	private static final int ISRETRY = 3;
	
	int inittime = (int) Calendar.getInstance().getTimeInMillis() / 1000 / 60;
	
	/** The list of standard resource properties defined by every OpenSearch WS resource */
    public static final String RP_ADAPTOR_ID = "AdaptorID";
    public static final String RP_COLLECTION_ID = "CollectionID";
    public static final String RP_SUPPORTED_RELATIONS = "SupportedRelations";
    public static final String RP_FIELDS = "Fields";
    public static final String RP_DESCRIPTION_DOCUMENT_URI = "DescriptionDocumentURI";
    public static final String RP_FIXEDPARAMETERS = "FixedParameters";
    public static final String RP_OPENSEARCH_RESOURCE = "OpenSearchResource";
    protected static String[] RPNames = { RP_ADAPTOR_ID, RP_COLLECTION_ID, RP_SUPPORTED_RELATIONS, RP_FIELDS,
    	RP_DESCRIPTION_DOCUMENT_URI, RP_OPENSEARCH_RESOURCE, RP_FIXEDPARAMETERS};
    
   
    private Document retrieveDescriptionDocument(String descriptionDocumentURI) throws Exception {
    	URL DDUrl = new URL(descriptionDocumentURI);
    	URLConnection conn = DDUrl.openConnection();
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setNamespaceAware(true);
		DocumentBuilder builder = factory.newDocumentBuilder();
		BufferedReader in = new BufferedReader(new InputStreamReader(conn.getInputStream()));
		Document descriptionDocument = builder.parse(new InputSource(in));
		in.close();
		return descriptionDocument;
    }
    
    public static String[] retrieveTemplates(DescriptionDocument dd, OpenSearchResource res) throws Exception {
    	Map<String, List<String>> uniqueTemplates = dd.getUniqueTemplates();	
//    	List<String> specifiedMimeTypes =  res.getTransformationTypes();
    	List<String> supportedTemplates = new ArrayList<String>();
//    	for(String key : uniqueTemplates.keySet()) {
//    		if(specifiedMimeTypes.contains(key))
//    			supportedTemplates.addAll(uniqueTemplates.get(key));
//    	}
//    	return supportedTemplates.toArray(new String[1]);
    	for(List<String> templates : uniqueTemplates.values())
    		supportedTemplates.addAll(templates);
    	return supportedTemplates.toArray(new String[1]);
    }
    
    /**
     * Retrieves the property names
     * @return An array containing the property names
     */
    public String[] getPropertyNames() {
		return RPNames;
	}
    
    /** Is the resource in the initialization state or not? */
	protected boolean bInitializing;
	
    /**
     * Returns the current initialization state of the resource
     * @return true if the resoure is currently initializing, false otherwise
     */
    public boolean isInitializing() {
    	return bInitializing;
    }
    
    /**
     * Specifies the current state of the resource (initializing or not)
     * @param isInitializing the initialization state to set
     */
    public void setIsInitializing(boolean isInitializing) {
    	bInitializing = isInitializing;
    }

    /**
	 * Empty constructor. Initialization is done through the initialize(...) method.
	 */ 
	public OpenSearchDataSourceResource() {}

	/**
	 * This method is used to initialize the resource instance, and must be called before using the resource instance.
	 * 
	 * @param key <code>String</code> - used as the key for this OpenSearch adaptor
	 * @param collectionIDs <code>String[]</code> - a list of IDs of the collections(for now we have only one collection) that refer to this OpenSearch adaptor
	 * @param genericResource <code>String</code> - the openSearch generic resource expressed as an XMLResult
	 * @param openSearchResource <code>String</code> - the ID of the openSearch generic resource
	 * @param fixedParameters <code>String</code> - parameters that must be fixed in the templates of a broker, in the broker-case
	 * @return <code>Object</code> - the key/UniqueID of the resource
	 */
	protected void initialise(Object... args) throws Exception {

		envHints.AddHint(new NamedEnvHint("GCubeActionScope", new EnvHint(ServiceContext.getContext().getScope().toString())));
		
		logger.debug("initialise is called!");
		
		setIsInitializing(true);
		
		//this adaptorID may be used in case we split up OpenSearch adaptors, using a multiRole distributed approach
		GCUBEWSResourceKey key = (GCUBEWSResourceKey)args[0];
		String adaptorID = key.getValue().toString();
		
		String[] fields = (String[])args[1];
		String[] collectionID = (String[]) args[2];
		String[][] fixedParameters = (String[][]) args[5];
		XMLResult[] openSearchResourceXML = (XMLResult[])args[3];
		String[] openSearchResourceIds = (String[])args[4];
		this.getResourcePropertySet().get(RP_ADAPTOR_ID).clear();
    	this.getResourcePropertySet().get(RP_ADAPTOR_ID).add(adaptorID);
    	
    	int size = OpenSearchDataSource.SupportedRelations.values().length;
    	this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).clear();
//    	for (int i=0; i<size; i++)
//    		this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).add(supportedRelations[i]);
    	this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).add("="); //TODO change when all relations are supported
    	
    	logger.debug("Just added to the resource property set RP_ADAPTOR_ID: " + this.getResourcePropertySet().get(RP_ADAPTOR_ID).get(0));
		this.getResourcePropertySet().get(RP_COLLECTION_ID).clear();
		this.getResourcePropertySet().get(RP_FIXEDPARAMETERS).clear();
		this.getResourcePropertySet().get(RP_OPENSEARCH_RESOURCE).clear();
		this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).clear();
		int providerCount = collectionID.length;
		this.openSearchGenericResources = new ISOpenSearchResource[providerCount];
		this.fixedParameters = new String[providerCount][];
		int fixedParamNo = 0;
		for(int i = 0; i < providerCount; i++) {
			try {	
				synchronized(cache) {
					this.openSearchGenericResources[i] = new ISOpenSearchResource(openSearchResourceXML[i].toString(), this.cache.descriptionDocuments, this.cache.resourcesXML, this.cache.XSLTs, this.envHints);
				}
			}catch(Exception e) {
				logger.error("Could not create ISOpenSearchResource :", e);
				throw e;
			}
			DescriptionDocument dd = new DescriptionDocument(this.openSearchGenericResources[i].getDescriptionDocument(), new BasicURLElementFactory(), new BasicQueryElementFactory());
			String[] templates = retrieveTemplates(dd, this.openSearchGenericResources[i]);
	    
	    	this.getResourcePropertySet().get(RP_COLLECTION_ID).add(collectionID[i]);
	    	logger.debug("Just added to the resource property set RP_COLLECTION_ID : " + this.getResourcePropertySet().get(RP_COLLECTION_ID).get(i));
	
	    	this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).add(this.openSearchGenericResources[i].getDescriptionDocURL().toString());
	    	logger.debug("Just added to the resource property set RP_DESCRIPTION_DOCUMENT_URI: " + this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).get(i));
	
	//		size = templates.length;
	//    	this.getResourcePropertySet().get(RP_TEMPLATES).clear();
	//    	for (int i=0; i<size; i++)
	//    		this.getResourcePropertySet().get(RP_TEMPLATES).add(templates[i]);
	//    	logger.debug("Just added to the resource property set RP_TEMPLATES with size: " + this.getResourcePropertySet().get(RP_TEMPLATES).size());
	    	
	    	this.fixedParameters[i] = fixedParameters[i];
	    	size = fixedParameters[i].length;
	    	
	    	for (int j=0; j<size; j++) {
	    		this.getResourcePropertySet().get(RP_FIXEDPARAMETERS).add(fixedParameters[i][j]);
	    		logger.debug("Just added to the resource property set RP_FIXEDPARAMETERS: " + this.getResourcePropertySet().get(RP_FIXEDPARAMETERS).get(fixedParamNo++));
	    	}
	    	
	        this.getResourcePropertySet().get(RP_OPENSEARCH_RESOURCE).add(openSearchResourceIds[i]);
	        logger.debug("Just added to the resource property set RP_OPENSEARCH_RESOURCE: " + this.getResourcePropertySet().get(RP_OPENSEARCH_RESOURCE).get(i));
		}
        setIsInitializing(false);
        
        Map<String, String> parameters = new HashMap<String, String>();
        for(int i = 0; i < providerCount; i++)
        	parameters.putAll(this.openSearchGenericResources[i].getParameters()); //No harm in more than one resources containing the same mapping as they are the same by definition
      	this.getResourcePropertySet().get(RP_FIELDS).clear();
    	this.allPresentableNames.clear();
    	for(int i = 0; i < fields.length; i++) {
    		String[] splitField = fields[i].split(":");
    		this.getResourcePropertySet().get(RP_FIELDS).add(fields[i]+":" + parameters.get(splitField[splitField.length-1]));
    		logger.debug("Just added to the resource property set RP_FIELDS : " + this.getResourcePropertySet().get(RP_FIELDS).get(i));
    		if(splitField[splitField.length-2].equals("p")) {
    			List<Field> f = Field.getFieldsWithName(false, splitField[splitField.length-1]);
    			if(!f.isEmpty())
    				this.allPresentableNames.add(f.get(0).getName());
    		}
    	}
    	filterFieldInfo(this.presentableFields, this.searchableFields);
        logger.debug("initialise returned!");

	}
	
	public void addProviders(String[] fields, String[] collectionIds, String[] openSearchResourceIDs, String[][] fixedParams) throws Exception {
		
		logger.debug("addProviders is called!");
		
		int providerToAddCount = collectionIds.length;
		
		String[][] savedFixedParams = this.fixedParameters;
		ISOpenSearchResource[] savedOpenSearchResources = this.openSearchGenericResources;
		ISOpenSearchResource[] openSearchResourcesToAdd = new ISOpenSearchResource[providerToAddCount];
		
		XMLResult[] openSearchResourceXML = new XMLResult[providerToAddCount];
		for(int i = 0; i < providerToAddCount; i++) {
			openSearchResourceXML[i] = retrieveGenericResource(openSearchResourceIDs[i], GCUBEScope.getScope(this.getResourcePropertySet().getScope().get(0)));
			synchronized(cache) {
				openSearchResourcesToAdd[i] = new ISOpenSearchResource(openSearchResourceXML[i].toString(), this.cache.descriptionDocuments, this.cache.resourcesXML, this.cache.XSLTs, this.envHints);
			}
		}
		for(int i = 0; i < fields.length; i++) {
    		this.getResourcePropertySet().get(RP_FIELDS).add(fields[i]);
    		logger.debug("Just added to the resource property set RP_FIELDS : " + this.getResourcePropertySet().get(RP_FIELDS).get(i));
    	}
		this.allPresentableNames.clear();
		for(int i = 0; i < this.getResourcePropertySet().get(RP_FIELDS).size(); i++) {
			String[] splitField = ((String)this.getResourcePropertySet().get(RP_FIELDS).get(i)).split(":");
			if(splitField[splitField.length-2].equals("p")) {
				List<Field> f = Field.getFieldsWithName(false, splitField[splitField.length-1]);
				if(!f.isEmpty())
					this.allPresentableNames.add(f.get(0).getName());
			}
		}
			
    	filterFieldInfo(this.presentableFields, this.searchableFields);
		
		try {
			int providerCount = this.getResourcePropertySet().get(RP_COLLECTION_ID).size();
			this.fixedParameters = new String[providerCount+providerToAddCount][];
			this.openSearchGenericResources = new ISOpenSearchResource[providerCount+providerToAddCount];
			int i = 0;
			for(i = 0; i < providerCount; i++) {
				this.fixedParameters[i] = savedFixedParams[i];
				this.openSearchGenericResources[i] = savedOpenSearchResources[i];
			}
			for(int j = 0; j < providerToAddCount; j++) {
				this.fixedParameters[i+j] = fixedParams[j];
				this.openSearchGenericResources[i+j] = openSearchResourcesToAdd[j];
				this.getResourcePropertySet().get(RP_COLLECTION_ID).add(collectionIds[j]);
		    	logger.debug("Just added to the resource property set RP_COLLECTIONID : " + this.getResourcePropertySet().get(RP_COLLECTION_ID).get(providerCount+j));
		    	this.getResourcePropertySet().get(RP_OPENSEARCH_RESOURCE).add(openSearchResourceIDs[j]);
		    	logger.debug("Just added to the resource property set RP_OPENSEARCHRESOURCE : " + this.getResourcePropertySet().get(RP_OPENSEARCH_RESOURCE).get(providerCount+j));
		    	this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).add(openSearchGenericResources[i+j].getDescriptionDocURL().toString());
		    	logger.debug("Just added to the resource property set RP_DESCRIPTION_DOCUMENT_URI : " + this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).get(providerCount+j));
			}
		}catch(Exception e) {
			this.fixedParameters = savedFixedParams;
			this.openSearchGenericResources = savedOpenSearchResources;
			throw e;
		}
	}
	
	/**
	 * Invoked when a resource is being created from a serialized, previously saved state.
	 * 
	 * @param ois the input stream through which the state can be read
	 * @param indicates if the resource is being loaded for the first time (hard load) or not (soft load)
	 * @throws Exception an error occured during resource deserialization
	 */
	protected void onLoad(ObjectInputStream ois, boolean firstLoad) throws Exception {
		
		try {
		logger.debug("OnLoad started!");
		setIsInitializing(true);
		
		boolean clearCacheOnStartup = true;
		OpenSearchDataSourceConfig config = (OpenSearchDataSourceConfig) StatefulContext.getPortTypeContext().getProperty("config", false);
		if(config != null) {
			clearCacheOnStartup = config.getClearCacheOnStartup();
			logger.debug("Read configuration property: clearCacheOnStartup = " + config.getClearCacheOnStartup());
		}else
			logger.warn("Could not read configuration property: clearCacheOnStartup. Using default: true");
		
		String scope = this.getResourcePropertySet().getScope().get(0);
		
		logger.debug("Loaded scope: " + scope);
		
		this.getServiceContext().setScope(GCUBEScope.getScope(scope));
		
		envHints.AddHint(new NamedEnvHint("GCubeActionScope", new EnvHint(scope)));
		
		int size;
		synchronized(cache) {
	    	this.cache = new ISOpenSearchResourceCache();
	    	size = ois.readInt();
	    	for(int i = 0; i < size; i++)
	    		this.cache.XSLTs.put((String)ois.readObject(), (String)ois.readObject());
	    	size = ois.readInt();
	    	for(int i = 0; i < size; i++)
	    		this.cache.descriptionDocuments.put((String)ois.readObject(), (String)ois.readObject());
	    	size = ois.readInt();
	    	for(int i = 0; i < size; i++)
	    		this.cache.resourcesXML.put((String)ois.readObject(), (String)ois.readObject());
			
			//Load generic resource xml representation and deserialize
	    	for(Map.Entry<String, String> e : this.cache.resourcesXML.entrySet())
	    		this.cache.resources.put(e.getKey(), new ISOpenSearchResource(this.cache.resourcesXML.get(e.getKey()), this.cache.descriptionDocuments, this.cache.resourcesXML, this.cache.XSLTs, this.envHints));
		}
		/* Create and initialize the standard resource properties from the given input stream */
		String adaptorID = (String) ois.readObject();
		this.getResourcePropertySet().get(RP_ADAPTOR_ID).clear();
    	this.getResourcePropertySet().get(RP_ADAPTOR_ID).add(adaptorID);
    	
    	size = ois.readInt();
    	this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).clear();
    	for(int i = 0; i < size; i++)
    		this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).add((String)ois.readObject());
    	logger.debug("Just read supported relations size= " + size);

    	size = ois.readInt();
    	this.getResourcePropertySet().get(RP_FIELDS).clear();
    	this.allPresentableNames.clear();
    	for (int i=0; i<size; i++)
    		this.getResourcePropertySet().get(RP_FIELDS).add((String) ois.readObject());
    	for(int i = 0; i < size; i++) {
    		String[] splitField = ((String)this.getResourcePropertySet().get(RP_FIELDS).get(i)).split(":");
    		if(splitField[splitField.length-2].equals("p")) {
    			List<Field> f = Field.getFieldsWithName(false, splitField[splitField.length-1]);
    			if(!f.isEmpty())
    				this.allPresentableNames.add(f.get(0).getName());
    		}
    	}
    	logger.debug("Just read fields size= " + size);
    	filterFieldInfo(this.presentableFields, this.searchableFields);
    	
    	size = ois.readInt();
    	this.getResourcePropertySet().get(RP_COLLECTION_ID).clear();
    	for (int i=0; i<size; i++)
    		this.getResourcePropertySet().get(RP_COLLECTION_ID).add((String) ois.readObject());
    	logger.debug("Just read collections size= " + size);
    	int providerCount = this.getResourcePropertySet().get(RP_COLLECTION_ID).size();
    
    	size = ois.readInt();
    	this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).clear();
    	for (int i=0; i<size; i++)
    		this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).add((String) ois.readObject());
    	logger.debug("Just read description document uris size= " + size);
    	
    	
//    	size = ois.readInt();
//    	this.getResourcePropertySet().get(RP_TEMPLATES).clear();
//    	for (int i=0; i<size; i++)
//    		this.getResourcePropertySet().get(RP_TEMPLATES).add((String) ois.readObject());

    	int fixedParameterProviderCount = 0;
    	fixedParameters = new String[providerCount][];
    	size = ois.readInt();
    	this.getResourcePropertySet().get(RP_FIXEDPARAMETERS).clear();
    	@SuppressWarnings("unchecked")
		List<String>[] tmpFixedParameters = new List[providerCount];
    	for(int i = 0; i < providerCount; i++)
    		tmpFixedParameters[i] = new ArrayList<String>();
    	for (int i=0; i<size; i++) {
    		String fixedParameter = (String)ois.readObject();
    		this.getResourcePropertySet().get(RP_FIXEDPARAMETERS).add(fixedParameter);
    		if(i > 0 && !((String)this.getResourcePropertySet().get(RP_FIXEDPARAMETERS).get(i)).split(":")[0].equals(((String)this.getResourcePropertySet().get(RP_FIXEDPARAMETERS).get(i)).split(":")[0]))
    			fixedParameterProviderCount++;
    		tmpFixedParameters[fixedParameterProviderCount].add(fixedParameter);
    	}
    	for(int i = 0; i < providerCount; i++)
    		fixedParameters[i] = tmpFixedParameters[i].toArray(new String[0]);
    	logger.debug("Just read fixed parameters size= " + size);
    	
    	size = ois.readInt();
    	this.getResourcePropertySet().get(RP_OPENSEARCH_RESOURCE).clear();
    	for (int i=0; i<size; i++)
    		this.getResourcePropertySet().get(RP_OPENSEARCH_RESOURCE).add((String) ois.readObject());
    	logger.debug("Just read opensearch resource size= " + size);

    	this.openSearchGenericResources = new ISOpenSearchResource[providerCount];
        synchronized(cache) {
        	for(int i = 0; i < providerCount; i++)
        		this.openSearchGenericResources[i] = (ISOpenSearchResource)this.cache.resources.get(this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).get(i));
        }

		if(firstLoad == true && clearCacheOnStartup == true) {
			logger.debug("OnLoad clearing cache");
			try {
				clearCache(GCUBEScope.getScope(scope));
			}catch(FailedToRevertException e) {
				logger.warn("Could not clear cache");
				throw e;
			}catch(Exception e) {
				logger.warn("Could not clear cache. Will use stored generic resource");
			}
			
		}
        	
        setIsInitializing(false);
        
        logger.debug("OnLoad finished!");
		}catch(Exception e) {
			logger.error("Caught exception during OnLoad", e);
			throw e;
		}
	}
	
	/**
	 * Invoked when the state of the resource must be saved (resource serialization)
	 * 
	 * @param oos the output stream to write the resource state to
	 * @throws Exception an error occured during resource serialization
	 */
	protected void onStore(ObjectOutputStream oos) throws Exception {
		
		logger.debug("OnStore started!");
		int size;
		
		//Store cache so that it will be available to ISOpenSearchResource constructor during onLoad
		synchronized(cache) {
			size = this.cache.XSLTs.size();
			oos.writeInt(size);
			for(Map.Entry<String, String> e : this.cache.XSLTs.entrySet()) {
				oos.writeObject(e.getKey());
				oos.writeObject(e.getValue());
			}
			size = this.cache.descriptionDocuments.size();
			oos.writeInt(size);
			for(Map.Entry<String, String> e : this.cache.descriptionDocuments.entrySet()) {
				oos.writeObject(e.getKey());
				oos.writeObject(e.getValue());
			}
			size = this.cache.resourcesXML.size();
			oos.writeInt(size);
			for(Map.Entry<String, String> e : this.cache.resourcesXML.entrySet()) {
				oos.writeObject(e.getKey());
				oos.writeObject(e.getValue());
			}
		}
		
		
		//Store remaining resource properties
		oos.writeObject((String) this.getResourcePropertySet().get(RP_ADAPTOR_ID).get(0));
		
		size = this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).size();
		oos.writeInt(size);
		logger.debug("Just wrote supported relations size= " + size);
		for(int i = 0; i < size; i++)
			oos.writeObject((String)this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).get(i));
		
		size = this.getResourcePropertySet().get(RP_FIELDS).size();
		oos.writeInt(size);
		logger.debug("Just wrote fields size= " + size);
		for (int i=0; i<size; i++)
			oos.writeObject((String) this.getResourcePropertySet().get(RP_FIELDS).get(i));
		
		size = this.getResourcePropertySet().get(RP_COLLECTION_ID).size();
		oos.writeInt(size);
		logger.debug("Just wrote collection size= " + size);
		for (int i=0; i<size; i++)
			oos.writeObject((String) this.getResourcePropertySet().get(RP_COLLECTION_ID).get(i));

		size = this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).size();
		oos.writeInt(size);
		logger.debug("Just wrote description document uris size= " + size);
		for (int i=0; i<size; i++)
			oos.writeObject((String) this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).get(i));

//		size = this.getResourcePropertySet().get(RP_TEMPLATES).size();
//		oos.writeInt(size);
//		for (int i=0; i<size; i++)
//			oos.writeObject((String) this.getResourcePropertySet().get(RP_TEMPLATES).get(i));
		
		size = this.getResourcePropertySet().get(RP_FIXEDPARAMETERS).size();
		oos.writeInt(size);
		for (int i=0; i<size; i++)
			oos.writeObject((String) this.getResourcePropertySet().get(RP_FIXEDPARAMETERS).get(i));
		
		size = this.getResourcePropertySet().get(RP_OPENSEARCH_RESOURCE).size();
		oos.writeInt(size);
		logger.debug("Just wrote opensearch resource size= " + size);
		for (int i=0; i<size; i++)
			oos.writeObject((String) this.getResourcePropertySet().get(RP_OPENSEARCH_RESOURCE).get(i));

		logger.debug("OnStore finished!");
	}
	
	private void clearCache(GCUBEScope scope) throws Exception {
		int providerCount = this.getResourcePropertySet().get(RP_COLLECTION_ID).size();
		String genericResourceXML[] = new String[providerCount];
		for(int i = 0; i < providerCount; i++)
			genericResourceXML[i] = OpenSearchDataSourceResource.retrieveGenericResource(this.getOpenSearchResource(i), scope).toString();
		
		synchronized(cache) {
			ISOpenSearchResource[] savedResources = new ISOpenSearchResource[providerCount];
			String[] savedDdUris = new String[providerCount];
			for(int i = 0; i < providerCount; i++) {
				savedResources[i] = this.openSearchGenericResources[i];
				savedDdUris[i] = this.getDescriptionDocumentURI(i);
			}
				
			cache.descriptionDocuments.clear();
			cache.resources.clear();
			cache.resourcesXML.clear();
			cache.XSLTs.clear();
			for(int i = 0; i < providerCount; i++) {
		     	String genericResourceId = this.getOpenSearchResource(i);
		     	ISOpenSearchResourceCache cache = this.cache;
		     	String ddURI = savedDdUris[i];
		     	EnvHintCollection envHints = this.getISEnvHints();
				
				Exception ex = null;
					
				boolean fail = false;
				try {
					this.openSearchGenericResources[i] = new ISOpenSearchResource(genericResourceXML[i], cache.descriptionDocuments, 
							cache.resourcesXML, cache.XSLTs, envHints);
				}catch(Exception e) {
					logger.warn("Could not create OpenSearch resource instance during cache clearing. Using old instance.", e);
					fail = true; ex = e;
				}
				cache.resources.put(ddURI, this.openSearchGenericResources[i]);
				if(!fail) {
					DescriptionDocument dd = null;
					ddURI = this.openSearchGenericResources[i].getDescriptionDocURL();
					try {
						dd = new DescriptionDocument(this.openSearchGenericResources[i].getDescriptionDocument(), new BasicURLElementFactory(), new BasicQueryElementFactory());
					}catch(Exception e) {
						logger.warn("Could not create Description Document instance during cache clearing. Using old OpenSearch resource instance.", e);
						fail = true; ex = e;
					}
					if(!fail) {
	//					String[] templates = null;
	//					try {
	//						templates = OpenSearchDataSourceResource.retrieveTemplates(dd, this.openSearchGenericResource);
	//					}catch(Exception e) {
	//						logger.warn("Could not retrieve templates from Description Document. Using old OpenSearch resource instance.", e);
	//						fail = true; ex = e;
	//					}
						if(!fail) {
							try {
								if(i == 0) this.setDescriptionDocumentURI(ddURI);
								else this.addDescriptionDocumentURI(ddURI);
							}catch(Exception e) {
								logger.warn("Could not update WS resource with the new Description Document URI. Using old OpenSearch resource instance", e);
								fail = true; ex = e;
							}
	//						if(!fail) {
	//							try {
	//								this.setTemplates(templates);
	//							}catch(Exception e) {
	//								logger.warn("Could not update WS resource with the new templates. Using old OpenSearch resource instance", e);
	//								ex = e;
	//								try {
	//									this.setDescriptionDocumentURI(savedDdUri); //First step toward reverting to old state
	//								}catch(Exception ee) {
	//									logger.error("Could not revert WS resource to its old state. Resource will remain inconsistent!", ee);
	//									fail = true; ex = ee;
	//								}
	//							}
	//						}
						}
					} 
					if(fail) { //Revert to old state on failure	
						cache.resources.clear();
						for(int j =0; j < providerCount; j++) {
							this.openSearchGenericResources[j] = savedResources[j];
							cache.resources.put(savedDdUris[j], savedResources[j]);
						}
						this.setDescriptionDocumentURI(savedDdUris);
						throw ex;
					}
				}
			}
		}
	}
	
	/**
	 * Retrieves a generic resource using the IS
	 * 
	 * @param id The id of the generic resource to retrieve
	 * @param scope The scope to use
	 * @return The generic resource expressed as an XMLResult
	 * @throws Exception In case of error
	 */
	public static XMLResult retrieveGenericResource(String id, GCUBEScope scope) throws Exception {
		//retrieve the openSearch generic resource from the IS
		ISClient client = null;
		GCUBEGenericQuery query = null;
	//	try{
			client =  GHNContext.getImplementation(ISClient.class);
			query = client.getQuery("GCUBEResourceQuery");
	//	}catch (Exception e) {
	//		throw new GCUBERetryEquivalentException("Could not get ISClient or GCUBEGenericQuery");
	//	}
		
		query.addParameters(new QueryParameter("FILTER", "$result/ID/string() eq '" + id+ "'"),
				new QueryParameter("TYPE", GCUBEGenericResource.TYPE),
				new QueryParameter("RESULT", "$result/Profile/Body"));

		List<XMLResult> res = null;
		Exception ex = null;
		for(int retryOnError=0;retryOnError<ISRETRY;retryOnError+=1){
			try{
				res = client.execute(query, scope);
			}catch(Exception e){
				logger.error("Could not retrieve generic resource. continuing");
				ex = e;
				continue;
			}
			break;
		}
		
		if(res == null || res.size() == 0) {
			if(ex != null)
				throw new GCUBERetrySameException("The generic resource with ID: " + id + " was not found.", ex);
			else
				throw new GCUBERetrySameException("The generic resource with ID: " + id + " was not found.");
		}
		if(res.size() > 1)
			throw new GCUBERetrySameException("The generic resource with ID: " + id + " was found " + res.size() + " times published.");
		
		return res.get(0);
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.gcube.common.core.state.GCUBEStatefulResource#store()
	 */
	public void store() {
		/* Store the resource only if it's not currently being initialized */
		if (!this.isInitializing())
			super.store();
	}
	
	/**
     * Getter method for the AdaptorID Resource Property
     * @return <code>String</code> the requested adaptorID 
     */
    public String getAdaptorID() {
        return (String) this.getResourcePropertySet().get(RP_ADAPTOR_ID).get(0);
    }

    /**
     * Setter method for the AdaptorID Resource Property
     * @param adaptorID <code>String</code> the new AdaptorID
     */
    public void setAdaptorID(String adaptorID) { 
    	this.getResourcePropertySet().get(RP_ADAPTOR_ID).clear();
    	this.getResourcePropertySet().get(RP_ADAPTOR_ID).add(adaptorID);    	
    }
    
    /**
     * Getter method for the Fields list Resource Property
     * @return <code>String[]</code> the requested Fields
     */
    public String[] getFields() {
    	int size = this.getResourcePropertySet().get(RP_FIELDS).size();
    	String[] fields = new String[size];
    	for (int i=0; i<size; i++)
    		fields[i] = (String) this.getResourcePropertySet().get(RP_FIELDS).get(i);
    	return fields;
    }
    
    /**
     * Getter method for the CollectionID list Resource Property
     * @return <code>String[]</code> the requested CollectionIDs
     */
    public String[] getCollectionID() {
    	int size = this.getResourcePropertySet().get(RP_COLLECTION_ID).size();
    	String[] colIDs = new String[size];
    	for (int i=0; i<size; i++)
    		colIDs[i] = (String) this.getResourcePropertySet().get(RP_COLLECTION_ID).get(i);
    	return colIDs;
    }
    
    /**
     * Setter method for the CollectionID list Resource Property
     * @param collectionIDs <code>String[]</code> the new CollectionIDs
     */
    public void setCollectionID(String[] collectionIDs) throws Exception {
    	int size = collectionIDs.length;
    	this.getResourcePropertySet().get(RP_COLLECTION_ID).clear();
    	for (int i=0; i<size; i++)
    		this.getResourcePropertySet().get(RP_COLLECTION_ID).add(collectionIDs[i]);
    }

    /**
     * Adds a collection id to the collectionID resource property
     * @param collectionID <code>String</code> -the ID to be added
     */ 
    public void addCollectionID(String collectionID){
        String[] collections = getCollectionID();
        String [] newCollections = new String[collections.length +1];
        for(int i = 0; i < collections.length; i++){
            newCollections[i] = collections[i];
        }
        newCollections[collections.length] = collectionID;
        try {
        	this.setCollectionID(newCollections);
        } catch (Exception e) { 
        	logger.error("Failed to set CollectionID resource property.", e);
        }
    }
    
    /**
     * Getter method for the DescriptionDocumentURI list Resource Property
     * @return <code>String[]</code> the requested DescriptionDocumentURIs
     */
    public String[] getDescriptionDocumentURI() {
    	int size = this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).size();
    	String[] ddURIs = new String[size];
    	for (int i=0; i<size; i++)
    		ddURIs[i] = (String) this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).get(i);
    	return ddURIs;
    }
    
    /**
     * Getter method for the DescriptionDocument Resource Property
     * @return <code>String</code> the requested DescriptionDocumentURI
     */
    public String getDescriptionDocumentURI(int i) {
        return (String) this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).get(i);
    }        
    /**
     * Setter method for the DescriptionDocument Resource Property
     * @param descriptionDocumentURI <code>String</code> the new DescriptionDocumentURI 
     * @throws Exception In case of error
     */
    public void setDescriptionDocumentURI(String descriptionDocumentURI) throws Exception {
        try{
            this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).clear();
            this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).add(descriptionDocumentURI);
        }
        catch(Exception e){
            logger.error(e.toString());
        }
    }
    
    public void addDescriptionDocumentURI(String descriptionDocumentURI) throws Exception {
        try{
            this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).add(descriptionDocumentURI);
        }
        catch(Exception e){
            logger.error(e.toString());
        }
    }
    
    /**
     * Setter method for the DescriptionDocumentURI list Resource Property
     * @param descriptionDocumentURIs <code>String[]</code> the new DescriptionDocumentURIs
     * @throws Exception In case of error
     */
    public void setDescriptionDocumentURI(String[] descriptionDocumentURIs) throws Exception {
        try{
            this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).clear();
            for(int i = 0; i < descriptionDocumentURIs.length; i++)
            	this.getResourcePropertySet().get(RP_DESCRIPTION_DOCUMENT_URI).add(descriptionDocumentURIs[i]);
        }
        catch(Exception e){
            logger.error(e.toString());
        }
    }
    
    /**
     * Getter method for the Templates list Resource Property
     * @return <code>String[]</code> the requested Templates
     */
//    public String[] getTemplates() {
//    	int size = this.getResourcePropertySet().get(RP_TEMPLATES).size();
//    	String[] templates = new String[size];
//    	for (int i=0; i<size; i++)
//    		templates[i] = (String) this.getResourcePropertySet().get(RP_TEMPLATES).get(i);
//    	return templates;
//    }
    
    /**
     * Setter method for the Templates list Resource Property
     * @param templates <code>String[]</code> the new Templates
     * @throws Exception In case of error
     */
//    public void setTemplates(String[] templates) throws Exception {
//    	int size = templates.length;
//    	this.getResourcePropertySet().get(RP_TEMPLATES).clear();
//    	for (int i=0; i<size; i++)
//    		this.getResourcePropertySet().get(RP_TEMPLATES).add(templates[i]);
//    }

    /**
     * Adds a template to the Templates resource property
     * @param template <code>String</code> -the template to be added
     */ 
//    public void addTemplate(String template){
//        String[] templates = getTemplates();
//        String [] newTemplates = new String[templates.length +1];
//        for(int i = 0; i < templates.length; i++){
//        	newTemplates[i] = templates[i];
//        }
//        newTemplates[templates.length] = template;
//        try {
//        	this.setTemplates(newTemplates);
//        } catch (Exception e) { 
//        	logger.error("Failed to set Templates resource property.", e);
//        }
//    }
    
    /**
     * Getter method for the FixedParameters list Resource Property
     * @return <code>String[]</code> the requested FixedParameters
     */
    public String[] getFixedParameters() {
    	int size = this.getResourcePropertySet().get(RP_FIXEDPARAMETERS).size();
    	String[] fixedParameters = new String[size];
    	for (int i=0; i<size; i++)
    		fixedParameters[i] = (String) this.getResourcePropertySet().get(RP_FIXEDPARAMETERS).get(i);
    	return fixedParameters;
    }
    
    /**
     * Setter method for the FixedParameters list Resource Property
     * @param fixedParameters <code>String[]</code> the new FixedParameters
     * @throws Exception In case of error
     */
    public void setFixedParameters(String[] fixedParameters) throws Exception {
    	int size = fixedParameters.length;
    	this.getResourcePropertySet().get(RP_FIXEDPARAMETERS).clear();
    	for (int i=0; i<size; i++)
    		this.getResourcePropertySet().get(RP_FIXEDPARAMETERS).add(fixedParameters[i]);
    }

    /**
     * Adds a fixed parameter to the FixedParameters resource property
     * @param fixedParameter <code>String</code> -the fixed parameter to be added
     */ 
//    public void addFixedParameter(String fixedParameter){
//        String[] fixedParameters = getFixedParameters();
//        String [] newFixedParameters = new String[fixedParameters.length +1];
//        for(int i = 0; i < fixedParameters.length; i++){
//        	newFixedParameters[i] = fixedParameters[i];
//        }
//        newFixedParameters[fixedParameters.length] = fixedParameter;
//        try {
//        	this.setTemplates(newFixedParameters);
//        } catch (Exception e) { 
//        	logger.error("Failed to set FixedParameters resource property.", e);
//        }
//    }
    
    /**
     * Getter method for the OpenSearchResource Resource Property
     * @return <code>String</code> the OpenSearchResource ID
     */
    public String getOpenSearchResource(int i) {
        return (String) this.getResourcePropertySet().get(RP_OPENSEARCH_RESOURCE).get(i); 
    }
    
    /**
     * Getter method for the OpenSearchResource list Resource Property
     * @return <code>String</code> the OpenSearchResource ID
     */
    public String[] getOpenSearchResource() {
    	String[] openSearchResources = new String[this.getResourcePropertySet().get(RP_OPENSEARCH_RESOURCE).size()];
    	for(int i = 0; i < openSearchResources.length; i++)
    		openSearchResources[i] = (String) this.getResourcePropertySet().get(RP_OPENSEARCH_RESOURCE).get(i); 
        return openSearchResources;
    } 
    
    /**
     * Setter method for the OpenSearchResource Resource Property
     * @param openSearchResource <code>String</code> the new OpenSearchResource ID
     */
    public void setOpenSearchResource(String openSearchResource) {
    	this.getResourcePropertySet().get(RP_OPENSEARCH_RESOURCE).clear();
    	this.getResourcePropertySet().get(RP_OPENSEARCH_RESOURCE).add(openSearchResource);
    }

    /**
     * Setter method for the OpenSearchResource list Resource Property
     * @param openSearchResource <code>String</code> the new OpenSearchResource ID
     */
    public void setOpenSearchResource(String[] openSearchResource) {
    	this.getResourcePropertySet().get(RP_OPENSEARCH_RESOURCE).clear();
    	for(int i = 0; i < openSearchResource.length; i++)
    		this.getResourcePropertySet().get(RP_OPENSEARCH_RESOURCE).add(openSearchResource[i]);
    }
    
    /**
     * Getter method for the cached Description document
     * @return The description document
     */
    public Document getDescriptionDocument(int i) {
    	synchronized(cache) {
    		return openSearchGenericResources[i].getDescriptionDocument();
    	}
    }
    
    /**
     * Returns the cached OpenSearch Resource of a brokered provider whose description document URL is ddURL
     * @param ddURL the URL of the brokered provider's description document
     * @return the brokered provider's cached OpenSearch Resource
     */
    public OpenSearchResource getCachedOpenSearchResource(String ddURL) {
    	synchronized(cache) {
    		return cache.resources.get(ddURL);
    	}
    }
    
    /**
     * Returns all cached OpenSearch Resources of the brokered providers associated with the provider of this WS-Resource
     * @return a list containing all OpenSearch Resources associated with this WS-Resource
     */
    public List<OpenSearchResource> getCachedOpenSearchResources() {
    	synchronized(cache) {
    		return new ArrayList<OpenSearchResource>(cache.resources.values());
    	}
    }
    
    /**
     * Adds an OpenSearch Resource to the brokered provider's resource cache
     * @param ddURL the URL of the brokered provider's description document
     * @param resource the OpenSearch Resource for the brokered provider
     * @throws Exception In case of error
     */
    public void addCachedOpenSearchResource(String ddURL, XMLResult resource) throws Exception {
    	synchronized(cache) {
    		cache.resources.put(ddURL, new ISOpenSearchResource(resource.toString(), envHints));
    	}
    }
    
    /**
     * Adds the OpenSearch Resources contained in the map to the brokered provider's resource cache
     * @param resources A map containing a mappings from description document URLs to OpenSearch Resources for
     * each brokered provider
     * @throws Exception In case of error
     */
    public void addCachedOpenSearchResources(Map<String, XMLResult> resources) throws Exception {
    	synchronized(cache) {
	    	for(Map.Entry<String, XMLResult> e : resources.entrySet()) {
	    		cache.resources.put(e.getKey(), new ISOpenSearchResource(e.getValue().toString(), envHints));
	    	}
    	}
    }
    
	/**
	 * Performs a query on the OpenSearch provider that is connected to this resource. 
	 * @param queryString <code>String</code>  - the query to be performed (using custom 
	 * syntax @see )
	 * @return <code>URI</code>  - representation of the EPR for a resultset service which holds the results of the query.
	 * @throws RemoteException In case of error
	 */
	public URI query(String cqlQuery) throws RemoteException
	{  
		try{
			Map<String, FactoryClassNamePair> factories = null;
			OpenSearchDataSourceConfig config = (OpenSearchDataSourceConfig) StatefulContext.getPortTypeContext().getProperty("config", false);
			if(config != null) {
				factories = config.getFactories();
				logger.debug("Read configuration property: factories = " + config.getFactories());
			}else
				logger.warn("Could not read configuration property: factories. Operating using only basic factories");
			OpenSearchOpConfig cfg = new OpenSearchOpConfig(null, null, null, factories);
			synchronized(cache) {
				cfg.ISCache = cache;
		//		res = this.openSearchGenericResource;
			}
			OpenSearchGcqlProcessor processor = new OpenSearchGcqlProcessor();
			OpenSearchGcqlAnnotator annotator = new OpenSearchGcqlAnnotator();
			GCQLNode queryTree = processor.parseQuery(cqlQuery);
			GCQLNodeAnnotation annotationTree = annotator.processNode(queryTree);
			String collection = annotator.getFirstEncounteredCollectionId(annotationTree); //TODO change
			processor.setAnnotationTree(annotationTree);
			processor.setCollection(collection);
			processor.setFields(this.allPresentableNames);
			processor.setDataSourceLocator(this.getID().getValue());
			logger.debug("Datasource locator is " + this.getID().getValue());
		
			OpenSearchGcqlQueryContainer query =  (OpenSearchGcqlQueryContainer)processor.processQuery(this.presentableFields, this.searchableFields);

			int index = -1;
			for(int i = 0; i < this.getResourcePropertySet().get(RP_COLLECTION_ID).size(); i++) {
				if(this.getResourcePropertySet().get(RP_COLLECTION_ID).get(i).equals(processor.getCollection())) {
					index = i;
					break;
				}
			}
			if(index == -1)
				throw new RemoteException("Could not locate provider");
			
			String[] custom_fixed = {"count:50"};
			
			OpenSearchOp op = new OpenSearchOp(this.openSearchGenericResources[index], this.fixedParameters[index], cfg, new EnvHintCollection());
			
			//OpenSearchOp op = new OpenSearchOp(this.openSearchGenericResources[index], this.fixedParameters[index], cfg, new EnvHintCollection());
			URI loc = op.query(query.queries.get(processor.getCollection()).get("*").get(0).toString());
			OpenSearchCollectionEnricher colEnricher = new OpenSearchCollectionEnricher(collection, loc);
			new Thread(colEnricher).start();
			loc = colEnricher.getLocator();
			
			FieldDefinitionInfo fieldDefInfo = createFieldDefinition(query.getProjectedFields());
			FieldDefinition[] fieldDef = fieldDefInfo.fieldDefinition;
			Map<String, Integer> fieldPositions = fieldDefInfo.fieldPositions;
			RecordDefinition[] definition = new RecordDefinition[]{new GenericRecordDefinition(fieldDef)};
			
			OpenSearchProjector projector = new OpenSearchProjector(loc, definition, processor.getProjectedFields(), fieldPositions);
			projector.setReaderTimeout(3, TimeUnit.MINUTES);
			new Thread(projector).start();
			
			return Locators.localToTCP(projector.getProjectionLocator());

		}catch(Exception e){
			logger.error("Unable to execute query: \"" + cqlQuery + "\"" , e);
			throw new RemoteException("Could not execute query", e);
		}
	}
	
	private FieldDefinitionInfo createFieldDefinition(Map<String, String> projections) {
		
		Integer pos = 0;
		Map<String, Integer> fieldPositions = new HashMap<String, Integer>();
		List<FieldDefinition> fieldDef = new ArrayList<FieldDefinition>();
		fieldDef.add(new StringFieldDefinition(OpenSearchDataSourceConstants.COLLECTION_FIELD));
		fieldPositions.put(OpenSearchDataSourceConstants.COLLECTION_FIELD, pos++);
		fieldDef.add(new StringFieldDefinition(OpenSearchDataSourceConstants.OBJECTID_FIELD));
		fieldPositions.put(OpenSearchDataSourceConstants.OBJECTID_FIELD, pos++);
		
		//if project was specified
        if(projections.size() > 0)
        {
            //get all the projections specified in this query
        	for(Map.Entry<String, String> current : projections.entrySet()) {
        		String proj = current.getValue();
        		String fieldId = current.getKey();
        		if(proj.equalsIgnoreCase(OpenSearchDataSourceConstants.COLLECTION_FIELD)) {
            		fieldDef.add(new StringFieldDefinition(fieldId));
            		fieldPositions.put(OpenSearchDataSourceConstants.COLLECTION_FIELD, pos++);
        		}
        	}
        	for(Map.Entry<String, String> current : projections.entrySet()) {
        		String proj = current.getValue();
        		String fieldId = current.getKey();
        		if(proj.equalsIgnoreCase(OpenSearchDataSourceConstants.LANGUAGE_FIELD)) {
            		fieldDef.add(new StringFieldDefinition(fieldId));
            		fieldPositions.put(OpenSearchDataSourceConstants.LANGUAGE_FIELD, pos++);
        		}
        	}

        	
            for(Map.Entry<String, String> current : projections.entrySet()) {
            	String proj = current.getValue();
            	String fieldId = current.getKey();
                if(proj.equalsIgnoreCase(OpenSearchDataSourceConstants.LANGUAGE_FIELD) || 
                		proj.equalsIgnoreCase(OpenSearchDataSourceConstants.COLLECTION_FIELD))
                	continue;
                 
                fieldDef.add(new StringFieldDefinition(fieldId));
                fieldPositions.put(fieldId, pos);
                pos++; 
            }
            //if there is no valid projection return 
            if(fieldDef.size() == 1)
            {
            	logger.error(" no valid projection ");
            }
        }
        
        return new FieldDefinitionInfo(fieldDef.toArray(new FieldDefinition[fieldDef.size()]), fieldPositions);
	}


	/**
	 * Retrieves the {@link InformationSystem} environmental hints that this resource is using
	 * 
	 * @return The environmental hints
	 */
	public EnvHintCollection getISEnvHints() {
		return envHints;
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.gcube.indexmanagement.common.IndexLookupWSResource#onResourceRemoval()
	 */
	public void onResourceRemoval() {}


protected void filterFieldInfo(List<String> presentableFields, List<String> searchableFields){
		
    	//first clear the arrays
    	presentableFields.clear();
    	searchableFields.clear();
    	
    	//to ensure each field is added only once
    	HashSet<String> searchableSet = new HashSet<String>();
    	HashSet<String> presentableSet =  new HashSet<String>();
    	
    	//create the arrays of presentable and searchable fields
		for(String field : this.getFields())
		{
			//add the field to the related array, depending on its type
			String fieldName;
			if((fieldName = getFieldName(field, OpenSearchDataSourceConstants.PRESENTABLE_TAG)) != null)
			{
				if(presentableSet.add(fieldName)) {
					logger.trace("added presentable field: " + fieldName);
					presentableFields.add(fieldName);
				}
				
			} else {
				fieldName = getFieldName(field, OpenSearchDataSourceConstants.SEARCHABLE_TAG);
				if(searchableSet.add(fieldName)) {
					logger.trace("added searchable field: " + fieldName);
					searchableFields.add(fieldName);
				}
			}
		}
	}
	
	private static String getFieldName(String field, String type) {
		//get the string after the second seperator, which indicates the type
		int from = field.indexOf(OpenSearchDataSourceConstants.FIELD_SEPARATOR, 
				field.indexOf(OpenSearchDataSourceConstants.FIELD_SEPARATOR) + 1) + 1;
		String t = field.substring(from, from + type.length());
		if(!t.equals(type))
			return null;
		//get the string after the third seperator, which indicates the fieldName
		from = field.indexOf(OpenSearchDataSourceConstants.FIELD_SEPARATOR, from) + 1;
		int to = field.indexOf(OpenSearchDataSourceConstants.FIELD_SEPARATOR, from);
		return field.substring(from, to);		
	}
    
}
