package org.gcube.common.clients;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Enumeration;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.apache.axis.client.AxisClient;
import org.reflections.Reflections;
import org.reflections.scanners.ResourcesScanner;
import org.reflections.util.ClasspathHelper;
import org.reflections.util.ConfigurationBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Sets;

/**
 * The runtime of gCube clients. 
 * <p>
 * Embeds and installs a tailored gHN distribution which allows clients to make calls to services deployed
 * in gCube infrastructures. Supports resource discovery, publication, and event notifications. Notification
 * support requires the availability of open ports and relies on the host name of the local machine. Alternative
 * host names can be configured as the value of the {@link #LOGICAL_HOST_SYSTEM_PROPERTY}.
 * 
 * 
 * @author Fabio Simeoni
 *
 */
public class ClientRuntime {

	private static Logger log = LoggerFactory.getLogger(ClientRuntime.class);

	private static final String LOGICAL_HOST_WSDD_PROPERTY = "logicalHost";
	private static final String LOGICAL_HOST_SYSTEM_PROPERTY = "gcube.client.hostname";

	public static final String location_property = "GLOBUS_LOCATION";
	
	private static final String installation_name = "ghn-client-distro";
	private static final String archive_name=installation_name+".zip"; 
	
	/**
	 * The path used to find service map configuration files.
	 */
	static final String mapConfigPattern = "ServiceMap_.*";
	
	/**
	 * Installs an embedded client distribution on a temporary location on the file system, and then starts it. 
	 */
	public static synchronized void start() {
		
		String location = System.getProperty(location_property);
		
		if (location==null || !new File(location).exists())
			try {
				installDistro();
				installMaps();
				setLogicalHost();
				
				Runtime.getRuntime().addShutdownHook(new Thread() {
					public void run() {
						try {
							ClientRuntime.stop();
						}
						catch(Throwable t) {
							log.warn("could not stop the client container and its services",t);
						}
					};
				});
				
			}
			catch(Throwable t) {
				throw new RuntimeException("could not start client runtime",t);
			}
		else 
			log.info("using client runtime already at "+new File(location).getAbsolutePath());
		
			
	}

	/**
	 * Deletes the client distribution. 
	 */
	public static synchronized void stop() {
		
		try {
			new File(tempDir(),archive_name).delete();
			new File(tempDir(),installation_name).delete();
			log.info("stopped client runtime");
		}
		catch(Throwable t) {
			throw new RuntimeException("could not stop client runtime",t);
		}
	}
	
	private static File tempDir() throws Exception {
		
		File random = File.createTempFile("temp", null);
		File dir = random.getParentFile();
		random.delete();
		return dir;
	}
	
	private static void copy(InputStream in, File file) throws Exception {
	
		byte data[] = new byte[2048];

		BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(file));
		
		int read = 0;
		while ((read=in.read(data))!=-1)
			out.write(data,0,read);
		
		out.flush();
		out.close();
		in.close();
	}
	
	private static void installDistro() throws Exception {

		
		InputStream embedded = ClientRuntime.class.getResourceAsStream("/"+archive_name);
		File archive = new File(tempDir(),archive_name);
		copy(embedded,archive);
		
		ZipFile zipFile = new ZipFile(archive);
		
		// Create an enumeration of the entries in the zip file
		Enumeration<? extends ZipEntry> zipFileEntries = zipFile.entries();

		// Process each entry
		while (zipFileEntries.hasMoreElements()) {

			ZipEntry entry = zipFileEntries.nextElement();

			File destFile = new File(tempDir(), entry.getName());

			if (entry.isDirectory())
				destFile.mkdirs();
			else
				copy(zipFile.getInputStream(entry),destFile);
		
//			if (!destFile.exists()) {
//			}

		}
		zipFile.close();

		String installation = new File(tempDir(),installation_name).getAbsolutePath();
		System.setProperty(location_property, installation);
//		addToEnvironment(installation);
		
		log.info("started client container in {}",installation);

	}

	private static void installMaps() {

		File configLocation =  new File(System.getProperty(location_property),"config");
		
		try {

			// we include urls specified in manifest files, which is required
			// when we run tests in surefire's forked-mode
			ConfigurationBuilder builder = new ConfigurationBuilder().setUrls(
					ClasspathHelper.forManifest(urlsToScan())).setScanners(new ResourcesScanner());

			Reflections reflections = new Reflections(builder);

			for (String resource : reflections.getResources(Pattern.compile(mapConfigPattern))) {
				InputStream map = Thread.currentThread().getContextClassLoader().getResourceAsStream(resource);
				log.info("loading map {} ",resource);
				copy(map,new File(configLocation,resource));
				
			}
		} catch (Exception e) {
			throw new RuntimeException("could not load service maps", e);
		}
	}

	// helper: we use reflections' code but exclude extension and primordial
	// classloaders
	// whose URLs we do not want to include. especially because these may have
	// non-standard URLs
	// that would need to be excluded individually from standard scanning,
	// or reflections will show (but ignore) a horrible error in the logs. and
	// we do no know how to predict what we will
	// find on any given machine
	private static Set<URL> urlsToScan() {
		final Set<URL> result = Sets.newHashSet();

		ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
		while (classLoader != null && classLoader.getParent() != null) {
			if (classLoader instanceof URLClassLoader) {
				URL[] urls = ((URLClassLoader) classLoader).getURLs();
				if (urls != null) {
					result.addAll(Sets.<URL> newHashSet(urls));
				}
			}
			classLoader = classLoader.getParent();
		}

		return result;
	}
	
	
	@SuppressWarnings("all")
	private static void setLogicalHost() {
		
		try {
			String hostname = System.getProperty(LOGICAL_HOST_SYSTEM_PROPERTY);
			if (hostname==null)
				hostname =  InetAddress.getLocalHost().getHostName();
			if (hostname==null)
				throw new Exception("cannot determine local hostname. configure "+LOGICAL_HOST_SYSTEM_PROPERTY+".");
			
			new AxisClient().getConfig().getGlobalOptions().put(LOGICAL_HOST_WSDD_PROPERTY,hostname);
		}
		catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
//	@SuppressWarnings({ "rawtypes", "unchecked" })
//	// helper
//	// this is a serious hack to deal with legacy code, we are reflectively
//	// changing the
//	// unmodifiable map that contains the env vars in Java
//	private static void addToEnvironment(String installation) throws Exception {
//		Class<?>[] classes = Collections.class.getDeclaredClasses();
//		Map<String, String> env = System.getenv();
//		for (Class<?> cl : classes) {
//			if ("java.util.Collections$UnmodifiableMap".equals(cl.getName())) {
//				Field field = cl.getDeclaredField("m");
//				field.setAccessible(true);
//				Map<String, String> map = (Map) field.get(env);
//				log.debug("setting GLOBUS_LOCATION to {}",installation);
//				map.put("GLOBUS_LOCATION", installation);
//			}
//		}
//	}
}
