package org.gcube.data.analysis.statisticalmanager.experimentspace.computation;

import java.awt.Image;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.util.Calendar;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.UUID;

import javax.imageio.ImageIO;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.contentmanagement.graphtools.data.conversions.ImageTools;
import org.gcube.data.analysis.statisticalmanager.SMOperationType;
import org.gcube.data.analysis.statisticalmanager.SMResourceType;
import org.gcube.data.analysis.statisticalmanager.persistence.RemoteStorage;
import org.gcube.data.analysis.statisticalmanager.persistence.SMPersistenceManager;
import org.gcube.dataanalysis.ecoengine.datatypes.OutputTable;
import org.gcube.dataanalysis.ecoengine.datatypes.PrimitiveType;
import org.gcube.dataanalysis.ecoengine.datatypes.StatisticalType;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.PrimitiveTypes;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMComputation;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMFile;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMObject;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMResource;
import org.gcube_system.namespaces.data.analysis.statisticalmanager.types.SMTable;

public class BuilderComputationOutput {
	
	private static GCUBELog logger = new GCUBELog(BuilderComputationOutput.class);
	
	private String portalLogin;
	private String scope;
	private SMComputation computation;
	
	public BuilderComputationOutput(String portalLogin, String scope, SMComputation computation) {
		this.portalLogin = portalLogin;
		this.scope = scope;
		this.computation = computation;
	}
	
	private SMTable serializeTabular(OutputTable output) throws Exception {
		
		String template = output.getTemplateNames().get(0).toString();
		SMTable table = new SMTable(template);
		table.setPortalLogin(portalLogin);
		table.setAlgorithm(computation.getAlgorithm());
		table.setResourceType(SMResourceType.TABULAR.ordinal());
		table.setResourceId( output.getTableName());
		table.setDescription(output.getDescription());
		table.setName(output.getName());
		table.setProvenance(SMOperationType.COMPUTED.ordinal());
		table.setCreationDate(Calendar.getInstance());
		table.setOperationId(computation.getOperationId());
		
		SMPersistenceManager.addCreatedResource(table);
		
		return table;
	}
	
	private SMFile serializeFile(PrimitiveType output) throws Exception {
		
		File outputFile = (File)output.getContent();
		RemoteStorage storage = new RemoteStorage(portalLogin, scope);
		
		String fileName = UUID.randomUUID().toString();
		String url = storage.putFile(outputFile, fileName);
		SMFile file = new SMFile("mimeType", fileName, url);
		file.setPortalLogin(portalLogin);
		file.setAlgorithm(computation.getAlgorithm());
		file.setResourceType(SMResourceType.FILE.ordinal());
		file.setResourceId(UUID.randomUUID().toString());
		file.setDescription(output.getDescription());
		file.setName(outputFile.getName());
		file.setProvenance(SMOperationType.COMPUTED.ordinal());
		file.setCreationDate(Calendar.getInstance());
		file.setOperationId(computation.getOperationId());
		
		SMPersistenceManager.addCreatedResource(file);
		
		return file;
	}
	
	private SMObject serializePrimitiveObject(PrimitiveType primitiveObject) {
		
		switch (primitiveObject.getType()) {
		case STRING:
			SMObject object = new SMObject((String)primitiveObject.getContent());
			object.setPortalLogin(portalLogin);
			object.setAlgorithm(computation.getAlgorithm());
			object.setName(PrimitiveTypes.STRING.toString());
			object.setResourceType(SMResourceType.OBJECT.ordinal());
			object.setDescription(primitiveObject.getDescription());
			object.setProvenance(SMOperationType.COMPUTED.ordinal());
			object.setCreationDate(Calendar.getInstance());
			object.setOperationId(computation.getOperationId());
			return object;
		default:
			break;
		} 
		return null;
	}
	
	private SMObject serializeImage(PrimitiveType primitiveObject) throws Exception {
		
		RemoteStorage storage = new RemoteStorage(portalLogin, scope);
		
		@SuppressWarnings("unchecked")
		Map<String,Image> map = (Map<String,Image>)primitiveObject.getContent();
		String dirName = File.separator + UUID.randomUUID().toString();
		for(Entry<String, Image> entry : map.entrySet()) {	
			ByteArrayOutputStream os = new ByteArrayOutputStream();
			ImageIO.write(ImageTools.toBufferedImage(entry.getValue()),"png",os);	
			storage.putStream(new ByteArrayInputStream(os.toByteArray()),
					dirName + File.separator + entry.getKey());
		}    		
		
		SMObject resource = new SMObject(dirName);
		resource.setPortalLogin(portalLogin);
		resource.setResourceType(SMResourceType.OBJECT.ordinal());
		resource.setResourceId(UUID.randomUUID().toString());
		resource.setName(PrimitiveTypes.IMAGES.toString());
		resource.setDescription(primitiveObject.getDescription());	
		resource.setProvenance(SMOperationType.COMPUTED.ordinal());
		resource.setCreationDate(Calendar.getInstance());
		resource.setOperationId(computation.getOperationId());
		return resource;
	}

	private SMObject serializeMap(PrimitiveType primitiveObject) throws Exception {
		
		@SuppressWarnings("unchecked")
		Map<String, StatisticalType> map = (Map<String,StatisticalType>)primitiveObject.getContent();
		Map<String, SMResource> outputs = new LinkedHashMap<String, SMResource>();
		for (Entry<String,StatisticalType> entry : map.entrySet()) {
			SMResource resource = serialize(entry.getValue());
			outputs.put(entry.getKey(), resource);
		}
		
		RemoteStorage storage = new RemoteStorage(portalLogin, scope);
		String url = storage.putObject(outputs);
		
		SMObject resource = new SMObject(url);
		resource.setPortalLogin(portalLogin);
		resource.setResourceType(SMResourceType.OBJECT.ordinal());
		resource.setResourceId(UUID.randomUUID().toString());
		resource.setName(PrimitiveTypes.MAP.toString());
		resource.setDescription(primitiveObject.getDescription());
		resource.setProvenance(SMOperationType.COMPUTED.ordinal());
		resource.setCreationDate(Calendar.getInstance());
		resource.setOperationId(computation.getOperationId());
		return resource;
	}
	
	public SMResource serialize(StatisticalType object) throws Exception {
		
		if (object instanceof OutputTable){
			return serializeTabular((OutputTable) object);
		}
		
		if (object instanceof PrimitiveType){
			PrimitiveType primitiveObject = (PrimitiveType)object;
			if(primitiveObject.getType() == PrimitiveTypes.MAP) {
				return serializeMap(primitiveObject);
			} else if (primitiveObject.getType() == PrimitiveTypes.IMAGES){
				return serializeImage(primitiveObject);
			} else if(primitiveObject.getType() == PrimitiveTypes.FILE) {
				return serializeFile(primitiveObject);
			} else {
				return serializePrimitiveObject(primitiveObject);
			}
		}
		
		throw new Exception();
	}

}


