package org.gcube.contentmanagement.timeseriesservice.impl.curation.state;

import java.sql.ResultSet;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.state.GCUBEWSResource;
import org.gcube.common.dbinterface.CastObject;
import org.gcube.common.dbinterface.Condition;
import org.gcube.common.dbinterface.attributes.AggregatedAttribute;
import org.gcube.common.dbinterface.attributes.AggregationFunctions;
import org.gcube.common.dbinterface.attributes.AssignedAttribute;
import org.gcube.common.dbinterface.attributes.Attribute;
import org.gcube.common.dbinterface.attributes.BooleanAttribute;
import org.gcube.common.dbinterface.attributes.SimpleAttribute;
import org.gcube.common.dbinterface.conditions.ANDCondition;
import org.gcube.common.dbinterface.conditions.NOTCondition;
import org.gcube.common.dbinterface.conditions.OperatorCondition;
import org.gcube.common.dbinterface.persistence.ObjectPersistency;
import org.gcube.common.dbinterface.pool.DBSession;
import org.gcube.common.dbinterface.queries.CreateTableLike;
import org.gcube.common.dbinterface.queries.DropTable;
import org.gcube.common.dbinterface.queries.InsertFromSelect;
import org.gcube.common.dbinterface.queries.Select;
import org.gcube.common.dbinterface.queries.Update;
import org.gcube.common.dbinterface.queries.alters.ModifyColumnType;
import org.gcube.common.dbinterface.tables.SimpleTable;
import org.gcube.common.dbinterface.tables.Table;
import org.gcube.common.dbinterface.tables.TableFromSubselect;
import org.gcube.common.dbinterface.types.Type;
import org.gcube.common.dbinterface.types.Type.Types;
import org.gcube.common.dbinterface.utils.Utility;
import org.gcube.contentmanagement.codelistmanager.entities.CodeList;
import org.gcube.contentmanagement.codelistmanager.util.csv.ImportUtil;
import org.gcube.contentmanagement.timeseriesservice.impl.context.ImportContext;
import org.gcube.contentmanagement.timeseriesservice.impl.context.ServiceContext;
import org.gcube.contentmanagement.timeseriesservice.impl.curation.Curation;
import org.gcube.contentmanagement.timeseriesservice.impl.editing.ColumnEditor;
import org.gcube.contentmanagement.timeseriesservice.impl.editing.DimensionEditor;
import org.gcube.contentmanagement.timeseriesservice.impl.editing.Edit;
import org.gcube.contentmanagement.timeseriesservice.impl.editing.RuleEditor;
import org.gcube.contentmanagement.timeseriesservice.impl.curation.guessing.GuessingObject;
import org.gcube.contentmanagement.timeseriesservice.impl.curation.guessing.GuessingPrefetcher;
import org.gcube.contentmanagement.timeseriesservice.impl.curation.rules.Rule;
import org.gcube.contentmanagement.timeseriesservice.impl.history.CurationHistory;
import org.gcube.contentmanagement.timeseriesservice.impl.history.CurationHistoryItem;
import org.gcube.contentmanagement.timeseriesservice.impl.history.CurationHistoryItem.OperationType;
import org.gcube.contentmanagement.timeseriesservice.impl.importer.state.ImportResource;
import org.gcube.contentmanagement.timeseriesservice.impl.timeseries.operations.Operation;
import org.gcube.contentmanagement.timeseriesservice.impl.timeseries.operations.util.FilterExplorer;
import org.gcube.contentmanagement.timeseriesservice.impl.utils.Constants;
import org.gcube.contentmanagement.timeseriesservice.impl.utils.Util;
import org.gcube.contentmanagement.timeseriesservice.stubs.ValueNotCompatibleFault;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.ColumnDefinition;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.DataType;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.EntryType;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.FilterCondition;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.GuessDimensionArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.RuleItem;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.Status;
import org.globus.wsrf.NoSuchResourceException;
import org.globus.wsrf.ResourceException;


public class CurationResource extends GCUBEWSResource {

	public static final String ID_COLUMN_SUFFIX="_id";
	
	private CurationHistory history;
	
	protected static final String RP_ID = "Id";
	protected static final String RP_SOURCEID = "SourceId";
	protected static final String RP_SOURCENAME = "SourceName";
	protected static final String RP_TITLE = "Title";
	protected static String[] RPNames = { RP_ID, RP_TITLE, RP_SOURCEID,  RP_SOURCENAME };
	private Edit fieldEditor=null;
	
	private int totalLine;
	
	private Map<String, int[]> fieldLenght;
	
	private ColumnDefinition[] columnDefinition; 
	
	private Status underCreationState; 
	
	private SimpleTable table;
	/**
     * {@inheritDoc}
     */
    public String[] getPropertyNames() {
    	return RPNames;
    }
	
	
	@Override
	protected void initialise(Object... initParameters) throws Exception {
		this.setUnderCreationState(Status.Open);
		this.setId((String) initParameters[0]);
		this.setSourceId((String) initParameters[4]);
		this.setSourceName(getImportResource().getImporterReference().getTitle());
		this.setTitle((String) initParameters[1]);
		logger.info("starting resource intialization "+this.getId());
		this.setTotalLine(getImportResource().getImporterReference().getTotalLines());
		Timestamp now= new Timestamp(System.currentTimeMillis());
		
		final DBSession session= DBSession.connect();
		//session.disableAutoCommit();
		
		//starting the guesser on import table
		
		Thread guessereTh = new Thread(new GuessingPrefetcher(getImportResource().getTable().getTableName(), this.getId(), 
				getImportResource().getImporterReference().getColumnsDefinition()));
		
		ServiceContext.getContext().setScope(guessereTh, ServiceContext.getContext().getScope());
		
		guessereTh.start();
		
		Curation curation = new Curation(this.getId());
		curation.setTitle(this.getTitle());
		curation.setDescription((String) initParameters[2]);
		curation.setCreator((String) initParameters[3]);
		curation.setSourceId(this.getSourceId());
		curation.setSourceName(this.getSourceName());
		curation.setDate(new Timestamp(System.currentTimeMillis()));
		curation.setLength(getImportResource().getImporterReference().getTotalLines());
		curation.setType(getImportResource().getImporterReference().getType());
		curation.setPublisher(getImportResource().getImporterReference().getPublisher());
		curation.setRights(getImportResource().getImporterReference().getRights());
		curation.setScope( ServiceContext.getContext().getScope().toString());
		curation.store();
				
		//CreateTable create= new CreateTable(this.getTableName());
		CreateTableLike createLike= DBSession.getImplementation(CreateTableLike.class);
		createLike.setTableLike(getImportResource().getTable());
		createLike.setTableName(Constants.getCurationTable(this.getId()));
		logger.trace(createLike.getExpression());
		this.table= createLike.execute(session);
		
		
		//indexes creation
		/*
		for (String field :this.table.getFieldsMapping().keySet())
			Utility.createIndexOnField(table, field, false).execute(session);
		*/
		
		this.history=new CurationHistory(this.getId(),this.getSourceId(),this.getSourceName(),now,(String) initParameters[3]);
		this.history.store();
		
		this.columnDefinition= getImportResource().getImporterReference().getColumnsDefinition();
		this.setFieldLenght(getImportResource().getFieldLenght());
		new Thread(){
			public void run(){
				try {
					InsertFromSelect insert= DBSession.getImplementation(InsertFromSelect.class);
					insert.setTable(table);
					Select subquery= DBSession.getImplementation(Select.class);
					subquery.setTables(getImportResource().getTable());
					insert.setSubQuery(subquery);
					logger.trace(insert.getExpression());
					insert.execute(session);					
					setUnderCreationState("Close");	
				} catch (Exception e) {
					logger.error("error creating curation ",e);
					setUnderCreationState("Error");
					remove();
				}finally{
					session.release();
				}
			}
		}.start();
		logger.info("finished initialialization for the resource "+this.getId());
	}
	
	/**
	 * retrieves the import resource
	 * 
	 * @return a Import Resource
	 * @throws NoSuchResourceException -
	 * @throws ResourceException -
	 */
	public ImportResource getImportResource() throws NoSuchResourceException, ResourceException{
		ImportResource resource = (ImportResource)ImportContext.getPortTypeContext().getWSHome().find(ImportContext.getPortTypeContext().makeKey(this.getSourceId()));
		return resource;
	}
				
	/**
	 * allows user to enter in edit mode form dimension checking
	 * 
	 * @param fieldId field id
	 * @param dimensionId dimension id
	 * @param keyName key name
	 * @throws Exception -
	 */
	public void editDimension(String fieldId, String codelistId, String keyId) throws Exception {
		logger.trace("entering in edit mode with dimId "+codelistId+" fieldId "+fieldId+" keyId "+keyId);
		if (this.fieldEditor!=null)
				this.fieldEditor.dismiss();
		this.fieldEditor= new DimensionEditor(this.getId(), fieldId, codelistId, keyId, this.getTable(),
				this.fieldLenght.get(fieldId), this.totalLine, (checkDimension(codelistId, fieldId, keyId)==0));
		new Thread(){
			public void run(){
				try {
					fieldEditor.initialize();
				} catch (Exception e) {
					fieldEditor.setIsUnderInitialization("Error");
					fieldEditor.dismiss();
					fieldEditor=null;
				}
			}
		}.start();
	}

	/**
	 * allows user to enter in edit mode for rules checking
	 * 
	 * @param fieldId field id
	 * @param dimensionId dimension id
	 * @param keyName key name
	 * @throws Exception -
	 */
	public void editRules(String fieldId) throws Exception {
		logger.trace("entering in rule edit mode with  fieldId "+fieldId);
		if (this.fieldEditor!=null)
				this.fieldEditor.dismiss();
		
		this.fieldEditor= new RuleEditor(this.getId(), fieldId, this.getTable(),
				this.fieldLenght.get(fieldId), false, Curation.getCurationItem(this.getId()).getFieldRulesMapping().get(fieldId),this.getColumnDefinition());
		new Thread(){
			public void run(){
				try {
					fieldEditor.initialize();
				} catch (Exception e) {
					fieldEditor.setIsUnderInitialization("Error");
					fieldEditor.dismiss();
					fieldEditor=null;
				}
			}
		}.start();
	}
	
	/**
	 * allows user to enter in edit mode for type checking
	 * 
	 * @param fieldId field id
	 * @param dimensionId dimension id
	 * @param keyName key name
	 * @throws Exception -
	 */
	public void editColumn(String fieldId, Type type) throws Exception {
		logger.trace("entering in column edit mode with fieldId "+fieldId+" type "+type.getType().toString());
		if (this.fieldEditor!=null)
				this.fieldEditor.dismiss();
		logger.debug("the fieldLength for the field is "+this.fieldLenght.get(fieldId));
		this.fieldEditor= new ColumnEditor(this.getId(), fieldId, this.getTable(),
				this.fieldLenght.get(fieldId), false, type);
		new Thread(){
			public void run(){
				try {
					fieldEditor.initialize();
				} catch (Exception e) {
					fieldEditor.setIsUnderInitialization("Error");
					fieldEditor.dismiss();
					fieldEditor=null;
				}
			}
		}.start();
	}
	
	/**
	 * returns the count entries for this resource
	 * 
	 * @param onlyErrors if true returns only the wrong entries
	 * @return the count
	 * @throws Exception -
	 */
	public long getCount(boolean onlyErrors) throws Exception{
		logger.info("is under editing?"+(this.getFieldEditor()!=null));
		if (onlyErrors){
			if (this.isUnderEdit()) return fieldEditor.errorCount();
			else throw new GCUBEFault("edit not ready");
		}else{
			if (this.isUnderEdit())	return fieldEditor.totalCount();
			else{
				this.table.initializeCount();
				return this.table.getCount();
			}
		}
	}
	
	/**
	 * returns the entries of the resource as JSon
	 * 
	 * @param query the query
	 * @param onlyErrors true if the user wants only the wrong entry 
	 * @return the String as JSon
	 * @throws Exception -
	 */
	public String getDataAsJson(Select query, boolean onlyErrors) throws Exception{
		if (fieldEditor==null) {
			query.setTables(this.getTable());
			return query.getResultAsJSon(true);	
		}else return fieldEditor.getResultAsJson(query, onlyErrors);	
	}
		
	
	/**
     * Returns the id.
     * 
     * @return the id.
     */
    public String getId() throws ResourceException {
    	return (String) this.getResourcePropertySet().get(RP_ID).get(0);
    }
    
    /**
     * 
     * @return
     * @throws ResourceException
     */
    public String getSourceId() throws ResourceException {
    	return (String) this.getResourcePropertySet().get(RP_SOURCEID).get(0);
    }
    
    /**
     * 
     * @return
     * @throws ResourceException
     */
    public String getSourceName() throws ResourceException {
    	return (String) this.getResourcePropertySet().get(RP_SOURCENAME).get(0);
    }
    
    /**
     * 
     * @return
     * @throws ResourceException
     */
    public String getTitle() throws ResourceException {
    	return (String) this.getResourcePropertySet().get(RP_TITLE).get(0);
    }
    
    /**
     * 
     * @param id
     * @throws ResourceException
     */
    public synchronized void setTitle(String title) throws ResourceException {
    	this.getResourcePropertySet().get(RP_TITLE).clear();
		this.getResourcePropertySet().get(RP_TITLE).add(title);
    }

    
    /**
     * 
     * @param id
     * @throws ResourceException
     */
    public synchronized void setId(String id) throws ResourceException {
    	this.getResourcePropertySet().get(RP_ID).clear();
		this.getResourcePropertySet().get(RP_ID).add(id);
    }

    /**
     * 
     * @param sourceId
     * @throws ResourceException
     */
    public synchronized void setSourceId(String sourceId) throws ResourceException {
    	this.getResourcePropertySet().get(RP_SOURCEID).clear();
		this.getResourcePropertySet().get(RP_SOURCEID).add(sourceId);
    }
    
    /**
     * 
     * @param sourceName
     * @throws ResourceException
     */
    public synchronized void setSourceName(String sourceName) throws ResourceException {
    	this.getResourcePropertySet().get(RP_SOURCENAME).clear();
		this.getResourcePropertySet().get(RP_SOURCENAME).add(sourceName);
    }
    
    /**
     * 
     * @return
     * 
     */
	public SimpleTable getTable(){
		return this.table;
	}
	
	/**
	 * 
	 * @return
	 */
	public boolean isUnderEdit(){
		return this.fieldEditor!=null;
	}

	/**
	 * 
	 * @return
	 */
	public Edit getFieldEditor() {
		return fieldEditor;
	}

	/**
	 * 
	 * @param fieldEditor
	 */
	public void setFieldEditor(Edit fieldEditor) {
		this.fieldEditor = fieldEditor;
	}
	
	
	/**
	 * 
	 * @return
	 */
	public ColumnDefinition[] getColumnDefinition() {
		return columnDefinition;
	}

	/**
	 * 
	 * @param columnDefinition
	 */
	public void setColumnDefinition(ColumnDefinition[] columnDefinition) {
		this.columnDefinition = columnDefinition;
	}
	
	/**
	 * 
	 * @throws Exception
	 */
	public void saveColumnDefinition() throws Exception{
		if (this.getFieldEditor()!=null){
			this.getFieldEditor().save();
			ColumnDefinition oldColumnDefinition =Operation.getColumnDefinitionReference(this.fieldEditor.getFieldId(), this.columnDefinition);
			logger.debug("the fieldLength for field "+this.getFieldEditor().getFieldId()+" is "+this.getFieldEditor().getFieldLength());
			this.fieldLenght.put(this.getFieldEditor().getFieldId(), this.getFieldEditor().getFieldLength());
			setColumnDefinitionReference(this.getFieldEditor().getFieldId(),this.fieldEditor.getTemporaryColumnDefinition(oldColumnDefinition));
			for (CurationHistoryItem item: fieldEditor.getHistoryItems())
				this.history.addItem(item);
			this.closeEditing();
		}else throw new GCUBEFault("the service is not in edit mode");
	}
	
	
	
	private ColumnDefinition getColumnDefinitionReference(String fieldId) throws Exception{
		for (ColumnDefinition def: this.columnDefinition)
			if (def.getId().compareTo(fieldId)==0)
				return def;
		throw new Exception("fieldId not found");	
	}
	
	private void setColumnDefinitionReference(String fieldId, ColumnDefinition columnDefintion) throws Exception{
		for (int i = 0; i < this.columnDefinition.length; i++)
			if (this.columnDefinition[i].getId().compareTo(fieldId)==0){
				this.columnDefinition[i] = columnDefintion;
				return;
			}
		throw new Exception("fieldId not found");	
	}
	
	/**
	 * perform the guessing for a selected field id
	 * 
	 * @param fieldId field id
	 * @param limit the query limit
	 * @return the Dimension Array
	 * @throws Exception -
	 */
	public GuessDimensionArray guess(String fieldId) throws Exception{
		GuessDimensionArray toReturn; 
		try{
			if ((toReturn=GuessingPrefetcher.getGuessing(this.getId(), fieldId))!=null) return toReturn;
		}catch (Exception e) {
			logger.warn("guesser prefetching not ready for id "+this.getId(), e);
		}
		logger.trace("guesser prefetching not ready for id "+this.getId());
		return GuessingPrefetcher.guess(this.table.getTableName(), fieldId, ServiceContext.getContext().getScope());
	}
	
	/**
	 * 
	 * @param fieldId field id
	 * @param label the label 
	 * @param type the type
	 * @throws Exception -
	 */
	public void setLabel(String fieldId, String label) throws Exception{
		ColumnDefinition ref= getColumnDefinitionReference(fieldId);
		ref.setLabel(label);
	}
	
	
	/**
     * {@inheritDoc}
     */
	public void remove(){
		try{
			if (this.fieldEditor!=null) this.fieldEditor.dismiss();
		}catch(Exception e){logger.error("error dismissing fieldEditor",e);}
		try{
			DBSession session= DBSession.connect();
			DropTable dropTable=DBSession.getImplementation(DropTable.class); 
			dropTable.setTableName(this.getTable().getTableName());
			dropTable.execute(session);
			session.release();
		}catch(Exception e){logger.error("error deleting table",e);}
		try{	
			ObjectPersistency.get(Curation.class).deleteByKey(this.getId());
		}catch(Exception e){logger.error("error deleting entry in curation table",e);}
		try{	
			ObjectPersistency.get(GuessingObject.class).deleteByKey(this.getId());
		}catch(Exception e){logger.error("error deleting guessing entry in curation table",e);}
	}
	
	public Status getUnderCreationState() {
		return underCreationState;
	}


	/**
	 * 
	 * @param underCreationState
	 */
	public void setUnderCreationState(Status underCreationState) {
		this.underCreationState = underCreationState;
	}

	/**
	 * 
	 * @param underCreationState
	 */
	public void setUnderCreationState(String underCreationState) {
		this.underCreationState = Status.fromString(underCreationState);
	}


	/**
	 * 
	 * @return
	 */
	public Map<String, int[]> getFieldLenght() {
		return fieldLenght;
	}


	/**
	 * 
	 * @param fieldLenght
	 */
	public void setFieldLenght(Map<String, int[]> fieldLenght) {
		this.fieldLenght = fieldLenght;
	}


	/**
	 * 
	 * @return a state
	 * @throws Exception -
	 */
	public Status getInitializeEditingState() throws Exception{
		if (this.fieldEditor!=null)
			return this.fieldEditor.getIsUnderInitialization();
		else throw new Exception("you are not in edit mode");
	}
	
	protected void onRemove() throws ResourceException {
		this.remove();
		super.onRemove();
	}

	/**
	 * 
	 * @param columnType column type
	 * @param fieldId field id 
	 * @param dataType data type
	 * @throws Exception -
	 */
	public void setColumn(EntryType columnType, String fieldId) throws Exception{
		ColumnDefinition def= getColumnDefinitionReference(fieldId);
		def.setDimension(null);
		def.setColumnType(columnType);
		def.setKey(null);
		if (columnType==EntryType.Undefined){
			if (this.table.getFieldsMapping().get(fieldId).getType()!=Util.mapJavaToSql(DataType.Text)){
				modifyColumnType(fieldId, DataType.Text, this.getFieldLenght().get(fieldId));
				def.setValueType(DataType.Text);
			}
			logger.debug("setColumnAs: changing nothing in the data type");
		}
		
		history.addItem(new CurationHistoryItem("","saved column "+this.getColumnDefinitionReference(fieldId).getLabel()+" to type "+columnType.toString(),new Timestamp(System.currentTimeMillis()),OperationType.COLUMN_TYPE_SET));	
		//this.history.addItem(new CurationHistoryItem("","set column "+def.getLabel()+" to "+def.getColumnType(),new Timestamp(System.currentTimeMillis()),OperationType.COLUMN_TYPE_SET));
	}
	
	
	/**
	 * 
	 * @param oldId old id 
	 * @param newId new id 
	 * @param fieldId field id 
	 * @throws Exception -
	 */
	public void replaceById(String oldId, String newId, String fieldId ) throws Exception {
		if (this.fieldEditor==null){
			if (this.getColumnDefinitionReference(fieldId).getColumnType()!=EntryType.Dimension){
				logger.error("replaceById error: the field MUST be a dimension");	
				throw new Exception("replaceById error: the field MUST be a dimension");
			}
			logger.debug("replacing a Id in a column currently not under editing");

			DBSession session=DBSession.connect();
			try{
				
				CodeList codelist = CodeList.get(this.getColumnDefinitionReference(fieldId).getDimension().getId());
				
				String dimTableName = codelist.getTable().getTableName();
				String codeField = codelist.getCodeColumnId();
				Type codeType = codelist.getTable().getFieldsMapping().get(codeField); 
				
				CastObject idCast = Utility.getCast(newId, codeType);
				
				Select retrieveIdQuery= DBSession.getImplementation(Select.class);
				SimpleTable dimensionTable=new SimpleTable(dimTableName);
				retrieveIdQuery.setAttributes(new SimpleAttribute(this.getColumnDefinitionReference(fieldId).getKey().getName()));
				retrieveIdQuery.setFilter(new OperatorCondition<SimpleAttribute, CastObject>(new SimpleAttribute(codeField),idCast,"="));
				retrieveIdQuery.setTables(dimensionTable);

				ResultSet retrieveIdQueryRes=retrieveIdQuery.getResults(session);

				String newValue;
				if (retrieveIdQueryRes.next())  {
					newValue=retrieveIdQueryRes.getString(0);
					Type fieldType=this.getTable().getFieldsMapping().get(fieldId);
					if (newValue.length()>this.fieldLenght.get(fieldId)[0]){
						this.modifyColumnType(fieldId, Util.mapSqlToJava(fieldType.getType()), newValue.length());
						int precisionFloat = 0;
						if (newValue.contains("."))
							precisionFloat= newValue.split("\\.")[1].length();
						this.fieldLenght.put(fieldId,new int[]{newValue.length(), precisionFloat});
					}
					Update updateQuery= DBSession.getImplementation(Update.class);
					updateQuery.setFilter(new OperatorCondition<SimpleAttribute, Long>(new SimpleAttribute(fieldId+ID_COLUMN_SUFFIX),new Long(oldId),"="));
					CastObject cast=Utility.getCast(newValue, fieldType);
					updateQuery.setOperators(new OperatorCondition<SimpleAttribute, CastObject>(new SimpleAttribute(fieldId), cast, "="),new OperatorCondition<SimpleAttribute, CastObject>(new SimpleAttribute(fieldId+ID_COLUMN_SUFFIX), idCast, "="));
					updateQuery.setTable(this.table);
					updateQuery.execute(session);
				} else throw new ValueNotCompatibleFault();
			}finally{
				session.release();
			}
			this.history.addItem(new CurationHistoryItem("","replaced value where id was "+oldId+" in field  "+this.getColumnDefinitionReference(fieldId).getLabel(),new Timestamp(System.currentTimeMillis()),OperationType.REPLACE_BY_ID));
		}else{
			if(this.fieldEditor.getFieldId().compareTo(fieldId)!=0){
					fieldEditor.replaceDistinctIds(fieldId, oldId, newId, this.getColumnDefinitionReference(fieldId).getKey().getName());
			}else fieldEditor.replaceIds(fieldId, oldId, newId);
			fieldEditor.getHistoryItems().add(new CurationHistoryItem("","replaced value where id was "+oldId+" in field  "+this.getColumnDefinitionReference(fieldId).getLabel(),new Timestamp(System.currentTimeMillis()),OperationType.REPLACE_BY_ID));
		}
	}

	
	
	/**
	 * 
	 * 
	 * @param oldValue old value
	 * @param newValue new value
	 * @param fieldId field id
	 * @throws Exception -
	 */
	public void replaceByValue(String oldValue, String newValue, String fieldId ) throws Exception {
		if (this.fieldEditor==null){
			if (this.getColumnDefinitionReference(fieldId).getColumnType()==EntryType.Dimension){
				logger.error("A dimension field is not modifiable by value");	
				throw new Exception("A dimension field is not modifiable by value");
			}
			DBSession session = DBSession.connect();
			logger.info("modifying a column currently not under editing");
			Type fieldType=this.getTable().getFieldsMapping().get(fieldId);
			
			if (newValue.length()>this.fieldLenght.get(fieldId)[0] || ImportUtil.getAfterDotLength(newValue)>this.fieldLenght.get(fieldId)[1]){
				int valueLength = this.fieldLenght.get(fieldId)[0];
				if (newValue.toString().length()>this.fieldLenght.get(fieldId)[0])
					valueLength = newValue.toString().length();
				int precisionFloat = this.fieldLenght.get(fieldId)[1];
				if (ImportUtil.getAfterDotLength(newValue.toString())>this.fieldLenght.get(fieldId)[1])
					precisionFloat = ImportUtil.getAfterDotLength(newValue.toString());
				this.fieldLenght.put(fieldId,new int[]{valueLength, precisionFloat});
				this.modifyColumnType(fieldId, Util.mapSqlToJava(fieldType.getType()), this.fieldLenght.get(fieldId));
			}
			
				
			//TODO insert rule control
			
			Update updateQuery= DBSession.getImplementation(Update.class);
			
			CastObject oldValueCast=Utility.getCast(oldValue, fieldType);
	
			updateQuery.setFilter(new OperatorCondition<SimpleAttribute, CastObject>(new SimpleAttribute(fieldId),oldValueCast,"="));
			
			CastObject newValueCast=Utility.getCast(newValue, fieldType);
								
			updateQuery.setOperators(new OperatorCondition<SimpleAttribute, CastObject>(new SimpleAttribute(fieldId), newValueCast, "="));
			updateQuery.setTable(this.table);
			updateQuery.execute(session);
			
			session.release();
			//setting history item
			this.history.addItem(new CurationHistoryItem("","replaced value where value was "+oldValue+" with "+newValue+" in field  "+this.getColumnDefinitionReference(fieldId).getLabel(),new Timestamp(System.currentTimeMillis()),OperationType.REPLACE_BY_VALUE));
		} else{
			this.fieldEditor.replaceDistinctValue(fieldId, newValue, oldValue);
			fieldEditor.getHistoryItems().add(new CurationHistoryItem("","replaced value where value was "+oldValue+" with "+newValue+" in field  "+this.getColumnDefinitionReference(fieldId).getLabel(),new Timestamp(System.currentTimeMillis()),OperationType.REPLACE_BY_VALUE));
		}
		
	}

	
	/**
	 * returns which column is actually in edit mode 
	 *  
	 * @return the column definition
	 * @throws Exception -
	 */
	public ColumnDefinition columnInEditMode() throws Exception {
		if (this.fieldEditor==null) throw new Exception("the service is not in edit mode");
		ColumnDefinition def= getColumnDefinitionReference(this.getFieldEditor().getFieldId());
		return this.fieldEditor.getTemporaryColumnDefinition(def);
	}

	/**
	 * 
	 * @return the total entries count
	 */
	public long getTotalLine() {
		return totalLine;
	}


	/**
	 * sets the total resource entries number
	 * 
	 * @param totalLine
	 */
	public void setTotalLine(int totalLine) {
		this.totalLine = totalLine;
	}

	
	/**
	 * removes a column
	 * 
	 * @param fieldId the field id
	 * @throws Exception -
	 */
	public void removeColumn(String fieldId) throws Exception {
		if (this.fieldEditor!=null) throw new Exception("a column cannot be removed in edit mode"); 
		DBSession session= DBSession.connect();
		String fieldLabel;
		try{
			fieldLabel=this.getColumnDefinitionReference(fieldId).getLabel();
			Utility.dropColumn(fieldId, this.table).execute(session);
			if (this.getColumnDefinitionReference(fieldId).getColumnType().equals(EntryType.Dimension))
				Utility.dropColumn(fieldId+ID_COLUMN_SUFFIX, this.table).execute(session);
		}catch(Exception e){
			logger.error("the fieldId "+fieldId+" does not exist in the table");
			throw new Exception("the fieldId "+fieldId+" does not exist in the table",e); 
		}finally{
			session.release();
		}
		List<ColumnDefinition> newColumnDefinitionArray= new ArrayList<ColumnDefinition>(this.getColumnDefinition().length-1);
		for (ColumnDefinition columnDefinition: this.getColumnDefinition())
			if (columnDefinition.getId().compareTo(fieldId)!=0) newColumnDefinitionArray.add(columnDefinition);
		this.setColumnDefinition(newColumnDefinitionArray.toArray(new ColumnDefinition[this.getColumnDefinition().length-1]));
		Curation cur = Curation.getCurationItem(this.getId());
		cur.removeAllRulesForField(fieldId);
		cur.store();
		//setting history item
		this.history.addItem(new CurationHistoryItem("","removed column "+fieldLabel,new Timestamp(System.currentTimeMillis()),OperationType.REMOVE_COLUMN));
	}
	
	/**
	 * checks how many errors there will be in edit mode for a specific dimension setting
	 * 
	 * @param dimensionId dimension id
	 * @param fieldId field id
	 * @param keyName key name
	 * @return the error count
	 * @throws Exception -
	 */
	public long checkDimension(String dimensionId, String fieldId, String keyId) throws Exception{
		DBSession session= DBSession.connect();

		try{
			CodeList codelist = CodeList.get(dimensionId);
			SimpleTable dimensionTable = codelist.getTable();
			
			
			final String curationTableAlias= "curtab";
			final String dimensionTableAlias="dimtab";
			final String countAlias="count";

			
			
			Select goodCellsCount= DBSession.getImplementation(Select.class);
			goodCellsCount.setAttributes(new AssignedAttribute<AggregatedAttribute>(new SimpleAttribute(countAlias),new AggregatedAttribute(keyId,AggregationFunctions.COUNT)), new SimpleAttribute(keyId));
			goodCellsCount.setTables(dimensionTable);
			goodCellsCount.setGroups(new SimpleAttribute(keyId));

			Select totalCorrectCount= DBSession.getImplementation(Select.class);
			totalCorrectCount.setAttributes(new AggregatedAttribute("*",AggregationFunctions.COUNT));
			CastObject fieldCastToString = Utility.getCastToString(new SimpleAttribute(fieldId,curationTableAlias)); 
			CastObject dimCastToString =Utility.getCastToString(new SimpleAttribute(keyId,dimensionTableAlias));
			totalCorrectCount.setFilter(new ANDCondition( new OperatorCondition<CastObject, CastObject>(fieldCastToString, dimCastToString,"=") , 
					new OperatorCondition<Attribute,Integer>(new SimpleAttribute(countAlias,dimensionTableAlias),1,"=")));
			totalCorrectCount.setTables(new Table(this.table.getTableName(),curationTableAlias), new TableFromSubselect(dimensionTableAlias,goodCellsCount));

			//CountQuery goodCellsCount= new CountQuery(countAlias,null,new Attribute[]{new Attribute(keyName)},null, new Table[]{new Table(tableName)});
			//goodCellsCount.setGroupBy(new Attribute[]{new Attribute(keyName)});
			//CountQuery totalCorrectCount= new CountQuery(null, null, new ANDCondition( new OperatorCondition<Attribute, Attribute>(new Attribute(curationTableAlias, fieldId), new Attribute(dimensionTableAlias, keyName),"=") , new OperatorCondition<Attribute,Integer>(new Attribute(dimensionTableAlias,countAlias),1,"=")),new Table[]{});  
			logger.trace(totalCorrectCount.getExpression());
			ResultSet totalCorrectRes=totalCorrectCount.getResults(session);
			totalCorrectRes.next();
			long size= this.getCount(false)-totalCorrectRes.getLong(1);
			logger.debug("the dimension check count is "+size);		
			return size;
		}finally{
			session.release();
		}
	}
	
	/**
	 * checks how many errors there will be in edit mode for applying a specific rule
	 * 
	 * @param fieldId field id
	 * @param filters list of the filter representing the applied rules
	 * @return the error count
	 * @throws Exception -
	 */
	public long checkRules(String fieldId, FilterCondition[] filters) throws Exception{
		DBSession session= null;
		try{
			session = DBSession.connect();
			Select select= DBSession.getImplementation(Select.class);
			select.setAttributes(new AggregatedAttribute("*",AggregationFunctions.COUNT));
			final String TABLE_ALIAS="tableAlias";
			select.setTables(new Table(this.table.getTableName(), TABLE_ALIAS));
			Condition[] conditions= new Condition[filters.length];
				
			for (int i= 0 ; i<filters.length; i++)
				conditions[i]=FilterExplorer.getCondition(filters[i], this.getColumnDefinition(), TABLE_ALIAS, fieldId);
			
			select.setFilter(new NOTCondition(new ANDCondition(conditions)));
			ResultSet result = select.getResults(session);
			result.next();
			return result.getLong(1);
		}finally {
			if(session!=null)session.release();
		}
	}
	
	/**
	 * checks how many errors there will be in edit mode for applying a specific rule
	 * 
	 * @param fieldId field id
	 * @param filters list of the filter representing the applied rules
	 * @return the error count
	 * @throws Exception -
	 */
	public long checkTypeChange(String fieldId, Type newType) throws Exception{
		DBSession session= null;
		try{
			session = DBSession.connect();
			Select select= DBSession.getImplementation(Select.class);
			select.setAttributes(new AggregatedAttribute("*",AggregationFunctions.COUNT));
			select.setTables(this.table);
			CastObject cast= Utility.getCast(new SimpleAttribute(fieldId), newType);
			cast.setUseCastFunction(true);
			
			Condition cond = new OperatorCondition<CastObject, Object>(cast, null," IS ");
		
			select.setFilter(cond);
			logger.debug("checking query is "+select.getExpression());
			ResultSet result = select.getResults(session);
			result.next();
			return result.getLong(1);
		}finally {
			if(session!=null)session.release();
		}
	}
	
	/**
	 * allows user to exit from edit mode
	 * 
	 * @throws Exception -
	 */
	public void closeEditing() throws Exception{
		logger.debug("closing the edit mode");
		if (fieldEditor!=null){
			fieldEditor.dismiss();
			this.fieldEditor=null;
			this.store();
		}
	}

	
	/**
	 * 
	 * @param fieldId field id
	 * @param newValue new value
	 * @param rowId row id
	 * @throws GCUBEFault -
	 */
	public void modifyEntryValue(String fieldId, String newValue, long rowId) throws Exception {
		if (this.fieldEditor==null){
			if (this.getColumnDefinitionReference(fieldId).getColumnType()==EntryType.Dimension){
				logger.error("A dimension field is not modifiable by value");	
				throw new Exception("A dimension field is not modifiable by value");
			}
			logger.info("modifying a column currently not under editing using modifyEntryValue");
			
			DBSession session =DBSession.connect();
			//check new value validity
			Iterator<Rule> ruleIt= this.applyedRules(fieldId);
			Select select = DBSession.getImplementation(Select.class);
			ArrayList<Condition> conditions= new ArrayList<Condition>();
			
			CastObject valueCast = Utility.getCast(newValue.toString(), this.table.getFieldsMapping().get(fieldId));
			
			conditions.add(new OperatorCondition<CastObject, Object>(valueCast, null," IS NOT "));
			
			while (ruleIt.hasNext()){
				Rule rule= ruleIt.next();	
				conditions.add(FilterExplorer.getConditionUsingValue(rule.getFilter(), valueCast));
			}
			
			select.setAttributes(new BooleanAttribute("field", new ANDCondition(conditions.toArray(new Condition[conditions.size()]))));
			
			
			logger.debug("checking the new value: "+ select.getExpression());
			
			if (conditions.size()>0){
				ResultSet rs = select.getResults(session);
				rs.next();
				if (!rs.getBoolean(1)){ 
					logger.warn("the new value "+newValue+" is not compatible with the applyed rules");
					throw new ValueNotCompatibleFault();
				}
			}
			
			
			
			Type fieldType=this.getTable().getFieldsMapping().get(fieldId);
			if (newValue.length()>this.fieldLenght.get(fieldId)[0] || ImportUtil.getAfterDotLength(newValue)>this.fieldLenght.get(fieldId)[1]){
				int valueLength = this.fieldLenght.get(fieldId)[0];
				if (newValue.toString().length()>this.fieldLenght.get(fieldId)[0])
					valueLength = newValue.length();
				int precisionFloat = this.fieldLenght.get(fieldId)[1];
				if (ImportUtil.getAfterDotLength(newValue)>this.fieldLenght.get(fieldId)[1])
					precisionFloat = ImportUtil.getAfterDotLength(newValue);
				this.fieldLenght.put(fieldId,new int[]{valueLength, precisionFloat});
				this.modifyColumnType(fieldId, Util.mapSqlToJava(fieldType.getType()), this.fieldLenght.get(fieldId));
				logger.trace("the new precision for "+fieldId+" is "+valueLength+","+precisionFloat);
			}
			
			Update updateQuery= DBSession.getImplementation(Update.class);
			
			updateQuery.setFilter(new OperatorCondition<SimpleAttribute, Long>(new SimpleAttribute("ID"), new Long(rowId), "="));
			
			CastObject cast=Utility.getCast(newValue, fieldType);
			
			updateQuery.setOperators(new OperatorCondition<SimpleAttribute, CastObject>(new SimpleAttribute(fieldId), cast, "="));
			updateQuery.setTable(this.table);
			logger.debug("update query is "+updateQuery.getExpression());
			updateQuery.execute(session); 
			
			//new UpdateQuery(new OperatorCondition[]{new OperatorCondition<String, CastObject>(fieldId, new CastObject(newValue,keyType), "=")},new Table(this.getTableName()),new OperatorCondition<String, Long>("ID",new Long(rowId),"=")).execute();
			//setting history item
			session.release();
			this.history.addItem(new CurationHistoryItem("","modified value of field "+this.getColumnDefinitionReference(fieldId).getLabel()+" where row id is "+rowId+" with "+newValue,new Timestamp(System.currentTimeMillis()),OperationType.MODIFY_VALUE));
		} else 
			this.fieldEditor.replaceValue((int)rowId, (Object)newValue);
	}
	
		
	public String getPossibleValues(String word) throws Exception{
		if (!this.isUnderEdit()) throw new Exception("service not in edit mode");
		return this.fieldEditor.getPossibleValues(word);
	}
	
	/**
	 * 
	 * @param fieldId field id
	 * @param newId new id
	 * @param rowId row id
	 * @throws GCUBEFault -
	 */
	public void modifyEntryId(String fieldId, String newId, long rowId) throws Exception {
		if (this.fieldEditor==null){
			if (this.getColumnDefinitionReference(fieldId).getColumnType()!=EntryType.Dimension){
				logger.error("It MUST be a dimension field");	
				throw new Exception("It MUST be a dimension field");
			}
			logger.info("modifying a dimension column currently not under editing");
			
			CodeList codelist = CodeList.get(this.getColumnDefinitionReference(fieldId).getDimension().getId());
			
			String dimTableName = codelist.getTable().getTableName();
			String codeField = codelist.getCodeColumnId();
			Type codeType = codelist.getTable().getFieldsMapping().get(codeField); 
			
			DBSession session=DBSession.connect();
				
			Select retrieveValueQuery= DBSession.getImplementation(Select.class);
			CastObject idCast = Utility.getCast(newId, codeType);
			retrieveValueQuery.setAttributes(new SimpleAttribute(this.getColumnDefinitionReference(fieldId).getKey().getId()));
			retrieveValueQuery.setFilter(new OperatorCondition<SimpleAttribute, CastObject>(new SimpleAttribute(codeField),idCast,"="));
			retrieveValueQuery.setTables(new Table(dimTableName));
			ResultSet retrieveValueRes= retrieveValueQuery.getResults(session);		
					
			if (retrieveValueRes.next())  {
				String newValue= retrieveValueRes.getString(1);
				Type fieldType=this.getTable().getFieldsMapping().get(fieldId);
				
				Update updateQuery= DBSession.getImplementation(Update.class);
				
				updateQuery.setFilter(new OperatorCondition<SimpleAttribute, Long>(new SimpleAttribute("ID"), new Long(rowId), "="));
				
				CastObject cast=Utility.getCast(newValue, fieldType);

				updateQuery.setOperators(new OperatorCondition<SimpleAttribute, CastObject>(new SimpleAttribute(fieldId), cast, "="),new OperatorCondition<SimpleAttribute, CastObject>(new SimpleAttribute(fieldId+ID_COLUMN_SUFFIX), idCast, "="));
				updateQuery.setTable(this.table);
				updateQuery.execute(session);
				session.release();
			} else{
				session.release();
				throw new ValueNotCompatibleFault();
			}
			this.history.addItem(new CurationHistoryItem("","modified reference ID of field "+this.getColumnDefinitionReference(fieldId).getLabel()+" where row id is "+rowId+" with "+newId,new Timestamp(System.currentTimeMillis()),OperationType.MODIFY_ID));
		} else{
			if (this.fieldEditor.getFieldId().compareTo(fieldId)==0) this.fieldEditor.modifyEntryId(fieldId, newId, rowId);	
			else
				fieldEditor.modifyDistinctEntryId(fieldId, newId, rowId,this.getColumnDefinitionReference(fieldId).getDimension().getId(),this.getColumnDefinitionReference(fieldId).getKey().getName());
			this.fieldEditor.getHistoryItems().add(new CurationHistoryItem("","modified reference ID of field "+this.getColumnDefinitionReference(fieldId).getLabel()+" where row id is "+rowId+" with "+newId,new Timestamp(System.currentTimeMillis()),OperationType.MODIFY_ID));
		}
		
	}
	
	/**
	 * remove all the rows with wrong field (only in edit mode)
	 * 
	 * @throws Exception -
	 */
	public void removeAllErrors() throws Exception{
		if (this.isUnderEdit()){
			this.fieldEditor.removeAllErrors();
			this.fieldEditor.getHistoryItems().add(new CurationHistoryItem("","removed all errors from field "+this.getColumnDefinitionReference(this.fieldEditor.getFieldId()).getLabel(),new Timestamp(System.currentTimeMillis()),OperationType.REMOVE_ALL_ERRORS));
		}else throw new GCUBEFault("the service is not in edit mode");
	}
	
	
	/**
	 * remove a row with a wrong field (only in edit mode)
	 * 
	 * @param rowId the row id
	 * @throws Exception -
	 */
	public void removeError(long rowId) throws Exception{
		if (this.isUnderEdit()){
			this.fieldEditor.removeSingleError(rowId);
			this.totalLine= totalLine-1;
			Curation curation = Curation.getCurationItem(this.getId());
			curation.setLength((int)this.totalLine);
			this.fieldEditor.getHistoryItems().add(new CurationHistoryItem("","removed error from field "+this.getColumnDefinitionReference(this.fieldEditor.getFieldId()).getLabel()+" where row ID is "+rowId,new Timestamp(System.currentTimeMillis()),OperationType.REMOVE_ERROR));
		}else throw new GCUBEFault("the service is not in edit mode");
	}


	public CurationHistory getHistory() {
		return history;
	}


	public void setHistory(CurationHistory history) {
		this.history = history;
	}


	public void setTable(SimpleTable table) {
		this.table = table;
	}
	
	private void modifyColumnType(String fieldId,DataType dataType, int ... precision) throws Exception{
		ModifyColumnType modifyCol= DBSession.getImplementation(ModifyColumnType.class);
		modifyCol.setTable(this.table);
		modifyCol.setColumn(new SimpleAttribute(fieldId));
		logger.debug("the precision array in the field mapping for "+fieldId+" is "+this.table.getFieldsMapping().get(fieldId).getPrecisionArray());
		Type newType;
		Types types=Util.mapJavaToSql(dataType);
		if (types==Types.FLOAT || types==Types.TEXT || types == Types.BOOLEAN
				||   types==Types.TIME || types==Types.DATE ) newType = new Type(Util.mapJavaToSql(dataType));
		else newType = new Type(Util.mapJavaToSql(dataType), precision);
		modifyCol.setNewType(newType);
		
		modifyCol.setUseCast(this.table.getFieldsMapping().get(fieldId).getType()!=types);
		
		DBSession session= null;
		try{
			session= DBSession.connect();
			logger.debug("modifyColumnQuery: "+modifyCol.getExpression());
			modifyCol.execute(session);
		}finally{
			if (session!=null) session.release();
		}
	}


	public void applyRules(RuleItem[] rules, String fieldId) throws Exception{
		Curation curation = Curation.getCurationItem(this.getId());
		for (RuleItem rule: rules)
			curation.addRuleToField(fieldId, new Rule(rule.getName(), rule.getDescription(), FilterExplorer.generateRulesDescription(rule.getFilter()).toString(), rule.getFilter(), Util.mapJavaToSql(rule.getType())));
		curation.store();
	}


	public Iterator<Rule> applyedRules(String fieldID) throws Exception{
		Curation curation = Curation.getCurationItem(this.getId());
		if (curation.getFieldRulesMapping().get(fieldID) ==null) return new ArrayList<Rule>().iterator();
		else return curation.getFieldRulesMapping().get(fieldID).iterator();
	}


	public Iterator<Rule> applyableRules(String fieldId) throws Exception {
		Types type =Util.mapJavaToSql(this.getColumnDefinitionReference(fieldId).getValueType());
		return Rule.getByType(type);
	}
	
	public void removeAllRules(String fieldId) throws Exception{
		Curation curation = Curation.getCurationItem(this.getId());
		curation.getFieldRulesMapping().remove(fieldId);
		curation.store();
	}


	public void setProperties(String title, String description,
			String publisher, String rights) throws Exception {
		Curation curation = Curation.getCurationItem(this.getId());
		curation.setTitle(title);
		curation.setDescription(description);
		curation.setPublisher(publisher);
		curation.setRights(rights);
		curation.store();
	}
}
