package org.gcube.contentmanagement.timeseriesservice.impl.context;

import java.io.File;
import java.io.FileInputStream;
import java.io.StringReader;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.xml.namespace.QName;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.gcube.common.core.contexts.GCUBEServiceContext;
import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.informationsystem.client.AtomicCondition;
import org.gcube.common.core.informationsystem.client.ISClient;
import org.gcube.common.core.informationsystem.client.queries.GCUBEGenericResourceQuery;
import org.gcube.common.core.informationsystem.client.queries.GCUBERuntimeResourceQuery;
import org.gcube.common.core.informationsystem.notifier.ISNotifier;
import org.gcube.common.core.informationsystem.notifier.ISNotifier.GCUBENotificationTopic;
import org.gcube.common.core.resources.GCUBEGenericResource;
import org.gcube.common.core.resources.GCUBERuntimeResource;
import org.gcube.common.core.resources.runtime.AccessPoint;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.scope.GCUBEScope.Type;
import org.gcube.common.core.types.MapItemType;
import org.gcube.common.dbinterface.pool.DBSession;
import org.gcube.contentmanagement.codelistmanager.entities.CodeList;
import org.gcube.contentmanagement.graphtools.core.StatisticsGenerator;
import org.gcube.contentmanagement.lexicalmatcher.analysis.core.LexicalEngineConfiguration;
import org.gcube.contentmanagement.lexicalmatcher.analysis.run.CategoryGuesser;
import org.gcube.contentmanagement.timeseries.geotools.engine.TSGeoToolsConfiguration;
import org.gcube.contentmanagement.timeseries.geotools.gisconnectors.GISInformation;
import org.gcube.contentmanagement.timeseries.geotools.vti.VTIDataExtender;
import org.gcube.contentmanagement.timeseriesservice.impl.timeseries.TimeSeries;
import org.gcube.contentmanagement.timeseriesservice.impl.utils.ConfigurationConsumer;
import org.gcube.contentmanagement.timeseriesservice.impl.utils.ConfigurationParserHandler;
import org.gcube.contentmanagement.timeseriesservice.impl.utils.ConfigurationParserHandler.Pair;
import org.xml.sax.InputSource;


public class ServiceContext extends GCUBEServiceContext{

	/** Single context instance, created eagerly */
	private static ServiceContext cache = new ServiceContext();
	
	/** Returns cached instance */
	public static ServiceContext getContext() {return cache;}
	
	/** Prevents accidental creation of more instances */
	private ServiceContext(){};
	
	private String dbUsername;
	private String dbPassword;
	private String dblocation;
	private String dbPackage;
	private String dbdialectforhibernate;
	
	private StatisticsGenerator statisticsGenerator;
	
	private CategoryGuesser guesser;
	private LexicalEngineConfiguration possibleValueRetrieverConfiguration;
	
	private VTIDataExtender vtiDataExtender;
	private HashMap<String, TSMode> modePerScope = new HashMap<String, ServiceContext.TSMode>();
	private HashMap<String, MapItemType[]> propsPerScope = new HashMap<String,MapItemType[]>();
	private GCUBENotificationTopic  configurationResourceTopic;
	private ISNotifier notifier;
	
	
	@Override
	protected String getJNDIName() {
		return "gcube/contentmanagement/timeseriesservice";
	}

	public enum TSMode { TimeSeries, VTI };
	
	/**
	 * @return the tsMode
	 */
	public TSMode getTsMode() {
		logger.trace("caller scope is "+ServiceContext.getContext().getScope().toString()+" and value is "+modePerScope.get(ServiceContext.getContext().getScope().toString()));
		TSMode mode = modePerScope.get(ServiceContext.getContext().getScope().toString());
		if (mode==null) return TSMode.TimeSeries;
		else return mode;
	}

	
	/**
	 * @return the propsPerScope
	 */
	public MapItemType[] getProperties() {
		GCUBEScope scope = ServiceContext.getContext().getScope();
		logger.trace("the request for properties is from scope "+scope);
		if (scope.getType()== GCUBEScope.Type.VRE)
			scope = scope.getEnclosingScope();
		
		if (!propsPerScope.containsKey(scope.toString())){
			logger.trace("the scope "+scope+" has not been found in the properties");
			return new MapItemType[0];
		}
		else return propsPerScope.get(scope.toString()); 
		
	}

	/**
	 * @return the vtiDataExtender
	 */
	public VTIDataExtender getVtiDataExtender() {
		return vtiDataExtender;
	}

	
	
	/**
     * {@inheritDoc}
     */
	public void onInitialisation(){
		try{
			notifier= GHNContext.getImplementation(ISNotifier.class);
			//initializing topic
			configurationResourceTopic= new GCUBENotificationTopic(new QName("http://gcube-system.org/namespaces/informationsystem/registry","GenericResource"));
			configurationResourceTopic.setPrecondition("//profile[contains(.,'<SecondaryType>ServicesConfiguration</SecondaryType>') and contains(.,'<Name>TimeSeriesServiceSetup</Name>')]");
			configurationResourceTopic.setUseRenotifier(false);
						
			for (GCUBEScope scope : ServiceContext.getContext().getInstance().getScopes().values()){
				if (scope.getType()== Type.VRE) continue;
				initializeProperties(scope);
				//register to configuration change topic
				notifier.registerToISNotification(new ConfigurationConsumer(scope), Collections.singletonList(configurationResourceTopic), ServiceContext.getContext(), scope);
				
				//TODO: change, it should be per scope
				if (scope.getType()== Type.VO){
					this.setScope(scope);
					try{
						vtiDataExtender = new VTIDataExtender(getTSGeotoolsConfiguration());
					}catch (Exception e) {
						logger.warn("erro initializing vtiExtender",e);
					}
				}
			}
			
			
			Properties prop= new Properties();
			prop.load(new FileInputStream(this.getFile("dbprop.properties", false)));
			///TODO:retrieve this information with RuntimeResources
			this.dbUsername=prop.getProperty("dbusername","");
			this.dbPassword=prop.getProperty("dbpassword","");
			this.dblocation=prop.getProperty("dblocation","");
			
			
			this.dbPackage= prop.getProperty("dbpackage", "org.gcube.dbinterface.postgres");
			this.dbdialectforhibernate= prop.getProperty("dbdialectforhibernate","org.hibernate.dialect.PostgreSQLDialect");
			logger.trace(this.dbUsername+" "+this.dbPassword+" "+this.dblocation+" "+this.dbPackage);
			DBSession.initialize(this.dbPackage,this.dbUsername, this.dbPassword,this.dblocation);
			
			//guesser initialization
			LexicalEngineConfiguration conf = new LexicalEngineConfiguration();
			conf.setDatabaseUserName(this.getDBUserName());
			conf.setDatabasePassword(this.getDBPassword());
			conf.setDatabaseDriver(DBSession.drivers);
			conf.setDatabaseURL("jdbc:postgresql:"+dblocation);
			conf.setDatabaseDialect(this.getDbdialectforhibernate());
			conf.setDatabaseAutomaticTestTable("connectiontesttable");
			conf.setDatabaseIdleConnectionTestPeriod("3600");
			
			Map<String, String> fieldMapping = CodeList.getInfo();
			
			conf.setReferenceTable(CodeList.getTableName());
			
			logger.debug("fields are: "+fieldMapping.get("tableName")+" "+fieldMapping.get("id")+" "+fieldMapping.get("name")+" "+fieldMapping.get("description"));
			logger.debug("config dir is "+(String)this.getProperty("configDir", true));
			
			conf.setReferenceColumn(fieldMapping.get("tableName"));
			conf.setIdColumn(fieldMapping.get("id"));
			conf.setNameHuman(fieldMapping.get("name"));
			conf.setDescription(fieldMapping.get("description"));
			
			guesser= new CategoryGuesser((String)this.getProperty("configDir", true));
			guesser.init(conf);
			
			//configuring the statistical generator
			this.statisticsGenerator = new StatisticsGenerator();
			this.statisticsGenerator.init((String)this.getProperty("configDir", true)+File.separator, conf);
			
			//configuring the second guesser instance ...
			conf.setEntryAcceptanceThreshold(30);
			conf.setReferenceChunksToTake(-1);
			conf.setTimeSeriesChunksToTake(-1);
			conf.setUseSimpleDistance(false);
						
			this.possibleValueRetrieverConfiguration= conf;
							
			logger.trace("extender is null ?"+(vtiDataExtender==null));
			
			//creating trigger for readOnlyUser in the db (only if it not exists)
			triggerCreationForReadOnlyUser();
					

		}catch(Throwable e){logger.error("error initializing service",e);}
	}
	
	public void onShutdown(){
		try{
			if (this.getVtiDataExtender()!=null)
				this.getVtiDataExtender().shutDown();
			notifier.unregisterFromISNotification(this, Collections.singletonList(configurationResourceTopic), ServiceContext.getContext().getInstance().getScopes().values().toArray(new GCUBEScope[0]));
		}catch(Exception e){logger.error("error shuttingdown service",e);}
	}
	
	/**
	 * 
	 * @return
	 */
	public String getDBUserName(){
		return dbUsername;
	}
	
	/**
	 * 
	 * @return
	 */
	public String getDBPassword(){
		return dbPassword;
	}

	/**
	 * 
	 * @return
	 */
	public String getDbdialectforhibernate() {
		return dbdialectforhibernate;
	}

	/**
	 * 	
	 * @return
	 */
	public CategoryGuesser getGuesser() {
		return guesser;
	}

	/**
	 * @return the possibleValueRetriever
	 */
	public LexicalEngineConfiguration getPossibleValueRetrieverConfiguration() {
		return possibleValueRetrieverConfiguration;
	}

	/**
	 * @return the statisticsGenerator
	 */
	public StatisticsGenerator getStatisticsGenerator() {
		return statisticsGenerator;
	}

	


	
	
	private void initializeProperties(GCUBEScope scope){
		try{
			ISClient client = GHNContext.getImplementation(ISClient.class);
			GCUBEGenericResourceQuery query = client.getQuery(GCUBEGenericResourceQuery.class);
			query.addAtomicConditions(new AtomicCondition("/Profile/SecondaryType","ServicesConfiguration"), new AtomicCondition("/Profile/Name","TimeSeriesServiceSetup") );
			List<GCUBEGenericResource> resources = client.execute(query, scope);
			if (resources.size()==0) throw new Exception(); 
			GCUBEGenericResource resource = resources.get(0);
			parseAndSaveConfiguration(resource, scope);
		}catch (Exception e) {
			logger.warn("error retrieving the configuration resource for scope "+scope.toString(),e);
		}
	}
	
	public void parseAndSaveConfiguration(GCUBEGenericResource configurationResource, GCUBEScope scope) throws Exception{
		String resourceBody =configurationResource.getBody();
		SAXParserFactory factory = SAXParserFactory.newInstance();
		SAXParser saxParser = factory.newSAXParser();
		ConfigurationParserHandler handler = new ConfigurationParserHandler();
		saxParser.parse(new InputSource( new StringReader(resourceBody)), handler);
		
		for (Pair pair : handler.getPairs())
			try{
				modePerScope.put(scope.toString()+"/"+pair.key, TSMode.valueOf(pair.value));
				logger.trace("found for ts mode: "+scope.toString()+"/"+pair.key+" with value "+pair.value );
			}catch (Throwable e) {
				logger.warn("erorr resolving the ts mode",e);
			}
		
		propsPerScope.put(scope.toString(), handler.getMapItems().toArray(new MapItemType[handler.getMapItems().size()]));	
			
	}
	
	private void triggerCreationForReadOnlyUser(){
		DBSession session = null;
		try{
			session = DBSession.connect();
			session.executeUpdate("CREATE role readonlyuser with login;");
		}catch (Exception e) {
			logger.warn("error creating readonlyuser role ",e);
		}
		try{
			String tableName = TimeSeries.getTableName();
			session.executeUpdate("CREATE OR REPLACE FUNCTION grant_permission() RETURNS trigger  AS $$ BEGIN  EXECUTE 'GRANT SELECT ON TABLE ' || 'ts_' || replace(NEW.ifield0, '-', '_') || ' TO readonlyuser' ; RETURN NULL; END; $$ language plpgsql;");
			session.executeUpdate("CREATE TRIGGER permission  AFTER  INSERT ON "+tableName+" FOR EACH ROW EXECUTE PROCEDURE grant_permission();");
		}catch (Exception e) {
			logger.warn("error creating trigger",e);
		}
		if (session!=null) session.release();
	}
		
	
	public TSGeoToolsConfiguration getTSGeotoolsConfiguration() throws Exception{
		TSGeoToolsConfiguration configuration = new TSGeoToolsConfiguration();
		configuration.setConfigPath((String)ServiceContext.getContext().getProperty("configDir", true));

		configuration.setTimeSeriesDatabase("jdbc:postgresql:"+this.dblocation);
		configuration.setTimeSeriesUserName(this.dbUsername);
		configuration.setTimeSeriesPassword(this.dbPassword);

		ISClient client = GHNContext.getImplementation(ISClient.class);
		GCUBERuntimeResourceQuery query = client.getQuery(GCUBERuntimeResourceQuery.class);

		query.addAtomicConditions(new AtomicCondition("/Profile/Category","Gis"), new AtomicCondition("/Profile/Name", "TimeSeriesDataStore"));
		List<GCUBERuntimeResource> timeSeriesDatastoreResources =client.execute(query, this.getScope());
		if (timeSeriesDatastoreResources.size()==0){
			String erroreMessage = "no runtimeResources found for timeseries datastore";
			logger.error(erroreMessage);
			throw new Exception(erroreMessage);
		}

		boolean timeseriesDatastoreAccesspointFound= false;
		if (timeSeriesDatastoreResources.size()>0)
			for (AccessPoint accessPoint: timeSeriesDatastoreResources.get(0).getAccessPoints()){
				if (accessPoint.getEntryname().equals("jdbc")){
					configuration.setGeoServerDatabase(accessPoint.getEndpoint());
					configuration.setGeoServerUserName(accessPoint.getUsername());
					configuration.setGeoServerPassword(accessPoint.getPassword());
				}else if (accessPoint.getEntryname().equals("jdbcaquamaps")){
					configuration.setAquamapsDatabase(accessPoint.getEndpoint());
					configuration.setAquamapsUserName(accessPoint.getUsername());
					configuration.setAquamapsPassword(accessPoint.getPassword());
				}
				timeseriesDatastoreAccesspointFound = true;
			}
		if (!timeseriesDatastoreAccesspointFound) {
			String erroreMessage = "no accesspoint found for timeseries datastore entry";
			logger.error(erroreMessage);
			throw new Exception(erroreMessage);
		}
		return configuration;
	}
	
	public GISInformation getGISInformation() throws Exception{
		GISInformation gisInfo = new GISInformation();
		//preparing Geonework connection
		ISClient client = GHNContext.getImplementation(ISClient.class);
		GCUBERuntimeResourceQuery query = client.getQuery(GCUBERuntimeResourceQuery.class);		
		query.clearConditions();
		query.addAtomicConditions(new AtomicCondition("/Profile/Category","Gis"), new AtomicCondition("/Profile/Name", "GeoNetwork"));
		List<GCUBERuntimeResource> geonetworkResources =client.execute(query, ServiceContext.getContext().getScope());
		if (geonetworkResources.size()==0){
			String erroreMessage = "no runtimeResources found for Geonetwork";
			logger.error(erroreMessage);
			throw new Exception(erroreMessage);
		}

		boolean geoNetworkAccesspointFound= false;
		for (AccessPoint accessPoint: geonetworkResources.get(0).getAccessPoints()){
			if (accessPoint.getEntryname().equals("geonetwork")){
				gisInfo.setGeoNetworkUrl(accessPoint.getEndpoint());
				gisInfo.setGeoNetworkUserName(accessPoint.getUsername());
				gisInfo.setGeoNetworkPwd(accessPoint.getPassword());
				geoNetworkAccesspointFound = true;
				break;
			}
		}
		if (!geoNetworkAccesspointFound) {
			String erroreMessage = "no accesspoint found for geonetwork entry";
			logger.error(erroreMessage);
			throw new Exception(erroreMessage);
		}


		//preparing Geoserver connection
		query.clearConditions();
		query.addAtomicConditions(new AtomicCondition("/Profile/Category","Gis"), new AtomicCondition("/Profile/Name", "GeoServer"));
		List<GCUBERuntimeResource> geoserverResources =client.execute(query, this.getScope());
		if (geoserverResources.size()==0){
			String erroreMessage = "no runtimeResources found for GeoServer";
			logger.error(erroreMessage);
			throw new Exception(erroreMessage);
		}
		boolean geoServerAccesspointFound= false;
		for (AccessPoint accessPoint: geoserverResources.get(0).getAccessPoints()){
			if (accessPoint.getEntryname().equals("geoserver")){
				gisInfo.setGisUserName(accessPoint.getUsername());
				gisInfo.setGisPwd(accessPoint.getPassword());
				gisInfo.setGisUrl(accessPoint.getEndpoint());
				gisInfo.setGisDataStore(accessPoint.getProperty("timeseriesDataStore"));
				gisInfo.setGisWorkspace(accessPoint.getProperty("timeseriesWorkspace"));
				geoServerAccesspointFound = true;
				break;
			}
		}
		if (!geoServerAccesspointFound) {
			String erroreMessage = "no accesspoint found for geoserver entry";
			logger.error(erroreMessage);
			throw new Exception(erroreMessage);
		}
		
		return gisInfo;
	}
	
}
