package eu.dnetlib.msro.workflows.nodes.dedup;

import java.util.List;
import java.util.Map;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.Gson;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.msro.workflows.graph.Arc;
import eu.dnetlib.msro.workflows.nodes.SimpleJobNode;
import eu.dnetlib.msro.workflows.procs.Env;
import eu.dnetlib.rmi.data.hadoop.actionmanager.RawSet;
import eu.dnetlib.rmi.enabling.ISLookUpDocumentNotFoundException;
import eu.dnetlib.rmi.enabling.ISLookUpService;
import eu.dnetlib.rmi.manager.MSROException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * The Class PrepareConfiguredActionSetJobNode.
 */
public class PrepareConfiguredActionSetJobNode extends SimpleJobNode {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(PrepareConfiguredActionSetJobNode.class);

	/**
	 * The dedup config sequence param.
	 */
	private String dedupConfigSequence;

	/**
	 * The job property.
	 */
	private String jobProperty;

	/**
	 * The action set path param name.
	 */
	private String actionSetPathParam;

	private String setsParam;

	/**
	 * The service locator.
	 */
	@Autowired
	private UniqueServiceLocator serviceLocator;

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.msro.workflows.nodes.SimpleJobNode#execute(com.googlecode.sarasvati.NodeToken)
	 */
	@Override
	protected String execute(final Env env) throws Exception {

		final List<Map<String, String>> setList = Lists.newArrayList();

		final Map<String, String> set = Maps.newHashMap();

		final String actionSetId = getDedupConfigSequence();
		final ISLookUpService isLookUpService = serviceLocator.getService(ISLookUpService.class);
		final String basePath = isLookUpService.getResourceProfileByQuery(
				"/RESOURCE_PROFILE[./HEADER/RESOURCE_TYPE/@value='ActionManagerServiceResourceType']//SERVICE_PROPERTIES/PROPERTY[@key='basePath']/@value/string()");
		if (StringUtils.isBlank(basePath)) {
			throw new IllegalStateException("missing basePath in ActionManagerService");
		}

		String actionSetDirectory;
		try {
			actionSetDirectory = isLookUpService.getResourceProfileByQuery(
					"/RESOURCE_PROFILE[./HEADER/RESOURCE_TYPE/@value='ActionManagerSetDSResourceType' and .//SET/@id = '" + actionSetId
							+ "']//SET/@ directory/string()");
		} catch (ISLookUpDocumentNotFoundException e) {
			throw new MSROException("missing directory in ActionSet profile: " + actionSetId);
		}

		final String rawSetId = RawSet.newInstance().getId();
		set.put("rawset", rawSetId);
		set.put("creationDate", DateUtils.now_ISO8601());
		set.put("set", actionSetId);
		set.put("enabled", "true");
		set.put("jobProperty", getJobProperty());

		env.setAttribute(set.get("jobProperty"), set.get("rawset"));

		final String path = basePath + "/" + actionSetDirectory + "/" + rawSetId;
		log.info("using action set path: " + path);
		env.setAttribute(getActionSetPathParam(), path);

		setList.add(set);
		final String sets = new Gson().toJson(setList);
		log.debug("built set: " + sets);

		env.setAttribute(getSetsParam(), sets);

		return Arc.DEFAULT_ARC;
	}

	/**
	 * Gets the job property.
	 *
	 * @return the job property
	 */
	public String getJobProperty() {
		return jobProperty;
	}

	/**
	 * Sets the job property.
	 *
	 * @param jobProperty the new job property
	 */
	public void setJobProperty(final String jobProperty) {
		this.jobProperty = jobProperty;
	}

	public String getActionSetPathParam() {
		return actionSetPathParam;
	}

	public void setActionSetPathParam(final String actionSetPathParam) {
		this.actionSetPathParam = actionSetPathParam;
	}

	public String getDedupConfigSequence() {
		return dedupConfigSequence;
	}

	public void setDedupConfigSequence(final String dedupConfigSequence) {
		this.dedupConfigSequence = dedupConfigSequence;
	}

	public String getSetsParam() {
		return setsParam;
	}

	public void setSetsParam(final String setsParam) {
		this.setsParam = setsParam;
	}
}
