package eu.dnetlib.openaire.blacklist;

import java.io.StringReader;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import javax.xml.ws.wsaddressing.W3CEndpointReference;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import com.google.common.collect.Lists;

import eu.dnetlib.enabling.database.rmi.DatabaseException;
import eu.dnetlib.enabling.database.rmi.DatabaseService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.resultset.client.IterableResultSetClient;
import eu.dnetlib.enabling.resultset.client.ResultSetClientFactory;
import eu.dnetlib.miscutils.collections.MappedCollection;
import eu.dnetlib.miscutils.functional.UnaryFunction;
import eu.dnetlib.openaire.hadoop.utils.HBaseTableUtils;

public class BlacklistManager {

	private static final Log log = LogFactory.getLog(BlacklistManager.class); // NOPMD by marko on 11/24/08 5:02 PM
	@Value("${dnet.openaire.blacklist.db.name}")
	private String blacklistDatabaseName;
	@Autowired
	private UniqueServiceLocator serviceLocator;
	@Autowired
	private ResultSetClientFactory resultSetClientFactory;
	@Autowired
	private OpenaireIdResolver openaireIdResolver;

	private List<String> getOriginalIds(final String id, final String entityType) {
		List<String> originalIds = Lists.newArrayList();
		// We need to find original ids only for entities that are deduplicated: result, organization, person.
		if (entityType.equals("result") || entityType.equals("organization") || entityType.equals("person")) {
			originalIds = openaireIdResolver.resolveIdentifier(id);
		}
		return originalIds;
	}

	public void addToBlacklist(final BlacklistEntry entry) throws DatabaseException {
		final List<String> sourceIds = getOriginalIds(entry.getSourceObject(), entry.getSourceType());
		final List<String> targetIds = getOriginalIds(entry.getTargetObject(), entry.getTargetType());
		entry.setOriginalSourceObjects(sourceIds);
		entry.setOriginalTargetObjects(targetIds);

		final DatabaseService dbService = serviceLocator.getService(DatabaseService.class);
		final String addQuery = String.format(
				"INSERT INTO blacklist(userid, relationship, provenance, iis_module, iis_status, status, source_object, source_type, target_object, target_type, "
						+ "ticket_id, original_source_objects, original_target_objects, note) "
						+ " VALUES(%1$s, %2$s, %3$s, %4$s, %5$s, %6$s, %7$s, %8$s, %9$s, %10$s, %11$s, %12$s, %13$s, %14$s)",
				asSqlParam(entry.getUser()),
				asSqlParam(entry.getRelationship()),
				asSqlParam(entry.getProvenance()),
				asSqlParam(entry.getIisModule()),
				asSqlParam(entry.getIisStatus()),
				asSqlParam(entry.getStatus()),
				asSqlParam(entry.getSourceObject()),
				asSqlParam(entry.getSourceType()),
				asSqlParam(entry.getTargetObject()),
				asSqlParam(entry.getTargetType()),
				asSqlParam(entry.getTicketId()),
				asSqlParam(joinCollectionForSQL(sourceIds.isEmpty() ? Lists.newArrayList(entry.getSourceObject()) : sourceIds)),
				asSqlParam(joinCollectionForSQL(targetIds.isEmpty() ? Lists.newArrayList(entry.getTargetObject()) : targetIds)),
				asSqlParam(entry.getNote()));
		log.debug("Adding new blacklist entry");
		this.safeUpdateSql(dbService, blacklistDatabaseName, addQuery);
	}

	private String joinCollectionForSQL(final Collection<String> coll) {
		return "{\"" + StringUtils.join(coll, "\",\"") + "\"}";
	}

	public void editBlacklistEntry(final BlacklistEntry entry) throws DatabaseException {

		final DatabaseService dbService = serviceLocator.getService(DatabaseService.class);
		final String editQuery = String.format(
				"UPDATE blacklist SET userid=%s, relationship=%s, provenance=%s, iis_module=%s, iis_status=%s, status=%s, source_type=%s, "
						+ "target_type=%s, ticket_id=%s, note=%s WHERE id=%s",
				asSqlParam(entry.getUser()),
				asSqlParam(entry.getRelationship()),
				asSqlParam(entry.getProvenance()),
				asSqlParam(entry.getIisModule()),
				asSqlParam(entry.getIisStatus()),
				asSqlParam(entry.getStatus()),
				asSqlParam(entry.getSourceType()),
				asSqlParam(entry.getTargetType()),
				asSqlParam(entry.getTicketId()),
				asSqlParam(entry.getNote()),
				asSqlParam(entry.getId()));
		log.debug("Editing blacklist entry: " + entry.getId());
		this.safeUpdateSql(dbService, blacklistDatabaseName, editQuery);
	}

	private String asSqlParam(final Object o) {
		if (o == null) {
			return "NULL";
		} else if ((o instanceof Number) || (o instanceof Boolean)) {
			return o.toString();
		} else {
			return "'" + o.toString() + "'";
		}
	}

	public void deleteFromBlacklist(final int entryId) throws DatabaseException {
		final DatabaseService dbService = serviceLocator.getService(DatabaseService.class);
		log.debug("Deleting entry " + entryId + " from blacklist");
		this.safeUpdateSql(dbService, blacklistDatabaseName, String.format("DELETE FROM blacklist WHERE id='%s' ", entryId));
	}

	public Iterable<BlacklistEntry> getBlacklist() throws DatabaseException {
		final String sqlQuery = "SELECT * from blacklist order by ticket_id";
		final W3CEndpointReference epr = serviceLocator.getService(DatabaseService.class).searchSQL(blacklistDatabaseName, sqlQuery);
		final IterableResultSetClient iter = resultSetClientFactory.getClient(epr);
		return this.getBlacklistIterable(iter);
	}

	public W3CEndpointReference getAcceptedBlacklistEntries() throws DatabaseException {
		final String sqlQuery =
				"SELECT source_type, unnest(original_source_objects) as source, target_type, unnest(original_target_objects) as target, relationship FROM blacklist WHERE status = 'ACCEPTED'";
		return serviceLocator.getService(DatabaseService.class).searchSQL(blacklistDatabaseName, sqlQuery);
	}

	private void safeUpdateSql(final DatabaseService dbService, final String dbName, final String sql) throws DatabaseException {
		log.info(sql);
		dbService.updateSQL(dbName, sql);
	}

	public Iterable<BlacklistEntry> getBlacklistIterable(final Iterable<String> xmlEntries) {
		return new MappedCollection<BlacklistEntry, String>(xmlEntries, new UnaryFunction<BlacklistEntry, String>() {

			@Override
			public BlacklistEntry evaluate(final String dbEntry) {
				final SAXReader saxReader = new SAXReader();
				final BlacklistEntry be = new BlacklistEntry();
				Document doc;
				try {
					doc = saxReader.read(new StringReader(dbEntry));
					be.setId(Integer.parseInt(doc.selectSingleNode("//FIELD[./@name='id']").getText()));
					be.setCreationDate(doc.selectSingleNode("//FIELD[./@name='creation_time']").getText());
					be.setLastUpdateDate(doc.selectSingleNode("//FIELD[./@name='last_update_time']").getText());
					be.setNote(doc.selectSingleNode("//FIELD[./@name='note']").getText());
					be.setRelationship(doc.selectSingleNode(".//FIELD[./@name='relationship']").getText());
					be.setStatus(STATUS.valueOf(StringUtils.upperCase(doc.selectSingleNode("//FIELD[./@name='status']").getText())));
					be.setTicketId(doc.selectSingleNode("//FIELD[./@name='ticket_id']").getText());
					be.setUser(doc.selectSingleNode("//FIELD[./@name='userid']").getText());
					be.setSourceObject(doc.selectSingleNode("//FIELD[./@name='source_object']").getText());
					be.setSourceType(doc.selectSingleNode("//FIELD[./@name='source_type']").getText());
					be.setTargetObject(doc.selectSingleNode("//FIELD[./@name='target_object']").getText());
					be.setTargetType(doc.selectSingleNode("//FIELD[./@name='target_type']").getText());
					final String provenance = doc.selectSingleNode("//FIELD[./@name='provenance']").getText();
					be.setProvenance(provenance);
					if (provenance.equalsIgnoreCase("iis")) {
						be.setIisModule(doc.selectSingleNode("//FIELD[./@name='iis_module']").getText());
						be.setIisStatus(IIS_STATUS.valueOf(StringUtils.upperCase(doc.selectSingleNode("//FIELD[./@name='iis_status']").getText())));
					}
					// get the array of original identifiers
					final List<Element> sources = doc.selectNodes("//FIELD[./@name='original_source_objects']/ITEM");
					if ((sources != null) && !sources.isEmpty()) {
						for (final Element e : sources) {
							be.getOriginalSourceObjects().add(e.getText());
						}
					}

					final List<Element> targets = doc.selectNodes("//FIELD[./@name='original_target_objects']/ITEM");
					if ((targets != null) && !targets.isEmpty()) {
						for (final Element e : targets) {
							be.getOriginalTargetObjects().add(e.getText());
						}
					}
				} catch (final DocumentException e) {
					log.error(e);
					throw new RuntimeException(e);
				}
				return be;
			}
		});
	}

	public Set<String> getListOfRelationships() {
		return HBaseTableUtils.listRelationships();
	}

	public enum IIS_STATUS {
		UNSOLVED, SOLVED
	}

	public enum STATUS {
		PENDING, ACCEPTED, REFUSED, DELETED
	}

}
