package eu.dnetlib.data.collector.plugins.filesystem;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileTime;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAccessor;
import java.util.Iterator;
import java.util.Set;

import eu.dnetlib.data.collector.rmi.CollectorServiceException;
import eu.dnetlib.data.collector.rmi.CollectorServiceRuntimeException;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.Iterators;
import com.google.common.collect.Sets;

/**
 * Class enabling lazy and recursive iteration of a filesystem tree. The iterator iterates over file paths.
 *
 * @author Andrea
 *
 */
public class FileSystemIterator implements Iterator<String> {

	/** The logger */
	private static final Log log = LogFactory.getLog(FileSystemIterator.class);

	private Set<String> extensions = Sets.newHashSet();
	private Iterator<Path> pathIterator;
	private String current;

	private boolean incremental = false;
	private LocalDate fromDate = null;
	private final DateTimeFormatter simpleDateTimeFormatter = DateTimeFormatter
			.ofPattern("yyyy-MM-dd")
			.withZone(ZoneId.systemDefault());

	public FileSystemIterator(final String baseDir, final String extensions) {
		this(baseDir,extensions, null);
	}

	public FileSystemIterator(final String baseDir, final String extensions, final String fromDate) {
		if(StringUtils.isNotBlank(extensions)) {
			this.extensions = Sets.newHashSet(extensions.split(","));
		}
		this.incremental = StringUtils.isNotBlank(fromDate);
		if (incremental) {
			//I expect fromDate in the format 'yyyy-MM-dd'. See class eu.dnetlib.msro.workflows.nodes.collect.FindDateRangeForIncrementalHarvestingJobNode .
			this.fromDate = LocalDate.parse(fromDate, simpleDateTimeFormatter);
			log.debug("fromDate string: " + fromDate + " -- parsed: " + this.fromDate.toString());
		}
		try {
			this.pathIterator = Files.newDirectoryStream(Paths.get(baseDir)).iterator();
			this.current = walkTillNext();
		} catch (Exception e) {
			log.error("Cannot initialize File System Iterator. Is this path correct? " + baseDir);
			e.printStackTrace();
			throw new CollectorServiceRuntimeException("Filesystem collection error.", e);
		}

	}

	@Override
	public boolean hasNext() {
		return current != null;
	}

	@Override
	public synchronized String next() {
		String pivot = new String(current);
		current = walkTillNext();
		log.debug("Returning: " + pivot);
		return pivot;
	}

	@Override
	public void remove() {}

	/**
	 * Walk the filesystem recursively until it finds a candidate. Strategies: a) For any directory found during the walk, an iterator is
	 * built and concat to the main one; b) Any file is checked against admitted extensions
	 *
	 * @return the next element to be returned by next call of this.next()
	 */
	private synchronized String walkTillNext() {
		while (pathIterator.hasNext()) {
			Path nextFilePath = pathIterator.next();
			if (Files.isDirectory(nextFilePath)) {
				// concat
				try {
					pathIterator = Iterators.concat(pathIterator, Files.newDirectoryStream(nextFilePath).iterator());
					log.debug("Adding folder iterator: " + nextFilePath.toString());
				} catch (IOException e) {
					log.error("Cannot create folder iterator! Is this path correct? " + nextFilePath.toString());
					return null;
				}
			} else {
				if (extensions.isEmpty() || extensions.contains(FilenameUtils.getExtension(nextFilePath.toString()))) {
					if(incremental){
						try {
							final FileTime lastModifiedTime = Files.getLastModifiedTime(nextFilePath);
							if(lastModifiedTime.toInstant().isAfter(fromDate.atStartOfDay().toInstant(ZoneOffset.UTC))){
								log.debug("Returning: " + nextFilePath.toString());
								return nextFilePath.toString();
							}
							else {
								log.debug("File "+nextFilePath.toString()+" has not changed.");
							}
						} catch (Exception e) {
							throw new CollectorServiceRuntimeException(e);
						}
					}
					else {
						log.debug("Returning: " + nextFilePath.toString());
						return nextFilePath.toString();
					}
				}
			}
		}
		return null;
	}
}
