package eu.dnetlib.organizations.model.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Expression;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.data.jpa.domain.Specification;

import eu.dnetlib.organizations.model.OrganizationSearchEntry;

public class OrganizationSearchEntrySpecification implements Specification<OrganizationSearchEntry> {

	private static final long serialVersionUID = 1832753188603502182L;

	private final List<Pair<SearchableField, String>> containsConds = new ArrayList<>();
	private final List<Pair<SearchableField, String>> notContainsConds = new ArrayList<>();
	private final List<Pair<SearchableField, String>> equalsConds = new ArrayList<>();
	private final List<Pair<SearchableField, String>> notEqualsConds = new ArrayList<>();

	public void addTermConditions(final SearchableField field, final String[] terms) {
		if (terms == null) { return; }

		for (final String t : terms) {
			if (StringUtils.isNotBlank(t)) {
				final char op = t.charAt(0);
				final String val = t.substring(1).trim();

				switch (op) {
				case '~':
					containsConds.add(Pair.of(field, val));
					break;
				case '-':
					notContainsConds.add(Pair.of(field, val));
					break;
				case ':':
					equalsConds.add(Pair.of(field, val));
					break;
				case '!':
					notEqualsConds.add(Pair.of(field, val));
					break;
				default:
					throw new RuntimeException("Invalid search operator: " + op);
				}
			}
		}
	}

	@Override
	public Predicate toPredicate(final Root<OrganizationSearchEntry> root, final CriteriaQuery<?> query, final CriteriaBuilder criteriaBuilder) {
		final Predicate fieldsPredicate = criteriaBuilder.conjunction();
		final List<Expression<Boolean>> expressions = fieldsPredicate.getExpressions();

		containsConds.stream()
				.map(pair -> createContainsExpr(root, criteriaBuilder, pair.getKey(), pair.getValue()))
				.filter(Objects::nonNull)
				.forEach(expressions::add);
		notContainsConds.stream()
				.map(pair -> createContainsExpr(root, criteriaBuilder, pair.getKey(), pair.getValue()))
				.filter(Objects::nonNull)
				.map(criteriaBuilder::not) // See the negation of the expression
				.forEach(expressions::add);
		equalsConds.stream()
				.map(pair -> createEqualsExpr(root, criteriaBuilder, pair.getKey(), pair.getValue()))
				.filter(Objects::nonNull)
				.forEach(expressions::add);
		notEqualsConds.stream()
				.map(pair -> createEqualsExpr(root, criteriaBuilder, pair.getKey(), pair.getValue()))
				.filter(Objects::nonNull)
				.map(criteriaBuilder::not) // See the negation of the expression
				.forEach(expressions::add);

		return fieldsPredicate;
	}

	private Expression<Boolean> createContainsExpr(final Root<OrganizationSearchEntry> root,
			final CriteriaBuilder cb,
			final SearchableField field,
			final String value) {

		final String likeVal = "%" + value.toLowerCase().replaceAll(" ", "%") + "%";

		switch (field) {
		case all:
			return cb.isTrue(cb.function("fulltext_search", Boolean.class, root.get("fullTextVector"), cb.literal(value)));
		case name:
			return cb.like(root.get("nameNormalized"), likeVal);
		case city:
			return cb.like(root.get("cityNormalized"), likeVal);
		case othername:
			return cb.like(root.get("otherNamesNormalized"), likeVal);
		default:
			throw new RuntimeException("The contains/notContains predicates are not available for field: " + field);
		}

	}

	private Expression<Boolean> createEqualsExpr(final Root<OrganizationSearchEntry> root,
			final CriteriaBuilder cb,
			final SearchableField field,
			final String value) {
		switch (field) {
		case id:
			return cb.equal(root.get("id"), value);
		case openaireId:
			return cb.equal(root.get("openaireId"), value);
		case name:
			return cb.equal(cb.lower(root.get("nameNormalized")), value.toLowerCase());
		case othername:
			return cb.like(root.get("otherNamesNormalized"), "%§" + value.toLowerCase() + "§%");
		case acronym:
			return cb.like(root.get("acronymsNormalized"), "%§" + value.toLowerCase() + "§%");
		case type:
			return cb.equal(root.get("typeNormalized"), value.toLowerCase());
		case city:
			return cb.equal(root.get("cityNormalized"), value.toLowerCase());
		case country:
			return cb.equal(root.get("countryNormalized"), value.toLowerCase());
		case status:
			return cb.equal(root.get("statusNormalized"), value.toLowerCase());
		case pid:
			return cb.like(root.get("pidsNormalized"), "%§" + value.toLowerCase() + "§%");
		default:
			throw new RuntimeException("The 'equals' predicate is not available for field: " + field);
		}
	}

	public enum SearchableField {
		all, id, openaireId, name, othername, acronym, type, city, country, pid, status
	}

}
