BEGIN;

--DROP TABLE IF EXISTS sysconf;
--DROP TABLE IF EXISTS journal;
--DROP TABLE IF EXISTS org_index_search;

--DROP TABLE IF EXISTS notes;
--DROP TABLE IF EXISTS other_ids;
--DROP TABLE IF EXISTS other_names;
--DROP TABLE IF EXISTS acronyms;
--DROP TABLE IF EXISTS relationships;
--DROP TABLE IF EXISTS urls;
--DROP TABLE IF EXISTS oa_duplicates;
--DROP TABLE IF EXISTS oa_conflicts;
--DROP TABLE IF EXISTS persistent_orgs;

--DROP TABLE IF EXISTS organizations;
--DROP TABLE IF EXISTS org_types;

--DROP TABLE IF EXISTS user_countries;
--DROP TABLE IF EXISTS users;

--DROP TABLE IF EXISTS user_roles;
--DROP TABLE IF EXISTS countries;
--DROP TABLE IF EXISTS id_types;
--DROP TABLE IF EXISTS languages;

--DROP SEQUENCE IF EXISTS organizations_id_seq;

CREATE TABLE sysconf (
	id              text PRIMARY KEY DEFAULT 'default',
	title           text NOT NULL,
	homepage_msg    text NOT NULL DEFAULT '',
	homepage_msg_import text NOT NULL DEFAULT '<p>The portal is temporarily in read-only mode because we are updating the suggestions.<br /><br />We apologize for the inconvenience.</p>',
	readonly        boolean NOT NULL DEFAULT false,
	smtp_enabled    boolean NOT NULL DEFAULT false,
	smtp_host       text NOT NULL DEFAULT 'localhost',
	smtp_port       int NOT NULL DEFAULT 587,
	smtp_user                text NOT NULL DEFAULT '',
	smtp_password            text NOT NULL DEFAULT '',
	smtp_from_mail           text NOT NULL DEFAULT 'no-reply@openaire.eu',
	smtp_from_name           text NOT NULL DEFAULT 'OpenOrgs Database',
	smtp_to_mail_admin       text NOT NULL DEFAULT 'openorgs-admin@openaire.eu',
	smtp_new_user_message    text NOT NULL DEFAULT '-- message template here --',
	smtp_update_user_message text NOT NULL DEFAULT '-- message template here --'
);
INSERT INTO sysconf(id, title) VALUES ('default', 'OpenOrgs Database');

CREATE TABLE org_types (val text PRIMARY KEY, name text);
INSERT INTO org_types(val) VALUES ('Archive'), ('Company'), ('Education'), ('Facility'), ('Government'), ('Healthcare'), ('Nonprofit'), ('Other'), ('UNKNOWN');
UPDATE org_types SET name = val;

CREATE TABLE id_types (val text PRIMARY KEY, name text);
INSERT INTO id_types(val) VALUES ('CNRS'), ('FundRef'), ('HESA'), ('ISNI'), ('LinkedIn'), ('OrgRef'), ('UCAS'), ('UKPRN'), ('Wikidata'), ('GRID'), ('ROR'), ('OrgReg');
UPDATE id_types SET name = val;

CREATE TABLE languages (val text PRIMARY KEY, name text);
INSERT INTO languages(val) VALUES ('UNKNOWN'), 
			('aa'),('af'),('am'),('ar'),('as'),('az'),('ba'),('be'),('bg'),('bn'),('br'),('bs'),('ca'),('ch'),('co'),('cs'),('cy'),('da'),('de'),('dv'),
			('dz'),('el'),('en'),('eo'),('es'),('et'),('eu'),('fa'),('fi'),('fo'),('fr'),('fy'),('ga'),('gd'),('gl'),('gu'),('he'),('hi'),('hr'),('ht'),
			('hu'),('hy'),('id'),('is'),('it'),('iu'),('ja'),('jv'),('ka'),('kk'),('kl'),('km'),('kn'),('ko'),('ku'),('ky'),('la'),('lb'),('lo'),('lt'),
			('lv'),('mg'),('mi'),('mk'),('ml'),('mn'),('mr'),('ms'),('mt'),('my'),('nb'),('ne'),('nl'),('no'),('oc'),('om'),('or'),('pa'),('pl'),('ps'),
			('pt'),('rm'),('ro'),('ru'),('rw'),('sa'),('sd'),('si'),('sk'),('sl'),('sm'),('so'),('sq'),('sr'),('sv'),('sw'),('ta'),('te'),('tg'),('th'),
			('tk'),('tl'),('tr'),('tt'),('ug'),('uk'),('ur'),('uz'),('vi'),('xh'),('yo'),('zh'),('zu');
UPDATE languages SET name = val;

CREATE TABLE countries (val text PRIMARY KEY, name text);
INSERT INTO countries(val, name) VALUES
			('AD', 'Andorra'), 
			('AE', 'United Arab Emirates'), 
			('AF', 'Afghanistan'), 
			('AG', 'Antigua and Barbuda'), 
			('AI', 'Anguilla'), 
			('AL', 'Albania'), 
			('AM', 'Armenia'), 
			('AN', 'Netherlands Antilles'), 
			('AO', 'Angola'), 
			('AQ', 'Antarctica'), 
			('AR', 'Argentina'), 
			('AS', 'American Samoa'), 
			('AT', 'Austria'), 
			('AU', 'Australia'), 
			('AW', 'Aruba'), 
			('AX', 'Åland Islands'), 
			('AZ', 'Azerbaijan'), 
			('BA', 'Bosnia and Herzegovina'), 
			('BB', 'Barbados'), 
			('BD', 'Bangladesh'), 
			('BE', 'Belgium'), 
			('BF', 'Burkina Faso'), 
			('BG', 'Bulgaria'), 
			('BH', 'Bahrain'), 
			('BI', 'Burundi'), 
			('BJ', 'Benin'), 
			('BL', 'Saint-Barthélemy'), 
			('BM', 'Bermuda'), 
			('BN', 'Brunei Darussalam'), 
			('BO', 'Bolivia'), 
			('BQ', 'Bonaire, Sint Eustatius and Saba'), 
			('BR', 'Brazil'), 
			('BS', 'Bahamas'), 
			('BT', 'Bhutan'), 
			('BV', 'Bouvet Island'), 
			('BW', 'Botswana'), 
			('BY', 'Belarus'), 
			('BZ', 'Belize'), 
			('CA', 'Canada'), 
			('CC', 'Cocos (Keeling) Islands'), 
			('CD', 'Congo (Democratic Republic of)'), 
			('CF', 'Central African Republic'), 
			('CG', 'Congo'), 
			('CH', 'Switzerland'), 
			('CI', 'Cote d''Ivoire'), 
			('CK', 'Cook Islands'), 
			('CL', 'Chile'), 
			('CM', 'Cameroon'), 
			('CN', 'China (People''s Republic of)'), 
			('CO', 'Colombia'), 
			('CR', 'Costa Rica'), 
			('CS', 'Serbia and Montenegro'), 
			('CU', 'Cuba'), 
			('CV', 'Cape Verde'), 
			('CW', 'Curaçao'), 
			('CX', 'Christmas Island'), 
			('CY', 'Cyprus'), 
			('CZ', 'Czech Republic'), 
			('DE', 'Germany'), 
			('DJ', 'Djibouti'), 
			('DK', 'Denmark'), 
			('DM', 'Dominica'), 
			('DO', 'Dominican Republic'), 
			('DZ', 'Algeria'), 
			('EC', 'Ecuador'), 
			('EE', 'Estonia'), 
			('EG', 'Egypt'), 
			('EH', 'Western Sahara'), 
			('ER', 'Eritrea'), 
			('ES', 'Spain'), 
			('ET', 'Ethiopia'), 
			('EU', 'European Union'), 
			('FI', 'Finland'), 
			('FJ', 'Fiji'), 
			('FK', 'Falkland Islands (Malvinas)'), 
			('FM', 'Micronesia, Federated States of'), 
			('FO', 'Faroe Islands'), 
			('FR', 'France'), 
			('GA', 'Gabon'), 
			('GB', 'United Kingdom'), 
			('GD', 'Grenada'), 
			('GE', 'Georgia'), 
			('GF', 'French Guiana'), 
			('GG', 'Guernsey'), 
			('GH', 'Ghana'), 
			('GI', 'Gibraltar'), 
			('GL', 'Greenland'), 
			('GM', 'Gambia'), 
			('GN', 'Guinea'), 
			('GP', 'Guadeloupe'), 
			('GQ', 'Equatorial Guinea'), 
			('GR', 'Greece'), 
			('GS', 'South Georgia and the South Sandwich Islands'), 
			('GT', 'Guatemala'), 
			('GU', 'Guam'), 
			('GW', 'Guinea-Bissau'), 
			('GY', 'Guyana'), 
			('HK', 'Hong Kong'), 
			('HM', 'Heard Island and McDonald Islands'), 
			('HN', 'Honduras'), 
			('HR', 'Croatia'), 
			('HT', 'Haiti'), 
			('HU', 'Hungary'), 
			('ID', 'Indonesia'), 
			('IE', 'Ireland'), 
			('IL', 'Israel'), 
			('IM', 'Isle of Man'), 
			('IN', 'India'), 
			('IO', 'British Indian Ocean Territory'), 
			('IQ', 'Iraq'), 
			('IR', 'Iran (Islamic Republic of)'), 
			('IS', 'Iceland'), 
			('IT', 'Italy'), 
			('JE', 'Jersey'), 
			('JM', 'Jamaica'), 
			('JO', 'Jordan'), 
			('JP', 'Japan'), 
			('KE', 'Kenya'), 
			('KG', 'Kyrgyzstan'), 
			('KH', 'Cambodia'), 
			('KI', 'Kiribati'), 
			('KM', 'Comoros'), 
			('KN', 'Saint Kitts and Nevis'), 
			('KO', 'Kosovo * UN resolution'), 
			('KP', 'Korea, Democatric People''s Republic of'), 
			('KR', 'Korea (Republic of)'), 
			('KW', 'Kuwait'), 
			('KY', 'Cayman Islands'), 
			('KZ', 'Kazakhstan'), 
			('LA', 'Lao (People''s Democratic Republic)'), 
			('LB', 'Lebanon'), 
			('LC', 'Saint Lucia'), 
			('LI', 'Liechtenstein'), 
			('LK', 'Sri Lanka'), 
			('LR', 'Liberia'), 
			('LS', 'Lesotho'), 
			('LT', 'Lithuania'), 
			('LU', 'Luxembourg'), 
			('LV', 'Latvia'), 
			('LY', 'Libyan Arab Jamahiriya'), 
			('MA', 'Morocco'), 
			('MC', 'Monaco'), 
			('MD', 'Moldova (Republic of)'), 
			('ME', 'Montenegro'), 
			('MF', 'Saint Martin (French Part)'), 
			('MG', 'Madagascar'), 
			('MH', 'Marshall Islands'), 
			('MK', 'Former Yugoslav Republic of Macedonia'), 
			('ML', 'Mali'), 
			('MM', 'Myanmar'), 
			('MN', 'Mongolia'), 
			('MO', 'Macao'), 
			('MP', 'Northern Mariana Islands'), 
			('MQ', 'Martinique'), 
			('MR', 'Mauritania'), 
			('MS', 'Montserrat'), 
			('MT', 'Malta'), 
			('MU', 'Mauritius'), 
			('MV', 'Maldives'), 
			('MW', 'Malawi'), 
			('MX', 'Mexico'), 
			('MY', 'Malaysia'), 
			('MZ', 'Mozambique'), 
			('NA', 'Namibia'), 
			('NC', 'New Caledonia'), 
			('NE', 'Niger'), 
			('NF', 'Norfolk Island'), 
			('NG', 'Nigeria'), 
			('NI', 'Nicaragua'), 
			('NL', 'Netherlands'), 
			('NO', 'Norway'), 
			('NP', 'Nepal'), 
			('NR', 'Nauru'), 
			('NU', 'Niue'), 
			('NZ', 'New Zealand'), 
			('OC', 'Oceania'), 
			('OM', 'Oman'), 
			('PA', 'Panama'), 
			('PE', 'Peru'), 
			('PF', 'French Polynesia'), 
			('PG', 'Papua New Guinea'), 
			('PH', 'Philippines'), 
			('PK', 'Pakistan'), 
			('PL', 'Poland'), 
			('PM', 'Saint Pierre and Miquelon'), 
			('PN', 'Pitcairn'), 
			('PR', 'Puerto Rico'), 
			('PS', 'Palestinian-administered areas'), 
			('PT', 'Portugal'), 
			('PW', 'Palau'), 
			('PY', 'Paraguay'), 
			('QA', 'Qatar'), 
			('RE', 'Réunion'), 
			('RO', 'Romania'), 
			('RS', 'Serbia'), 
			('RU', 'Russian Federation'), 
			('RW', 'Rwanda'), 
			('SA', 'Saudi Arabia'), 
			('SB', 'Solomon Islands'), 
			('SC', 'Seychelles'), 
			('SD', 'Sudan'), 
			('SE', 'Sweden'), 
			('SG', 'Singapore'), 
			('SH', 'Saint Helena, Ascension and Tristan da Cunha'), 
			('SI', 'Slovenia'), 
			('SJ', 'Svalbard and Jan Mayen'), 
			('SK', 'Slovakia'), 
			('SL', 'Sierra Leone'), 
			('SM', 'San Marino'), 
			('SN', 'Senegal'), 
			('SO', 'Somalia'), 
			('SR', 'Suriname'), 
			('SS', 'South Sudan'), 
			('ST', 'São Tomé and Príncipe'), 
			('SV', 'El Salvador'), 
			('SX', 'Sint Maarten (Dutch Part)'), 
			('SY', 'Syrian Arab Republic'), 
			('SZ', 'Swaziland'), 
			('TC', 'Turks and Caicos Islands'), 
			('TD', 'Chad'), 
			('TF', 'French Southern Territories'), 
			('TG', 'Togo'), 
			('TH', 'Thailand'), 
			('TJ', 'Tajikistan'), 
			('TK', 'Tokelau'), 
			('TL', 'Timor-Leste'), 
			('TM', 'Turkmenistan'), 
			('TN', 'Tunisia'), 
			('TO', 'Tonga'), 
			('TR', 'Turkey'), 
			('TT', 'Trinidad and Tobago'), 
			('TV', 'Tuvalu'), 
			('TW', 'Taiwan'), 
			('TZ', 'Tanzania (United Republic of)'), 
			('UA', 'Ukraine'), 
			('UG', 'Uganda'), 
			('UM', 'United States Minor Outlying Islands'), 
			('UNKNOWN', 'UNKNOWN'), 
			('US', 'United States'), 
			('UY', 'Uruguay'), 
			('UZ', 'Uzbekistan'), 
			('VA', 'Holy See (Vatican City State)'), 
			('VC', 'Saint Vincent and the Grenadines'), 
			('VE', 'Venezuela'), 
			('VG', 'Virgin Islands (British)'), 
			('VI', 'Virgin Islands, U.S.'), 
			('VN', 'Viet Nam'), 
			('VU', 'Vanuatu'), 
			('WF', 'Wallis and Futuna'), 
			('WS', 'Samoa'), 
			('XK', 'Kosovo * UN resolution'), 
			('YE', 'Yemen'), 
			('YT', 'Mayotte'), 
			('YU', 'Yugoslavia'), 
			('ZA', 'South Africa'), 
			('ZM', 'Zambia'), 
			('ZW', 'Zimbabwe');


CREATE TABLE user_roles(role text PRIMARY KEY);
INSERT INTO user_roles VALUES ('ADMIN'), ('NATIONAL_ADMIN'), ('USER'), ('PENDING'), ('NOT_AUTHORIZED');

CREATE TABLE users (
	email            text PRIMARY KEY,
	fullname         text,
	organization     text,
	reference_person text,
	request_message  text, 
	valid            boolean DEFAULT true,
	role             text NOT NULL default 'USER' REFERENCES user_roles(role),
	first_access     timestamp with time zone DEFAULT now(),
	last_access      timestamp with time zone DEFAULT now()
);

CREATE TABLE user_countries (
	email     text REFERENCES users(email) ON UPDATE CASCADE ON DELETE CASCADE,
	country   text REFERENCES countries(val),
	PRIMARY KEY(email, country)	
);

CREATE SEQUENCE organizations_id_seq;

CREATE TABLE organizations (
    id                text PRIMARY KEY DEFAULT 'openorgs____::'||lpad(nextval('organizations_id_seq')::text,10,'0'),
    name              text,
   	type              text NOT NULL DEFAULT 'UNKNOWN' REFERENCES org_types(val),
    lat               double precision,
	lng               double precision,
	city              text,
	country           text REFERENCES countries(val),
	created_by        text,
	creation_date     timestamp with time zone DEFAULT now(),
	modified_by       text,
	modification_date timestamp with time zone DEFAULT now(),
	status            text NOT NULL DEFAULT 'suggested',
	ec_legalbody                             boolean,
	ec_legalperson                           boolean,
	ec_nonprofit                             boolean,
	ec_researchorganization                  boolean,
	ec_highereducation                       boolean,
	ec_internationalorganizationeurinterests boolean,
	ec_internationalorganization             boolean,
	ec_enterprise                            boolean,
	ec_smevalidated                          boolean,
	ec_nutscode                              boolean	
);
CREATE INDEX organizations_type_idx ON organizations(type);
CREATE INDEX organizations_country_idx ON organizations(country);

CREATE TABLE persistent_orgs (
	id text PRIMARY KEY REFERENCES organizations(id) ON UPDATE CASCADE ON DELETE CASCADE
);

CREATE TABLE other_ids (
	id      text REFERENCES organizations(id) ON UPDATE CASCADE ON DELETE CASCADE,
	otherid text,
	type    text REFERENCES id_types(val),
	PRIMARY KEY (id, otherid, type)
);
CREATE INDEX other_ids_id_idx ON other_ids(id);

CREATE TABLE other_names (
	id    text REFERENCES organizations(id) ON UPDATE CASCADE ON DELETE CASCADE,
	name  text,
	lang  text REFERENCES languages(val),
	PRIMARY KEY (id, name, lang)
);
CREATE INDEX other_names_id_idx ON other_names(id);

CREATE TABLE acronyms (
	id text  REFERENCES organizations(id) ON UPDATE CASCADE ON DELETE CASCADE,
	acronym text,
	PRIMARY KEY (id, acronym)
);
CREATE INDEX acronyms_id_idx ON acronyms(id);

CREATE TABLE relationships (
	id1     text REFERENCES organizations(id) ON UPDATE CASCADE ON DELETE CASCADE,
	reltype text,
	id2     text REFERENCES organizations(id) ON UPDATE CASCADE ON DELETE CASCADE,
	status  text NOT NULL DEFAULT 'suggested',
	creation_date     timestamp DEFAULT NOW(),
	created_by        text,
	modification_date timestamp,
	modified_by       text,
    PRIMARY KEY (id1, reltype, id2)
);
CREATE INDEX relationships_id1_idx ON relationships(id1);
CREATE INDEX relationships_id2_idx ON relationships(id2);

CREATE TABLE urls (
	id  text REFERENCES organizations(id) ON UPDATE CASCADE ON DELETE CASCADE,
	url text,
	PRIMARY KEY (id, url)
);
CREATE INDEX urls_id_idx ON urls(id);

CREATE TABLE notes (
	id                text PRIMARY KEY REFERENCES organizations(id) ON UPDATE CASCADE ON DELETE CASCADE,
	note              text,
	modified_by       text,
	modification_date timestamp
);

CREATE TABLE journal (
	jid         SERIAL PRIMARY KEY,
	id          text REFERENCES organizations(id) ON UPDATE CASCADE ON DELETE CASCADE,
	operation   text,
	description text,
	op_date     timestamp DEFAULT NOW(),
	email       text
);
CREATE INDEX journal_id_idx ON journal(id);

CREATE TABLE oa_duplicates (
	local_id          text REFERENCES organizations(id) ON UPDATE CASCADE ON DELETE CASCADE,
	oa_original_id    text REFERENCES organizations(id) ON UPDATE CASCADE ON DELETE CASCADE,
	oa_collectedfrom  text,
	status            text NOT NULL DEFAULT 'suggested',
	creation_date     timestamp DEFAULT NOW(),
	created_by        text,
	modification_date timestamp,
	modified_by       text,
	PRIMARY KEY (local_id, oa_original_id)
);
CREATE INDEX oa_duplicates_local_id_idx ON oa_duplicates(local_id);
CREATE INDEX oa_duplicates_oa_original_id_idx ON oa_duplicates(oa_original_id);


CREATE TABLE oa_conflicts (
	id1               text REFERENCES organizations(id) ON UPDATE CASCADE ON DELETE CASCADE,
	id2               text REFERENCES organizations(id) ON UPDATE CASCADE ON DELETE CASCADE,
	status            text NOT NULL DEFAULT 'suggested',
	idgroup           text,
	creation_date     timestamp DEFAULT NOW(),
	created_by        text,
	modification_date timestamp,
	modified_by       text,
	PRIMARY KEY (id1, id2)
);
CREATE INDEX oa_conflicts_id1_idx ON oa_conflicts(id1);
CREATE INDEX oa_conflicts_id2_idx ON oa_conflicts(id2);
CREATE INDEX oa_conflicts_idgroup_idx ON oa_conflicts(idgroup);

CREATE TABLE org_index_search(
	id                    text PRIMARY KEY,
	txt                   tsvector,
	openaire_id           text,
	name                  text, 
	type                  text, 
	city                  text, 
	country               text, 
	status                text, 
	acronyms              text[], 
	othernames            text[], 
	urls                  text[], 
	pids                  text[], 
	n_similar_dups        bigint, 
	n_suggested_dups      bigint, 
	n_different_dups      bigint,
	name_norm             text, 
	type_norm             text, 
	city_norm             text, 
	country_norm          text, 
	status_norm           text, 
	acronyms_norm         text, 
	othernames_norm       text, 
	urls_norm             text, 
	pids_norm             text
);
CREATE INDEX org_index_search_txt_gin_idx ON org_index_search USING gin(txt);
CREATE INDEX org_index_search_openaire_id_idx ON org_index_search(openaire_id);
CREATE INDEX org_index_search_name_norm_idx ON org_index_search(name_norm);
CREATE INDEX org_index_search_type_norm_idx ON org_index_search(type_norm); 
CREATE INDEX org_index_search_city_norm_idx ON org_index_search(city_norm);
CREATE INDEX org_index_search_country_norm_idx ON org_index_search(country_norm); 
CREATE INDEX org_index_search_status_norm_idx ON org_index_search(status_norm);

CREATE OR REPLACE FUNCTION fulltext_search(vector tsvector, s text) RETURNS boolean AS $$
        BEGIN
                RETURN vector @@ plainto_tsquery(s);
        END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION refresh_index_search() RETURNS bigint AS $$
		DELETE FROM org_index_search;
		WITH t as (
			INSERT INTO org_index_search(
				id,
				txt,
				openaire_id,
				name, 
				type, 
				city, 
				country, 
				status, 
				acronyms, 
				othernames, 
				urls, 
				pids, 
				n_similar_dups, 
				n_suggested_dups, 
				n_different_dups,
				name_norm, 
				type_norm, 
				city_norm, 
				country_norm, 
				status_norm, 
				acronyms_norm, 
				othernames_norm, 
				urls_norm, 
				pids_norm
			) SELECT
				o.id, 
				to_tsvector(o.id||' '||substr(o.id, 1, 14)||md5(substr(o.id, 15))||' '||o.name||' '||array_to_string(array_agg(DISTINCT n.name), ' ','')||' '||array_to_string(array_agg(DISTINCT i.otherid), ' ','')||' '||array_to_string(array_agg(DISTINCT a.acronym), ' ','')||' '||array_to_string(array_agg(DISTINCT u.url), ' ','')),
				substr(o.id, 1, 14)||md5(substr(o.id, 15)),
				o.name,
				o.type,
				o.city,
				o.country,
				o.status,
				array_remove(array_agg(DISTINCT a.acronym), NULL),
				array_remove(array_agg(DISTINCT n.name), NULL),
				array_remove(array_agg(DISTINCT u.url), NULL),
				array_remove(array_agg(DISTINCT i.otherid), NULL),
				count(DISTINCT d1.oa_original_id) FILTER (WHERE d1.status = 'is_similar'  ),
				count(DISTINCT d1.oa_original_id) FILTER (WHERE d1.status = 'suggested'   ),
				count(DISTINCT d1.oa_original_id) FILTER (WHERE d1.status = 'is_different'),
				lower(trim(o.name)), 
				lower(trim(o.type)), 
				lower(trim(o.city)), 
				lower(trim(o.country)), 
				lower(trim(o.status)),
				lower('§'||array_to_string(array_remove(array_agg(DISTINCT trim(a.acronym)), NULL), '§')||'§'),
				lower('§'||array_to_string(array_remove(array_agg(DISTINCT trim(n.name)), NULL), '§')||'§'),
				lower('§'||array_to_string(array_remove(array_agg(DISTINCT trim(u.url)), NULL), '§')||'§'),
				lower('§'||array_to_string(array_remove(array_agg(DISTINCT trim(i.otherid)), NULL), '§')||'§')
			FROM organizations o 
				LEFT OUTER JOIN other_names n on (o.id = n.id)
				LEFT OUTER JOIN other_ids i on (o.id = i.id)
				LEFT OUTER JOIN acronyms a on (o.id = a.id)
				LEFT OUTER JOIN urls u on (o.id = u.id)
				LEFT OUTER JOIN oa_duplicates d1 ON (o.id = d1.local_id)
			GROUP BY o.id, o.name, o.type, o.city, o. country, o.status RETURNING *
		) SELECT COUNT(*) FROM t;
$$ LANGUAGE SQL;

SELECT refresh_index_search();

CREATE OR REPLACE FUNCTION delete_index_search() RETURNS trigger LANGUAGE plpgsql AS $$
BEGIN
	DELETE FROM org_index_search WHERE id = old.id;
	RETURN OLD;
END;
$$;

CREATE OR REPLACE FUNCTION insert_or_update_index_search_trigger() RETURNS trigger LANGUAGE plpgsql AS $$
BEGIN
	INSERT INTO org_index_search(
		id,
		txt,
		openaire_id,
		name, 
		type, 
		city, 
		country, 
		status, 
		acronyms, 
		othernames, 
		urls, 
		pids, 
		n_similar_dups, 
		n_suggested_dups, 
		n_different_dups,
		name_norm, 
		type_norm, 
		city_norm, 
		country_norm, 
		status_norm, 
		acronyms_norm, 
		othernames_norm, 
		urls_norm, 
		pids_norm
	) (
		SELECT
			o.id, 
			to_tsvector(o.id||' '||substr(o.id, 1, 14)||md5(substr(o.id, 15))||' '||o.name||' '||array_to_string(array_agg(DISTINCT n.name), ' ','')||' '||array_to_string(array_agg(DISTINCT i.otherid), ' ','')||' '||array_to_string(array_agg(DISTINCT a.acronym), ' ','')||' '||array_to_string(array_agg(DISTINCT u.url), ' ','')),
			substr(o.id, 1, 14)||md5(substr(o.id, 15)),
			o.name,
			o.type,
			o.city,
			o.country,
			o.status,
			array_remove(array_agg(DISTINCT a.acronym), NULL),
			array_remove(array_agg(DISTINCT n.name), NULL),
			array_remove(array_agg(DISTINCT u.url), NULL),
			array_remove(array_agg(DISTINCT i.otherid), NULL),
			count(DISTINCT d1.oa_original_id) FILTER (WHERE d1.status = 'is_similar'  ),
			count(DISTINCT d1.oa_original_id) FILTER (WHERE d1.status = 'suggested'   ),
			count(DISTINCT d1.oa_original_id) FILTER (WHERE d1.status = 'is_different'),
			lower(trim(o.name)), 
			lower(trim(o.type)), 
			lower(trim(o.city)), 
			lower(trim(o.country)), 
			lower(trim(o.status)),
			lower('§'||array_to_string(array_remove(array_agg(DISTINCT trim(a.acronym)), NULL), '§')||'§'),
			lower('§'||array_to_string(array_remove(array_agg(DISTINCT trim(n.name)), NULL), '§')||'§'),
			lower('§'||array_to_string(array_remove(array_agg(DISTINCT trim(u.url)), NULL), '§')||'§'),
			lower('§'||array_to_string(array_remove(array_agg(DISTINCT trim(i.otherid)), NULL), '§')||'§')
		FROM organizations o 
		LEFT OUTER JOIN other_names n on (o.id = n.id)
		LEFT OUTER JOIN other_ids i on (o.id = i.id)
		LEFT OUTER JOIN acronyms a on (o.id = a.id)
		LEFT OUTER JOIN urls u on (o.id = u.id)
		LEFT OUTER JOIN oa_duplicates d1 ON (o.id = d1.local_id)
		WHERE o.id = new.id
		GROUP BY o.id, o.name, o.type, o.city, o. country, o.status
	) ON CONFLICT (id) DO UPDATE SET (
		txt,
		openaire_id,
		name, 
		type, 
		city, 
		country, 
		status, 
		acronyms, 
		othernames, 
		urls, 
		pids, 
		n_similar_dups, 
		n_suggested_dups, 
		n_different_dups,
		name_norm, 
		type_norm, 
		city_norm, 
		country_norm, 
		status_norm, 
		acronyms_norm, 
		othernames_norm, 
		urls_norm, 
		pids_norm
	) = (
		EXCLUDED.txt,
		EXCLUDED.openaire_id,
		EXCLUDED.name, 
		EXCLUDED.type, 
		EXCLUDED.city, 
		EXCLUDED.country, 
		EXCLUDED.status, 
		EXCLUDED.acronyms, 
		EXCLUDED.othernames, 
		EXCLUDED.urls, 
		EXCLUDED.pids, 
		EXCLUDED.n_similar_dups, 
		EXCLUDED.n_suggested_dups, 
		EXCLUDED.n_different_dups,
		EXCLUDED.name_norm, 
		EXCLUDED.type_norm, 
		EXCLUDED.city_norm, 
		EXCLUDED.country_norm, 
		EXCLUDED.status_norm, 
		EXCLUDED.acronyms_norm, 
		EXCLUDED.othernames_norm, 
		EXCLUDED.urls_norm, 
		EXCLUDED.pids_norm
	); 
	RETURN NEW;
END;
$$;

-- I create a trigger on the journal to manage all the insert/update operation (metadata, duplicates, ...) and because it should be executed at the end of all the necessary updates.
CREATE TRIGGER insert_or_update_index_search_trigger AFTER INSERT  ON journal FOR EACH ROW EXECUTE PROCEDURE insert_or_update_index_search_trigger();
CREATE TRIGGER delete_index_search_trigger           BEFORE DELETE ON organizations FOR EACH ROW EXECUTE PROCEDURE delete_index_search();

COMMIT;

