package eu.dnetlib.app.directindex.solr;

import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.client.solrj.response.UpdateResponse;
import org.apache.solr.client.solrj.util.ClientUtils;
import org.apache.solr.common.SolrDocumentList;
import org.apache.solr.common.SolrInputDocument;
import org.apache.solr.common.params.MapSolrParams;
import org.apache.solr.common.params.SolrParams;

import com.fasterxml.jackson.databind.ObjectMapper;

import eu.dnetlib.app.directindex.errors.DirectIndexApiException;
import eu.dnetlib.dhp.schema.solr.SolrRecord;

public class SolrIndexClient {

	private final SolrClient solrClient;

	public SolrIndexClient(final SolrClient solrClient) {
		this.solrClient = solrClient;
	}

	public void commit() throws DirectIndexApiException {
		try {
			solrClient.commit();
		} catch (final Throwable e) {
			throw new DirectIndexApiException("Error performing commit", e);
		}
	}

	public UpdateResponse addRecords(final Iterator<SolrInputDocument> iter) throws DirectIndexApiException {
		try {
			return solrClient.add(iter);
		} catch (final Throwable e) {
			throw new DirectIndexApiException("Error creating solr document", e);
		}
	}

	public UpdateResponse addRecords(final SolrInputDocument... records) throws Exception {
		return addRecords(Arrays.stream(records).iterator());
	}

	public UpdateResponse deleteRecord(final String id) throws DirectIndexApiException {
		try {
			// final String query = String.format("objidentifier:%s OR resultdupid:%s", ClientUtils.escapeQueryChars(id),
			// ClientUtils.escapeQueryChars(id));
			final String query = String.format("objidentifier:\"%s\" OR resultdupid:\"%s\"", id, id);

			return solrClient.deleteByQuery(query);
		} catch (SolrServerException | IOException e) {
			throw new DirectIndexApiException(e);
		}
	}

	public SolrRecord findRecord(final String id) throws DirectIndexApiException {
		try {
			final QueryResponse response = solrClient.query(queryParamsForId(id));
			final SolrDocumentList documents = response.getResults();

			if (documents.isEmpty()) { return null; }

			final String json = (String) documents.get(0).getFirstValue("__json");

			return new ObjectMapper().readValue(json, SolrRecord.class);
		} catch (SolrServerException | IOException e) {
			throw new DirectIndexApiException("Error executing solr query", e);
		}
	}

	public boolean existsRecord(final String id) throws DirectIndexApiException {
		try {
			final QueryResponse response = solrClient.query(queryParamsForId(id));
			final SolrDocumentList documents = response.getResults();
			return !documents.isEmpty();
		} catch (SolrServerException | IOException e) {
			throw new DirectIndexApiException("Error executing solr query", e);
		}
	}

	private SolrParams queryParamsForId(final String id) {

		final Map<String, String> params = new HashMap<String, String>();

		params.put("q", "objidentifier:" + ClientUtils.escapeQueryChars(id));
		params.put("fl", "__json");

		return new MapSolrParams(params);
	}

}
