package eu.dnetlib.data.mdstore.manager.utils;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avro.generic.GenericRecord;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;

import com.fasterxml.jackson.databind.ObjectMapper;

import eu.dnetlib.data.mdstore.manager.exceptions.MDStoreManagerException;
import parquet.avro.AvroParquetReader;
import parquet.hadoop.ParquetReader;

@Component
@Profile("!dev")
public class HdfsClientImpl implements HdfsClient {

	@Value("${dhp.mdstore-manager.hadoop.cluster}")
	private String hadoopCluster;

	@Value("${dhp.mdstore-manager.hadoop.user}")
	private String hadoopUser;

	@Value("${dhp.mdstore-manager.hdfs.base-path}")
	private String hdfsBasePath;

	private static final Log log = LogFactory.getLog(HdfsClientImpl.class);

	@Override
	public void deletePath(final String path) throws MDStoreManagerException {

		try (final FileSystem fs = FileSystem.get(conf())) {
			fs.delete(new Path(path), true);
			log.info("HDFS Path deleted: " + path);
		} catch (final FileNotFoundException e) {
			log.warn("Missing path: " + hdfsBasePath);
		} catch (IllegalArgumentException | IOException e) {
			log.error("Eror deleting path: " + path, e);
			throw new MDStoreManagerException("Eror deleting path: " + path, e);
		}
	}

	@Override
	public Set<String> listHadoopDirs() {
		final Set<String> res = new LinkedHashSet<>();

		try (final FileSystem fs = FileSystem.get(conf())) {
			for (final FileStatus mdDir : fs.listStatus(new Path(hdfsBasePath))) {
				if (isMdStoreOrVersionDir(mdDir)) {
					res.add(String.format("%s/%s", hdfsBasePath, mdDir.getPath().getName()));
					for (final FileStatus verDir : fs.listStatus(mdDir.getPath())) {
						if (isMdStoreOrVersionDir(verDir)) {
							res.add(String.format("%s/%s/%s", hdfsBasePath, mdDir.getPath().getName(), verDir.getPath().getName()));
						}
					}
				}
			}
		} catch (final FileNotFoundException e) {
			log.warn("Missing path: " + hdfsBasePath);
		} catch (final Exception e) {
			log.error("Error Listing path: " + hdfsBasePath, e);
		}

		return res;
	}

	@Override
	public Set<String> listContent(final String path) {

		final Set<String> res = new LinkedHashSet<>();
		try (final FileSystem fs = FileSystem.get(conf())) {
			for (final FileStatus f : fs.listStatus(new Path(path))) {
				if (HdfsClientImpl.isParquetFile(f)) {
					res.add(String.format("%s/%s", path, f.getPath().getName()));
				}
			}
		} catch (final FileNotFoundException e) {
			log.warn("Missing path: " + hdfsBasePath);
		} catch (final Exception e) {
			log.error("Error Listing path: " + path, e);
		}
		return res;
	}

	@Override
	@SuppressWarnings("unchecked")
	public List<Map<String, String>> readParquetFiles(final String path, final long limit) throws MDStoreManagerException {

		final List<Map<String, String>> list = new ArrayList<>();

		final Configuration conf = conf();

		long i = 0;

		final Set<String> fields = new LinkedHashSet<>();

		for (final String f : listContent(path)) {
			if (i < limit) {
				log.info("Opening parquet file: " + f);

				try (final ParquetReader<GenericRecord> reader = AvroParquetReader.<GenericRecord> builder(new Path(f)).withConf(conf).build()) {
					log.debug("File parquet OPENED");

					GenericRecord rec = null;
					while (i++ < limit && (rec = reader.read()) != null) {
						if (fields.isEmpty()) {
							rec.getSchema().getFields().forEach(field -> fields.add(field.name()));
							log.debug("Found schema: " + fields);
						}
						final Map<String, String> map = new LinkedHashMap<>();
						for (final String field : fields) {

							final Object v = rec.get(field);

							if ("validationResults".equals(field)) {
								final Map<String, String> reports = new HashMap<String, String>();

								if (v != null) {
									for (final Map.Entry<?, ?> e : ((Map<?, ?>) v).entrySet()) {
										if (e.getKey() != null && e.getValue() != null) {
											reports.put(e.getKey().toString(), e.getValue().toString());
										}
									}
								}

								map.put(field, new ObjectMapper().writeValueAsString(reports));
							} else {
								map.put(field, v != null ? v.toString() : "");
							}
						}
						list.add(map);
						log.debug("added record");
					}
				} catch (final FileNotFoundException e) {
					log.warn("Missing path: " + hdfsBasePath);
				} catch (final Throwable e) {
					log.error("Error reading parquet file: " + f, e);
					throw new MDStoreManagerException("Error reading parquet file: " + f, e);
				}
			}
		}
		return list;
	}

	public static boolean isParquetFile(final FileStatus fileStatus) {
		return fileStatus.isFile() && fileStatus.getPath().getName().endsWith(".parquet");
	}

	public static boolean isMdStoreOrVersionDir(final FileStatus fileStatus) {
		return fileStatus.isDirectory() && fileStatus.getPath().getName().startsWith("md-");
	}

	private Configuration conf() throws MDStoreManagerException {
		final Configuration conf = new Configuration();

		System.setProperty("HADOOP_USER_NAME", hadoopUser);

		if ("OCEAN".equalsIgnoreCase(hadoopCluster)) {
			conf.addResource(getClass().getResourceAsStream("/hadoop/OCEAN/core-site.xml"));
			conf.addResource(getClass().getResourceAsStream("/hadoop/OCEAN/ocean-hadoop-conf.xml"));
		} else if ("GARR".equalsIgnoreCase(hadoopCluster)) {
			conf.addResource(getClass().getResourceAsStream("/hadoop/GARR/core-site.xml"));
			conf.addResource(getClass().getResourceAsStream("/hadoop/GARR/garr-hadoop-conf.xml"));
		} else if ("MOCK".equalsIgnoreCase(hadoopCluster)) {
			// NOTHING
		} else {
			log.error("Invalid Haddop Cluster: " + hadoopCluster);
			throw new MDStoreManagerException("Invalid Haddop Cluster: " + hadoopCluster);
		}
		return conf;
	}

	@Override
	public String getHadoopCluster() {
		return hadoopCluster;
	}

	public void setHadoopCluster(final String hadoopCluster) {
		this.hadoopCluster = hadoopCluster;
	}

	@Override
	public String getHadoopUser() {
		return hadoopUser;
	}

	public void setHadoopUser(final String hadoopUser) {
		this.hadoopUser = hadoopUser;
	}

	@Override
	public String getHdfsBasePath() {
		return hdfsBasePath;
	}

	public void setHdfsBasePath(final String hdfsBasePath) {
		this.hdfsBasePath = hdfsBasePath;
	}
}
