package eu.dnetlib.broker.openaire;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.search.join.ScoreMode;
import org.elasticsearch.action.search.SearchType;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.NestedQueryBuilder;
import org.elasticsearch.index.query.Operator;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Profile;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.elasticsearch.core.ElasticsearchOperations;
import org.springframework.data.elasticsearch.core.SearchHit;
import org.springframework.data.elasticsearch.core.SearchHits;
import org.springframework.data.elasticsearch.core.mapping.IndexCoordinates;
import org.springframework.data.elasticsearch.core.query.NativeSearchQuery;
import org.springframework.data.elasticsearch.core.query.NativeSearchQueryBuilder;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import eu.dnetlib.broker.LiteratureBrokerServiceApplication;
import eu.dnetlib.broker.common.elasticsearch.Event;
import eu.dnetlib.broker.common.elasticsearch.Notification;
import eu.dnetlib.broker.common.elasticsearch.SuggestionEvent;
import eu.dnetlib.broker.common.elasticsearch.SuggestionNotification;
import eu.dnetlib.broker.common.elasticsearch.SuggestionNotificationRepository;
import eu.dnetlib.broker.common.properties.ElasticSearchProperties;
import eu.dnetlib.broker.common.subscriptions.MapCondition;
import eu.dnetlib.broker.common.subscriptions.Subscription;
import eu.dnetlib.broker.common.subscriptions.SubscriptionRepository;
import eu.dnetlib.broker.events.output.DispatcherManager;
import eu.dnetlib.broker.objects.OaBrokerEventPayload;
import eu.dnetlib.common.controller.AbstractDnetController;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;

@Profile("openaire")
@RestController
@RequestMapping("/api/openaireBroker")
@Tag(name = LiteratureBrokerServiceApplication.TAG_OPENAIRE)
public class OpenaireBrokerController extends AbstractDnetController {

	@Autowired
	private ElasticsearchOperations esOperations;

	@Autowired
	private SuggestionNotificationRepository suggestionNotificationRepository;

	@Autowired
	private SubscriptionRepository subscriptionRepo;

	@Autowired
	private ElasticSearchProperties props;

	@Autowired
	private DispatcherManager dispatcher;

	@Autowired
	private JdbcTemplate jdbcTemplate;

	private static final Log log = LogFactory.getLog(OpenaireBrokerController.class);

	@Operation(summary = "Return the datasources having events")
	@GetMapping("/datasources")
	@Deprecated
	public List<BrowseEntry> findDatasourcesWithEvents() {
		return findDatasourcesWithEvents(0, 100);
	}

	@Operation(summary = "Return a page of datasources having events")
	@GetMapping("/datasources/{page}/{size}")
	public List<BrowseEntry> findDatasourcesWithEvents(@PathVariable final int page, @PathVariable final int size) {
		try {
			final String sql = IOUtils.toString(getClass().getResourceAsStream("/sql/datasourceTopics.sql"), "UTF-8");
			final RowMapper<BrowseEntry> mapper = (rs, rowNum) -> new BrowseEntry(rs.getString("name"), rs.getLong("size"));
			return jdbcTemplate.query(sql, mapper, size, page * size);
		} catch (final Exception e) {
			log.error("Error executing query", e);
			return new ArrayList<>();
		}
	}

	@Operation(summary = "Return the topics of the events of a datasource")
	@GetMapping("/topicsForDatasource")
	public List<BrowseEntry> findTopicsForDatasource(@RequestParam final String ds) {
		try {
			final String sql = IOUtils.toString(getClass().getResourceAsStream("/sql/datasourceTopicsDetailed.sql"), "UTF-8");
			final RowMapper<BrowseEntry> mapper = (rs, rowNum) -> new BrowseEntry(rs.getString("topic"), rs.getLong("size"));
			return jdbcTemplate.query(sql, mapper, ds);
		} catch (final Exception e) {
			log.error("Error executing query", e);
			return new ArrayList<>();
		}
	}

	@Operation(summary = "Return a page of events of a datasource (by topic)")
	@GetMapping("/events/{nPage}/{size}")
	public EventsPage showEvents(@RequestParam final String ds, @RequestParam final String topic, @PathVariable final int nPage, @PathVariable final int size) {

		final NestedQueryBuilder nestedQuery = QueryBuilders.nestedQuery("map", QueryBuilders.matchQuery("map.targetDatasourceName", ds), ScoreMode.None);

		final QueryBuilder q = StringUtils.isNotBlank(topic) && !"*".equals(topic) ? QueryBuilders.boolQuery()
				.must(QueryBuilders.matchQuery("topic", topic).operator(Operator.AND))
				.must(nestedQuery) : nestedQuery;

		final NativeSearchQuery searchQuery = new NativeSearchQueryBuilder()
				.withQuery(q)
				.withSearchType(SearchType.DEFAULT)
				.withFields("payload")
				.withPageable(PageRequest.of(nPage, size))
				.build();

		final SearchHits<SuggestionEvent> page =
				esOperations.search(searchQuery, SuggestionEvent.class, IndexCoordinates.of(props.getSuggestionEventsIndexName()));

		final List<OaBrokerEventPayload> list = page.stream()
				.map(SearchHit::getContent)
				.map(Event::getPayload)
				.map(OaBrokerEventPayload::fromJSON)
				.collect(Collectors.toList());

		return new EventsPage(ds, topic, nPage, overrideGetTotalPage(page, size), page.getTotalHits(), list);
	}

	@Operation(summary = "Return a page of events of a datasource (by query)")
	@PostMapping("/events/{nPage}/{size}")
	public EventsPage advancedShowEvents(@PathVariable final int nPage, @PathVariable final int size, @RequestBody final AdvQueryObject qObj) {

		final BoolQueryBuilder mapQuery = QueryBuilders.boolQuery();
		ElasticSearchQueryUtils.addMapCondition(mapQuery, "map.targetDatasourceName", qObj.getDatasource());
		ElasticSearchQueryUtils.addMapCondition(mapQuery, "map.targetResultTitle", qObj.getTitles());
		ElasticSearchQueryUtils.addMapCondition(mapQuery, "map.targetAuthors", qObj.getAuthors());
		ElasticSearchQueryUtils.addMapCondition(mapQuery, "map.targetSubjects", qObj.getSubjects());
		ElasticSearchQueryUtils.addMapConditionForTrust(mapQuery, "map.trust", qObj.getTrust());
		ElasticSearchQueryUtils.addMapConditionForDates(mapQuery, "map.targetDateofacceptance", qObj.getDates());

		final NestedQueryBuilder nestedQuery = QueryBuilders.nestedQuery("map", mapQuery, ScoreMode.None);

		final QueryBuilder q = StringUtils.isNotBlank(qObj.getTopic()) && !"*".equals(qObj.getTopic()) ? QueryBuilders.boolQuery()
				.must(QueryBuilders.matchQuery("topic", qObj.getTopic()).operator(Operator.AND))
				.must(nestedQuery) : nestedQuery;

		final NativeSearchQuery searchQuery = new NativeSearchQueryBuilder()
				.withQuery(q)
				.withSearchType(SearchType.DEFAULT)
				.withFields("payload")
				.withPageable(PageRequest.of(nPage, size))
				.build();

		final SearchHits<SuggestionEvent> page =
				esOperations.search(searchQuery, SuggestionEvent.class, IndexCoordinates.of(props.getSuggestionEventsIndexName()));

		final List<OaBrokerEventPayload> list = page.stream()
				.map(SearchHit::getContent)
				.map(Event::getPayload)
				.map(OaBrokerEventPayload::fromJSON)
				.collect(Collectors.toList());

		return new EventsPage(qObj.getDatasource(), qObj.getTopic(), nPage, overrideGetTotalPage(page, size), page.getTotalHits(), list);
	}

	@Operation(summary = "Perform a subscription")
	@PostMapping("/subscribe")
	public Subscription registerSubscription(@RequestBody final OpenaireSubscription oSub) {
		final Subscription sub = oSub.asSubscription();

		subscriptionRepo.save(sub);

		return sub;
	}

	@Operation(summary = "Return the subscriptions of an user (by email and datasource (optional))")
	@GetMapping("/subscriptions")
	public Map<String, List<SimpleSubscriptionDesc>> subscriptions(@RequestParam final String email, @RequestParam(required = false) final String ds) {
		final Iterable<Subscription> iter = subscriptionRepo.findBySubscriber(email);
		return StreamSupport.stream(iter.spliterator(), false)
				.filter(s -> !s.getTopic().startsWith("ALERT/"))
				.map(this::subscriptionDesc)
				.filter(s -> StringUtils.isBlank(ds) || StringUtils.equalsIgnoreCase(s.getDatasource(), ds))
				.collect(Collectors.groupingBy(SimpleSubscriptionDesc::getDatasource));
	}

	@Operation(summary = "Return a page of notifications")
	@GetMapping("/notifications/{subscrId}/{nPage}/{size}")
	public EventsPage notifications(@PathVariable final String subscrId, @PathVariable final int nPage, @PathVariable final int size) {

		final Optional<Subscription> optSub = subscriptionRepo.findById(subscrId);

		if (optSub.isPresent()) {
			final Subscription sub = optSub.get();

			final NativeSearchQuery searchQuery = new NativeSearchQueryBuilder()
					.withQuery(QueryBuilders.termQuery("subscriptionId.keyword", subscrId))
					.withSearchType(SearchType.DEFAULT)
					.withFields("payload")
					.withPageable(PageRequest.of(nPage, size))
					.build();

			final SearchHits<SuggestionNotification> page =
					esOperations.search(searchQuery, SuggestionNotification.class, IndexCoordinates.of(props.getSuggestionNotificationsIndexName()));

			final List<OaBrokerEventPayload> list = page.stream()
					.map(SearchHit::getContent)
					.map(Notification::getPayload)
					.map(OaBrokerEventPayload::fromJSON)
					.collect(Collectors.toList());

			return new EventsPage(extractDatasource(sub), sub.getTopic(), nPage, overrideGetTotalPage(page, size), page.getTotalHits(), list);
		}
		log.warn("Invalid subscription: " + subscrId);
		return new EventsPage("", "", nPage, 0, 0, new ArrayList<>());

	}

	@Operation(summary = "Send notifications")
	@GetMapping("/notifications/send/{date}")
	private List<String> sendMailForNotifications(@PathVariable final long date) {
		new Thread(() -> innerSendMailForNotifications(date)).start();
		return Arrays.asList("Sending ...");
	}

	@Operation(summary = "Update stats")
	@GetMapping("/stats/update")
	private List<String> updateStats() {
		new Thread(() -> {
			try {
				jdbcTemplate.update(IOUtils.toString(getClass().getResourceAsStream("/sql/updateStats.sql"), "UTF-8"));
			} catch (final Exception e) {
				log.error("Error updating stats", e);
			}
		}).start();
		return Arrays.asList("Sending ...");
	}

	private void innerSendMailForNotifications(final long date) {
		for (final Subscription s : subscriptionRepo.findAll()) {
			if (s.getTopic().startsWith("ALERT/")) {
				continue; // Skip alert subscriptions
			}

			final long count = suggestionNotificationRepository.countBySubscriptionIdAndDateAfter(s.getSubscriptionId(), date);
			if (count > 0) {
				final Map<String, Object> params = new HashMap<>();
				params.put("oa_notifications_total", count);
				params.put("oa_datasource", extractDatasource(s));
				dispatcher.sendNotification(s, params);
			}

			s.setLastNotificationDate(new Date());
			subscriptionRepo.save(s);
		}
	}

	private SimpleSubscriptionDesc subscriptionDesc(final Subscription s) {
		return new SimpleSubscriptionDesc(s.getSubscriptionId(), extractDatasource(s), s.getTopic(), s.getCreationDate(), s.getLastNotificationDate(),
				OpenaireBrokerController.this.suggestionNotificationRepository.countBySubscriptionId(s.getSubscriptionId()));
	}

	private String extractDatasource(final Subscription sub) {
		return sub.getConditionsAsList()
				.stream()
				.filter(c -> "targetDatasourceName".equals(c.getField()))
				.map(MapCondition::getListParams)
				.filter(l -> !l.isEmpty())
				.map(l -> l.get(0).getValue())
				.findFirst()
				.orElse("");
	}

	private long overrideGetTotalPage(final SearchHits<?> page, final int size) {
		return (page.getTotalHits() + size - 1) / size;
	}

}
