package eu.dnetlib.data.transform.xml;

import java.util.List;
import java.util.Map;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.protobuf.Descriptors.Descriptor;
import eu.dnetlib.data.mapreduce.util.OafRowKeyDecoder;
import eu.dnetlib.data.proto.FieldTypeProtos.KeyValue;
import eu.dnetlib.data.proto.FieldTypeProtos.StructuredProperty;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafEntity;
import eu.dnetlib.data.proto.PersonProtos.Person;
import eu.dnetlib.data.proto.ResultProtos.Result;
import eu.dnetlib.data.proto.ResultProtos.Result.Context;
import eu.dnetlib.data.proto.ResultProtos.Result.ExternalReference;
import eu.dnetlib.data.proto.ResultProtos.Result.Instance;
import eu.dnetlib.data.proto.ResultProtos.Result.Journal;
import eu.dnetlib.data.proto.TypeProtos.Type;
import org.apache.commons.lang.StringUtils;
import org.w3c.dom.NodeList;

public class OafToHbaseXsltFunctions extends CommonDNetXsltFunctions {

	public static String oafResult(
			final String resultId,
			final String provenance,
			final String trust,
			final NodeList about,
			final String hostedbyId,
			final String hostedbyName,
			final String collectedFromId,
			final String collectedFromName,
			final String originalId,
			final String dateOfCollection,
			final String dateOfTransformation,
			final NodeList nodelist) {
		try {
			final String entityId = OafRowKeyDecoder.decode(resultId).getKey();
			final ValueMap values = ValueMap.parseNodeList(nodelist);
			final Descriptor mDesc = Result.Metadata.getDescriptor();

			final Result.Metadata.Builder metadata = buildMetadata(values, mDesc);
			final Result.Builder result = buildResult(metadata, values, mDesc, hostedbyId, hostedbyName);
			final OafEntity.Builder entity = buildOafEntity(result, entityId, nodelist, getKV(collectedFromId, collectedFromName), originalId);
			entity.setDateofcollection(dateOfCollection)
					.setDateoftransformation(dateOfTransformation).setOaiprovenance(getOAIProvenance(about));

			final Oaf oaf = getOaf(entity, getDataInfo(about, provenance, trust, false, false));
			return base64(oaf.toByteArray());
		} catch (final Throwable e) {
			handleException(e, resultId, hostedbyId, hostedbyName, provenance, trust, collectedFromId, collectedFromName, originalId, dateOfCollection);
		}
		return null;
	}

	public static String oafResultUpdate(final String resultId,
			final String provenance,
			final String trust,
			final NodeList nodelist,
			final String hostedbyId,
			final String hostedbyName) {
		try {
			final String entityId = OafRowKeyDecoder.decode(resultId).getKey();
			final ValueMap values = ValueMap.parseNodeList(nodelist);
			final Descriptor mDesc = Result.Metadata.getDescriptor();

			final Result.Metadata.Builder metadata = buildMetadata(values, mDesc);
			final Result.Builder result = buildResult(metadata, values, mDesc, hostedbyId, hostedbyName);

			final OafEntity.Builder entity = buildOafEntity(result, entityId, nodelist, null, null);
			final Oaf oaf = getOaf(entity, null);
			return base64(oaf.toByteArray());
		} catch (final Throwable e) {
			handleException(e, resultId, hostedbyId, hostedbyName, provenance, trust, null, null, null, null);
		}
		return null;
	}

	private static OafEntity.Builder buildOafEntity(final Result.Builder result,
			final String entityId,
			final NodeList nodelist,
			KeyValue collectedFrom,
			String originalId) {

		final List<StructuredProperty> pids = Lists.newArrayList();
		pids.addAll(parsePids(nodelist));

		final OafEntity.Builder entity =
				getEntity(Type.result, entityId, collectedFrom, StringUtils.isBlank(originalId) ? null : Lists.newArrayList(originalId), null, null, pids)
						.setResult(result);
		return entity;
	}

	private static Result.Metadata.Builder buildMetadata(final ValueMap values, final Descriptor mDesc) {
		final Result.Metadata.Builder metadata = Result.Metadata.newBuilder();

		addStructuredProps(metadata, mDesc.findFieldByName("subject"), values.get("subject"), "keyword", "dnet:subject_classification_typologies");
		addStructuredProps(metadata, mDesc.findFieldByName("title"), values.get("title"), "main title", "dnet:dataCite_title");
		for (final String fieldname : Lists.newArrayList("description", "source", "contributor")) {
			if (values.get(fieldname) != null) {
				for (final String s : values.get(fieldname).listValues()) {
					addField(metadata, mDesc.findFieldByName(fieldname), s);
				}
			}
		}
		addField(metadata, mDesc.findFieldByName("language"), setQualifier(getDefaultQualifier("dnet:languages"), values.get("language").listValues()));
		addField(metadata, mDesc.findFieldByName("dateofacceptance"), values.get("dateaccepted").listValues());
		addField(metadata, mDesc.findFieldByName("publisher"), values.get("publisher").listValues());
		addField(metadata, mDesc.findFieldByName("embargoenddate"), values.get("embargoenddate").listValues());
		addField(metadata, mDesc.findFieldByName("storagedate"), values.get("storagedate").listValues());

		addField(metadata, mDesc.findFieldByName("resulttype"), getSimpleQualifier("publication", "dnet:result_typologies"));

		addField(metadata, mDesc.findFieldByName("fulltext"), values.get("fulltext").listValues());
		addField(metadata, mDesc.findFieldByName("format"), values.get("format").listValues());
		if (values.get("concept") != null) {
			for (final Element e : values.get("concept")) {
				final String id = e.getAttributes().get("id");
				if (StringUtils.isBlank(id)) throw new IllegalArgumentException("Context id cannot be blank");
				metadata.addContext(Context.newBuilder().setId(id));
			}
		}
		if (values.get("journal") != null) {
			for (final Element e : values.get("journal")) {

				final Journal.Builder journal = Journal.newBuilder();
				if (e.getText() != null) {
					journal.setName(e.getText());
				}

				final Map<String, String> attr = e.getAttributes();
				if (attr != null) {
					if (attr.get("issn") != null) {
						journal.setIssnPrinted(attr.get("issn"));
					}
					if (attr.get("eissn") != null) {
						journal.setIssnOnline(attr.get("eissn"));
					}
					if (attr.get("lissn") != null) {
						journal.setIssnLinking(attr.get("lissn"));
					}
				}
				metadata.setJournal(journal.build());
			}
		}
		return metadata;
	}

	private static Result.Builder buildResult(final Result.Metadata.Builder metadata,
			final ValueMap values,
			final Descriptor mDesc,
			final String hostedbyId,
			final String hostedbyName) {
		final Result.Builder result = Result.newBuilder();
		if (values.get("creator") != null) {
			for (final String fullname : Iterables.limit(values.get("creator").listValues(), 10)) {

				final Person.Metadata.Builder authorMetadata = Person.Metadata.newBuilder();

				authorMetadata.setFullname(sf(fullname));

				final eu.dnetlib.pace.model.Person p = new eu.dnetlib.pace.model.Person(fullname, false);
				if (p.isAccurate()) {
					authorMetadata.setFirstname(sf(p.getNormalisedFirstName()));
					authorMetadata.clearSecondnames().addSecondnames(sf(p.getNormalisedSurname()));
					authorMetadata.setFullname(sf(p.getNormalisedFullname()));
				}

				result.addAuthor(Person.newBuilder().setMetadata(authorMetadata));
			}
		}

		final Instance.Builder instance = Instance.newBuilder().setHostedby(getKV(hostedbyId, hostedbyName));

		addField(instance, Instance.getDescriptor().findFieldByName("licence"),
				setQualifier(getDefaultQualifier("dnet:access_modes"), values.get("accessrights").listValues()));
		addField(instance, Instance.getDescriptor().findFieldByName("instancetype"),
				setQualifier(getDefaultQualifier("dnet:publication_resource"), values.get("cobjcategory").listValues()));

		if (values.get("identifier") != null) {
			addField(instance, Instance.getDescriptor().findFieldByName("url"),
					Lists.newArrayList(Iterables.filter(values.get("identifier").listValues(), urlFilter)));
		}

		result.addInstance(instance);

		final List<Element> extrefs = values.get("reference");
		if (!extrefs.isEmpty()) {
			final Descriptor extDesc = ExternalReference.getDescriptor();
			for (final Element element : extrefs) {
				final ExternalReference.Builder extref = ExternalReference.newBuilder();
				addField(extref, extDesc.findFieldByName("url"), element.getText());
				addField(extref, extDesc.findFieldByName("sitename"), element.getAttributes().get("source"));
				addField(extref, extDesc.findFieldByName("refidentifier"), element.getAttributes().get("identifier"));
				addField(extref, extDesc.findFieldByName("label"), element.getAttributes().get("title"));
				addField(extref, extDesc.findFieldByName("query"), element.getAttributes().get("query"));
				addField(extref, extDesc.findFieldByName("qualifier"),
						setQualifier(getDefaultQualifier("dnet:externalReference_typologies"), Lists.newArrayList(element.getAttributes().get("type")))
								.build());

				result.addExternalReference(extref);
			}
		}

		return result.setMetadata(metadata);
	}

	private static void handleException(Throwable e, final String resultId, final String hostedbyId, final String hostedbyName,
			final String provenance, final String trust, final String collectedFromId, final String collectedFromName,
			final String originalId, final String dateOfCollection) {
		System.err.println("resultId: " + resultId);
		if (StringUtils.isNotBlank(hostedbyId)) System.err.println("hostedbyId: " + hostedbyId);
		if (StringUtils.isNotBlank(hostedbyName)) System.err.println("hostedbyName: " + hostedbyName);
		if (StringUtils.isNotBlank(provenance)) System.err.println("provenance: " + provenance);
		if (StringUtils.isNotBlank(trust)) System.err.println("trust: " + trust);
		if (StringUtils.isNotBlank(collectedFromId)) System.err.println("collectedFromId: " + collectedFromId);
		if (StringUtils.isNotBlank(collectedFromName)) System.err.println("collectedFromName: " + collectedFromName);
		if (StringUtils.isNotBlank(originalId)) System.err.println("originalId: " + originalId);
		if (StringUtils.isNotBlank(dateOfCollection)) System.err.println("dateOfCollection: " + dateOfCollection);
		e.printStackTrace();
		throw new RuntimeException(e);
	}
}
