package eu.dnetlib.data.transform.xml;

import java.util.List;

import com.google.common.collect.Lists;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.Message.Builder;
import eu.dnetlib.data.mapreduce.util.OafRowKeyDecoder;
import eu.dnetlib.data.proto.DatasourceOrganizationProtos.DatasourceOrganization;
import eu.dnetlib.data.proto.DatasourceOrganizationProtos.DatasourceOrganization.Provision;
import eu.dnetlib.data.proto.DatasourceProtos.Datasource;
import eu.dnetlib.data.proto.FieldTypeProtos.DataInfo;
import eu.dnetlib.data.proto.FieldTypeProtos.KeyValue;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafEntity;
import eu.dnetlib.data.proto.OafProtos.OafRel;
import eu.dnetlib.data.proto.OrganizationProtos.Organization;
import eu.dnetlib.data.proto.PersonProtos.Person;
import eu.dnetlib.data.proto.PersonResultProtos.PersonResult;
import eu.dnetlib.data.proto.PersonResultProtos.PersonResult.Authorship;
import eu.dnetlib.data.proto.ProjectOrganizationProtos.ProjectOrganization;
import eu.dnetlib.data.proto.ProjectOrganizationProtos.ProjectOrganization.Participation;
import eu.dnetlib.data.proto.ProjectPersonProtos.ProjectPerson;
import eu.dnetlib.data.proto.ProjectPersonProtos.ProjectPerson.ContactPerson;
import eu.dnetlib.data.proto.ProjectProtos.Project;
import eu.dnetlib.data.proto.RelMetadataProtos.RelMetadata;
import eu.dnetlib.data.proto.RelTypeProtos.RelType;
import eu.dnetlib.data.proto.RelTypeProtos.SubRelType;
import eu.dnetlib.data.proto.ResultProjectProtos.ResultProject;
import eu.dnetlib.data.proto.ResultProjectProtos.ResultProject.Outcome;
import eu.dnetlib.data.proto.ResultProtos.Result;
import eu.dnetlib.data.proto.ResultProtos.Result.Instance;
import eu.dnetlib.data.proto.TypeProtos.Type;
import org.apache.commons.lang.StringUtils;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class DbmfToHbaseXsltFunctions extends CommonDNetXsltFunctions {

	public static String oafEntity(final String type,
			final String id,
			final String collectedFromId,
			final String collectedFromName,
			final NodeList identities,
			final String dateOfCollection,
			final String dateOfTransformation,
			final NodeList nodeList) {

		final String entityId = OafRowKeyDecoder.decode(id).getKey();
		List<String> ids = Lists.newArrayList();
		for(int i = 0; i < identities.getLength(); i++){
			Node n = identities.item(i);
			String s = n.getTextContent();
			ids.add(s);
		}
		switch (Type.valueOf(type)) {
		case datasource:
			return serializeOafEntity(nodeList, Type.datasource, entityId, getKV(collectedFromId, collectedFromName), ids, dateOfCollection,
					dateOfTransformation, Datasource.newBuilder());
		case organization:
			return serializeOafEntity(nodeList, Type.organization, entityId, getKV(collectedFromId, collectedFromName), ids, dateOfCollection,
					dateOfTransformation, Organization.newBuilder());
		case person:
			return serializeOafEntity(nodeList, Type.person, entityId, getKV(collectedFromId, collectedFromName), ids, dateOfCollection,
					dateOfTransformation, Person.newBuilder());
		case project:
			return serializeOafEntity(nodeList, Type.project, entityId, getKV(collectedFromId, collectedFromName), ids, dateOfCollection,
					dateOfTransformation, Project.newBuilder());
		case result:
			return serializeOafEntity(nodeList, Type.result, entityId, getKV(collectedFromId, collectedFromName), ids, dateOfCollection,
					dateOfTransformation ,Result.newBuilder());
		default:
			throw new IllegalArgumentException("Invalid entity type: " + type);
		}
	}

	public static String oafRel(final String relationType,
			final String source,
			final String target,
			final NodeList nodeList,
			final String relClass,
			final String relScheme) {
		return oafRel(relationType, source, target, nodeList, relClass, relScheme, null, null);
	}

	public static String oafRel(final String relationType,
			final String source,
			final String target,
			final NodeList nodeList,
			final String relClass,
			final String relScheme,
			final String collectedFromId,
			final String collectedFromName) {

		final String eSource = OafRowKeyDecoder.decode(source).getKey();
		final String eTarget = OafRowKeyDecoder.decode(target).getKey();
		final RelType relType = RelType.valueOf(relationType);

		switch (relType) {
		case datasourceOrganization:
			Provision.Builder provision = Provision.newBuilder().setRelMetadata(
					RelMetadata.newBuilder().setSemantics(getSimpleQualifier(Provision.RelName.valueOf(relClass).toString(), relScheme)));
			DatasourceOrganization.Builder dorg = DatasourceOrganization.newBuilder().setProvision(provision);

			return serializeOafRel(nodeList, eSource, eTarget, relType, SubRelType.provision, relClass, collectedFromId, collectedFromName, false, dorg, provision);
		case personResult:
			Authorship.Builder auth = Authorship.newBuilder().setRelMetadata(
					RelMetadata.newBuilder().setSemantics(getSimpleQualifier(Authorship.RelName.valueOf(relClass).toString(), relScheme)));
			PersonResult.Builder personResult = PersonResult.newBuilder().setAuthorship(auth);

			return serializeOafRel(nodeList, eSource, eTarget, relType, SubRelType.authorship, relClass, collectedFromId, collectedFromName, false, personResult, auth);
		case projectPerson:
			ContactPerson.Builder contact = ContactPerson.newBuilder().setRelMetadata(
					RelMetadata.newBuilder().setSemantics(getSimpleQualifier(ContactPerson.RelName.valueOf(relClass).toString(), relScheme)));
			ProjectPerson.Builder projectPerson = ProjectPerson.newBuilder().setContactPerson(contact);

			return serializeOafRel(nodeList, eSource, eTarget, relType, SubRelType.contactPerson, relClass, collectedFromId, collectedFromName, false, projectPerson, contact);
		case projectOrganization:
			Participation.Builder participant = Participation.newBuilder().setRelMetadata(
					RelMetadata.newBuilder().setSemantics(getSimpleQualifier(Participation.RelName.valueOf(relClass).toString(), relScheme)));
			ProjectOrganization.Builder projectOrganization = ProjectOrganization.newBuilder().setParticipation(participant);

			return serializeOafRel(nodeList, eSource, eTarget, relType, SubRelType.participation, relClass, collectedFromId, collectedFromName, false, projectOrganization, participant);
		case resultProject:
			Outcome.Builder outcome = Outcome.newBuilder().setRelMetadata(
					RelMetadata.newBuilder().setSemantics(getSimpleQualifier(Outcome.RelName.valueOf(relClass).toString(), relScheme)));
			ResultProject.Builder resultProject = ResultProject.newBuilder().setOutcome(outcome);

			return serializeOafRel(nodeList, eSource, eTarget, relType, SubRelType.outcome, relClass, collectedFromId, collectedFromName, false, resultProject, outcome);
		default:
			throw new IllegalArgumentException("unhandled relType: " + relationType);
		}
	}

	// ////////////////////////////////////////////////////////

	protected static String serializeOafEntity(final NodeList nodelist,
			final Type type,
			final String id,
			final KeyValue collectedFrom,
			final List<String> identities,
			final String dateOfCollection,
			final String dateOfTransformation,
			final Builder entity) {
		try {
			final FieldDescriptor md = entity.getDescriptorForType().findFieldByName("metadata");

			final OafEntity.Builder parent = getEntity(type, id, collectedFrom, identities, dateOfCollection, dateOfTransformation, null);
			final Builder metadata = entity.newBuilderForField(md);
			final DataInfo.Builder dataInfo = DataInfo.newBuilder();

			if (type.equals(Type.result)) {
				final Instance.Builder instance = Instance.newBuilder();
				parseNodelist(nodelist, instance);
				FieldDescriptor instanceDescriptor = Result.getDescriptor().findFieldByName(Instance.getDescriptor().getName());
				if (instanceDescriptor != null) {
					entity.setField(instanceDescriptor, instance);
				}
			}
			parseNodelist(nodelist, parent, entity, metadata, dataInfo);

			final FieldDescriptor entityDescriptor = OafEntity.getDescriptor().findFieldByName(type.toString());

			final Oaf build = getOaf(parent.setField(entityDescriptor, entity.setField(md, metadata.build()).build()), dataInfo);

			return base64(build.toByteArray());
		} catch (Exception e) {
			e.printStackTrace(System.err);
			throw new RuntimeException(e);
		}
	}

	protected static String serializeOafRel(final NodeList nodeList,
			final String sourceId,
			final String targetId,
			final RelType relType,
			final SubRelType subRelType,
			final String relClass,
			final String collectedFromId,
			final String collectedFromName,
			final boolean isChild,
			final Builder rel,
			final Builder subRel) {
		try {

			final DataInfo.Builder dataInfo = DataInfo.newBuilder();

			parseNodelist(nodeList, rel, subRel, dataInfo);

			OafRel.Builder builder = getRel(sourceId, targetId, relType, subRelType, relClass, collectedFromId, collectedFromName, isChild);

			FieldDescriptor subRelDescriptor = rel.getDescriptorForType().findFieldByName(subRelType.toString());
			rel.setField(subRelDescriptor, subRel.build());

			FieldDescriptor relDescriptor = OafRel.getDescriptor().findFieldByName(relType.toString());
			builder.setField(relDescriptor, rel.build());

			Oaf build = getOaf(builder, dataInfo);
			return base64(build.toByteArray());
		} catch (Exception e) {
			e.printStackTrace(System.err);
			throw new RuntimeException(e);
		}
	}

	private static void parseNodelist(final NodeList nodeList, final Builder... builders) {
		
		for (int i = 0; i < nodeList.getLength(); i++) {

			final Node fieldNode = nodeList.item(i);
			final Node attr = fieldNode.getAttributes().getNamedItem("name");

			final String fieldName = attr.getNodeValue();
			final NodeList children = fieldNode.getChildNodes();

			for (int j = 0; j < children.getLength(); j++) {

				final Node child = children.item(j);
				final String childName = child.getLocalName();
				if ("ITEM".equals(childName) || StringUtils.isBlank(childName)) {
					for (Builder builder : builders) {
						FieldDescriptor desc = builder.getDescriptorForType().findFieldByName(fieldName);
						if (desc != null) {
							String text = getText((StringUtils.isBlank(childName)) ? fieldNode : child);
							if (!StringUtils.isBlank(text)) {
								addField(builder, desc, text);
							}
						}
					}
				}
			}
		}
	}

	private static String getText(final Node node) {
		StringBuffer result = new StringBuffer();
		if (!node.hasChildNodes()) { return ""; }

		NodeList list = node.getChildNodes();
		for (int i = 0; i < list.getLength(); i++) {
			Node subnode = list.item(i);
			if (subnode.getNodeType() == Node.TEXT_NODE) {
				result.append(subnode.getNodeValue());
			} else if (subnode.getNodeType() == Node.CDATA_SECTION_NODE) {
				result.append(subnode.getNodeValue());
			} else if (subnode.getNodeType() == Node.ENTITY_REFERENCE_NODE) {
				result.append(getText(subnode));
			}
		}
		return result.toString().trim();
	}

}
