package eu.dnetlib.data.transform;

import java.util.List;

import org.apache.commons.lang.StringUtils;

import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.protobuf.Descriptors.EnumValueDescriptor;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.GeneratedMessage;

/**
 * AbstractProtoMapper provide common navigation methods on the protocolbuffers Messages.
 *
 * @author claudio
 */
public abstract class AbstractProtoMapper {

	/** The Constant PATH_SEPARATOR. */
	private static final String PATH_SEPARATOR = "/";

	/**
	 * Process multi path.
	 *
	 * @param proto
	 *            the proto
	 * @param paths
	 *            the paths
	 * @return the list
	 */
	protected List<Object> processMultiPath(final GeneratedMessage proto, final List<String> paths) {
		final List<Object> response = Lists.newArrayList();
		for (final String pathElements : paths) {
			response.addAll(processPath(proto, pathElements));
		}
		return response;
	}

	/**
	 * Process path.
	 *
	 * @param proto
	 *            the proto
	 * @param path
	 *            the path
	 * @return the list
	 */
	protected List<Object> processPath(final GeneratedMessage proto, final String path) {
		return processPath(proto, Lists.newLinkedList(Splitter.on(PATH_SEPARATOR).trimResults().split(path)));
	}

	/**
	 * Process path.
	 *
	 * @param proto
	 *            the proto
	 * @param pathElements
	 *            the list
	 * @return the list
	 */
	protected List<Object> processPath(final GeneratedMessage proto, final List<String> pathElements) {

		final List<Object> response = Lists.newArrayList();

		if (pathElements.isEmpty()) throw new RuntimeException("ProtoBuf navigation path is empty");

		final FieldDescriptor fd = proto.getDescriptorForType().findFieldByName(pathElements.get(0));
		if (fd != null) {
			if (fd.isRepeated()) {
				final int count = proto.getRepeatedFieldCount(fd);
				for (int i = 0; i < count; i++) {
					final Object field = proto.getRepeatedField(fd, i);
					response.addAll(generateFields(fd, field, pathElements));
				}
			} else {
				final Object field = proto.getField(fd);
				response.addAll(generateFields(fd, field, pathElements));
			}
		} else throw new RuntimeException("Invalid protobuf path (field not found): " + StringUtils.join(pathElements, ">"));

		return response;
	}

	/**
	 * Generate fields.
	 *
	 * @param fd
	 *            the fd
	 * @param field
	 *            the field
	 * @param list
	 *            the list
	 * @return the list
	 */
	private List<Object> generateFields(final FieldDescriptor fd, final Object field, final List<String> list) {
		if (field instanceof GeneratedMessage) {
			if (list.size() > 1) return processPath((GeneratedMessage) field, list.subList(1, list.size()));
			else throw new RuntimeException("No primitive type found");
		} else {
			if (list.size() == 1) {
				final List<Object> res = Lists.newArrayList();
				switch (fd.getType()) {
				case ENUM:
					res.add(((EnumValueDescriptor) field).getName());
					break;
				default:
					res.add(field);
					break;
				}
				return res;
			}
			else throw new RuntimeException("Found a primitive type before the path end");
		}
	}
}
