package eu.dnetlib.data.download.worker;

import java.io.IOException;
import java.io.StringWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import eu.dnetlib.data.download.DownloadReport;
import eu.dnetlib.data.download.DownloadReportMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import eu.dnetlib.data.download.rmi.DownloadItem;
import eu.dnetlib.data.download.rmi.DownloadServiceImpl;
import eu.dnetlib.data.objectstore.modular.ObjectStoreRecord;
import eu.dnetlib.data.objectstore.modular.connector.ObjectStore;
import eu.dnetlib.data.objectstore.rmi.ObjectStoreFile;
import eu.dnetlib.data.objectstore.rmi.ObjectStoreServiceException;
import eu.dnetlib.data.objectstore.rmi.Protocols;

/**
 * The Class DownloadWorker is a worker responsible to download the data into the object store.
 */
public class DownloadWorker implements Callable<DownloadReportMap> {

    /**
     * The Constant log.
     */
    private static final Log log = LogFactory.getLog(DownloadWorker.class);

    /**
     * The queue.
     */
    private BlockingQueue<String> queue = null;

    /**
     * The object store.
     */
    private ObjectStore objectStore = null;

    /**
     * The protocol.
     */
    private Protocols protocol;

    /**
     * The mime type.
     */
    private String mimeType;

    private Function<String, DownloadItem> converter;

    /**
     * Instantiates a new download worker.
     *
     * @param queue       the queue
     * @param objectStore the object store
     * @param protocol    the protocol
     * @param mimeType    the mime type
     */
    public DownloadWorker(final BlockingQueue<String> queue, final ObjectStore objectStore, final Protocols protocol, final String mimeType, final Function<String, DownloadItem> converter) {
        this.setConverter(converter);
        this.setQueue(queue);
        this.setObjectStore(objectStore);
        this.setMimeType(mimeType);
        this.setProtocol(protocol);
    }

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Runnable#run()
     */
    @Override
    public DownloadReportMap call() throws Exception {
        DownloadReportMap report = new DownloadReportMap();
        try {
            DownloadItem di = getConverter().apply(queue.take());
            while (di != DownloadServiceImpl.END_QUEUE) {
                if ((di.getUrl() != null) && (di.getUrl().length() != 0) && !checkIfExists(di)) {
                    try {
                        URL toDownload = followURL(new URL(di.getUrl()), report, di);
                        if (toDownload == null) {
                            continue;
                        }
                        ObjectStoreRecord record = new ObjectStoreRecord();
                        ObjectStoreFile metadata = new ObjectStoreFile();
                        metadata.setObjectID(di.getFileName());
                        metadata.setMetadataRelatedID(di.getIdItemMetadata());
                        metadata.setAccessProtocol(this.protocol);
                        metadata.setMimeType(this.mimeType);
                        metadata.setDownloadedURL(di.getOriginalUrl());
                        record.setFileMetadata(metadata);
                        record.setInputStream(toDownload.openStream());
                        objectStore.feedObjectRecord(record);
                        report.addDowload();
                        log.debug("Saved object " + metadata.toJSON());
                    } catch (Exception e) {
                        //log.error(e);
                        reportException(report, di, e);
                    }
                }
                di = getConverter().apply(queue.take());
            }

            queue.put(DownloadServiceImpl.END_QUEUE_STRING);

        } catch (Exception e) {
            log.error(e);
            report.setStatus(false);
            return report;
        }

        log.info("CLOSED THREAD "+Thread.currentThread().getId());
        report.setStatus(true);
        return report;

    }

    private void reportException(DownloadReportMap report, DownloadItem di, Exception e) {
        String className = e.getClass().getName();
        if (!report.containsKey(className)) {
            DownloadReport dr = new DownloadReport();
            dr.setStackTrace(Joiner.on("\tat ").join(e.getStackTrace()));
            dr.setRecordId(di.getIdItemMetadata());
            report.put(className, dr);
        } else {
           report.get(className).incrementError();
        }
    }


    private URL followURL(URL inputURL, DownloadReportMap report, DownloadItem di) {

        URL startURL = inputURL;
        HttpURLConnection conn = null;
        try {
            conn = (HttpURLConnection) startURL.openConnection();
            conn.setInstanceFollowRedirects(true);  // you still need to handle redirect manully.
            HttpURLConnection.setFollowRedirects(true);
            String location = inputURL.toString();
            if ((conn.getResponseCode() >= 300) && (conn.getResponseCode() < 400)) {
                location = conn.getHeaderFields().get("Location").get(0);
                conn.disconnect();
            }

            if (location != inputURL.toString())
                return new URL(location);
            return inputURL;

        } catch (IOException e) {
            reportException(report,di,e);
            return null;
        }
    }

    private boolean checkIfExists(final DownloadItem di) {
        try {
            return objectStore.deliverObject(di.getFileName()) != null;
        } catch (ObjectStoreServiceException e) {
            return false;
        }
    }

    /**
     * Gets the object store.
     *
     * @return the objectStore
     */
    public ObjectStore getObjectStore() {
        return objectStore;
    }

    /**
     * Sets the object store.
     *
     * @param objectStore the objectStore to set
     */
    public void setObjectStore(final ObjectStore objectStore) {
        this.objectStore = objectStore;
    }

    /**
     * Gets the queue.
     *
     * @return the queue
     */
    public BlockingQueue<String> getQueue() {
        return queue;
    }

    /**
     * Sets the queue.
     *
     * @param queue the queue to set
     */
    public void setQueue(final BlockingQueue<String> queue) {
        this.queue = queue;
    }

    /**
     * Gets the protocol.
     *
     * @return the protocol
     */
    public Protocols getProtocol() {
        return protocol;
    }

    /**
     * Sets the protocol.
     *
     * @param protocol the protocol to set
     */
    public void setProtocol(final Protocols protocol) {
        this.protocol = protocol;
    }

    /**
     * Gets the mime type.
     *
     * @return the mimeType
     */
    public String getMimeType() {
        return mimeType;
    }

    /**
     * Sets the mime type.
     *
     * @param mimeType the mimeType to set
     */
    public void setMimeType(final String mimeType) {
        this.mimeType = mimeType;
    }

    public Function<String, DownloadItem> getConverter() {
        return converter;
    }

    public void setConverter(Function<String, DownloadItem> converter) {
        this.converter = converter;
    }
}
