package eu.dnetlib.data.download.rmi;

import java.util.List;
import java.util.concurrent.*;

import javax.annotation.Resource;

import eu.dnetlib.data.download.DownloadReport;
import eu.dnetlib.data.download.DownloadReportMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.google.common.base.Function;
import com.google.common.collect.Lists;

import eu.dnetlib.data.download.worker.DownloadWorker;
import eu.dnetlib.data.objectstore.modular.connector.ObjectStore;
import eu.dnetlib.data.objectstore.modular.connector.ObjectStoreDao;
import eu.dnetlib.data.objectstore.rmi.Protocols;
import eu.dnetlib.enabling.resultset.client.ResultSetClientFactory;

/**
 * The Class DownloadServiceFeeder.
 */
public class DownloadServiceFeeder {

	/** The Constant log. */
	private static final Log log = LogFactory.getLog(DownloadServiceFeeder.class);

	/** The download plugin enumerator. */
	@Resource
	DownloadPluginEnumeratorImpl downloadPluginEnumerator;

	/** The result set client factory. */
	@Resource
	private ResultSetClientFactory resultSetClientFactory;

	/** The object store dao. */
	@Autowired
	private ObjectStoreDao objectStoreDao;

	/**
	 * Download and feed file into the objectStore .
	 *
	 * @param epr
	 *            the end-point reference of the result-set of Serialized DownloadItem
	 * @param plugin
	 *            the plugin used to retrieve the correct URL
	 * @param objectStoreID
	 *            the object store id to store the data
	 * @param protocol
	 *            the protocol used to download the file
	 * @param mimeType
	 *            the mime type of the Files
	 * @param numberOfThreads
	 *            the number of threads to use for download at the same time
	 * @throws DownloadServiceException
	 *             the download service exception
	 */
	public DownloadReportMap download(final String epr,
			final String plugin,
			final String objectStoreID,
			final String protocol,
			final String mimeType,
			final int numberOfThreads) throws DownloadServiceException {
		final DownloadPlugin downloadPlugin = downloadPluginEnumerator.get(plugin);

		Iterable<String> urlInfo = resultSetClientFactory.getClient(epr);
		BlockingQueue<String> itemsQueue = new ArrayBlockingQueue<String>(1024);
		ExecutorService executor = Executors.newFixedThreadPool(numberOfThreads);
		ObjectStore objStore = objectStoreDao.getObjectStore(objectStoreID);

		List<Future<DownloadReportMap>> responses = Lists.newArrayList();

		for (int i = 0; i < numberOfThreads; i++) {
			responses.add(executor.submit(new DownloadWorker(itemsQueue, objStore, Protocols.valueOf(protocol), mimeType, new Function<String, DownloadItem>() {
                @Override
                public DownloadItem apply(final String input) {
                    if (input.equals(DownloadServiceImpl.END_QUEUE_STRING)) return DownloadServiceImpl.END_QUEUE;
                    DownloadItem di = DownloadItem.newObjectfromJSON(input);
                    if (downloadPlugin.retrieveUrl(di) == null) {
                        di.setUrl(null);
                        di.setOriginalUrl(null);
                    }
                    return di;
                }
            })));
		}
		for (String di : urlInfo) {
			try {
				itemsQueue.put(di);
			} catch (InterruptedException e) {
				log.error(e);
			}
		}

		try {
			itemsQueue.put(DownloadServiceImpl.END_QUEUE_STRING);
		} catch (InterruptedException e) {
			log.error(e);
		}

        DownloadReportMap resultMap = new DownloadReportMap();

        for(Future<DownloadReportMap> currentResponses : responses){
            try {
                DownloadReportMap currentMap = currentResponses.get();
                for (String key : currentMap.keySet()){
                    if (! resultMap.containsKey(key)){
                        resultMap.put(key, currentMap.get(key));
                        resultMap.setTotalDownloaded(currentMap.getTotalDownloaded());
                        resultMap.setStatus(currentMap.getStatus());
                    } else {
                        DownloadReport currentReport = currentMap.get(key);
                        resultMap.get(key).incrementError(currentReport.getNumberOfOccurrences());
                    }
                }
                resultMap.setStatus(resultMap.getStatus() && currentMap.getStatus());
                resultMap.setTotalDownloaded(currentMap.getTotalDownloaded()+resultMap.getTotalDownloaded());

            } catch (Exception e) {
                log.error(e);
                resultMap.setStatus(false);
            }
        }

        return resultMap;

	}
}
