package eu.dnetlib.data.download.worker;

import java.net.URL;
import java.util.concurrent.BlockingQueue;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import eu.dnetlib.data.download.rmi.DownloadItem;
import eu.dnetlib.data.download.rmi.DownloadServiceImpl;
import eu.dnetlib.data.objectstore.modular.ObjectStoreRecord;
import eu.dnetlib.data.objectstore.modular.connector.ObjectStore;
import eu.dnetlib.data.objectstore.rmi.ObjectStoreFile;
import eu.dnetlib.data.objectstore.rmi.Protocols;

/**
 * The Class DownloadWorker is a worker responsible to download the data into the object store.
 */
public class DownloadWorker implements Runnable {

	/** The Constant log. */
	private static final Log log = LogFactory.getLog(DownloadWorker.class);

	/** The queue. */
	private BlockingQueue<DownloadItem> queue = null;

	/** The object store. */
	private ObjectStore objectStore = null;

	/** The protocol. */
	private Protocols protocol;

	/** The mime type. */
	private String mimeType;

	/**
	 * Instantiates a new download worker.
	 * 
	 * @param queue
	 *            the queue
	 * @param objectStore
	 *            the object store
	 * @param protocol
	 *            the protocol
	 * @param mimeType
	 *            the mime type
	 */
	public DownloadWorker(final BlockingQueue<DownloadItem> queue, final ObjectStore objectStore, final Protocols protocol, final String mimeType) {
		this.setQueue(queue);
		this.setObjectStore(objectStore);
		this.setMimeType(mimeType);
		this.setProtocol(protocol);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public void run() {
		try {
			DownloadItem di = queue.take();
			while (di != DownloadServiceImpl.END_QUEUE) {
				if ((di.getUrl() != null) && (di.getUrl().length() != 0) && (objectStore.deliverObject(di.getFileName()) == null)) {
					// log.debug(di.getUrl());
					try {
						URL toDownload = new URL(di.getUrl());
						ObjectStoreRecord record = new ObjectStoreRecord();
						ObjectStoreFile metadata = new ObjectStoreFile();
						metadata.setObjectID(di.getFileName());
						metadata.setMetadataRelatedID(di.getIdItemMetadata());
						metadata.setAccessProtocol(this.protocol);
						metadata.setMimeType(this.mimeType);
						metadata.setDownloadedURL(di.getOriginalUrl());
						record.setFileMetadata(metadata);
						record.setInputStream(toDownload.openStream());
						objectStore.feedObjectRecord(record);
						log.debug("Saved object " + metadata.toJSON());
					} catch (Exception e) {
						log.error(e);
					}
				}

				di = queue.take();

			}
			queue.put(DownloadServiceImpl.END_QUEUE);

		} catch (Exception e) {
			log.error(e);
		}

		log.info("CLOSED THREAD");

	}

	/**
	 * Gets the object store.
	 * 
	 * @return the objectStore
	 */
	public ObjectStore getObjectStore() {
		return objectStore;
	}

	/**
	 * Sets the object store.
	 * 
	 * @param objectStore
	 *            the objectStore to set
	 */
	public void setObjectStore(final ObjectStore objectStore) {
		this.objectStore = objectStore;
	}

	/**
	 * Gets the queue.
	 * 
	 * @return the queue
	 */
	public BlockingQueue<DownloadItem> getQueue() {
		return queue;
	}

	/**
	 * Sets the queue.
	 * 
	 * @param queue
	 *            the queue to set
	 */
	public void setQueue(final BlockingQueue<DownloadItem> queue) {
		this.queue = queue;
	}

	/**
	 * Gets the protocol.
	 * 
	 * @return the protocol
	 */
	public Protocols getProtocol() {
		return protocol;
	}

	/**
	 * Sets the protocol.
	 * 
	 * @param protocol
	 *            the protocol to set
	 */
	public void setProtocol(final Protocols protocol) {
		this.protocol = protocol;
	}

	/**
	 * Gets the mime type.
	 * 
	 * @return the mimeType
	 */
	public String getMimeType() {
		return mimeType;
	}

	/**
	 * Sets the mime type.
	 * 
	 * @param mimeType
	 *            the mimeType to set
	 */
	public void setMimeType(final String mimeType) {
		this.mimeType = mimeType;
	}

}
