package eu.dnetlib.data.download.rmi;

import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;

import eu.dnetlib.data.download.worker.DownloadWorker;
import eu.dnetlib.data.objectstore.modular.connector.ObjectStore;
import eu.dnetlib.data.objectstore.modular.connector.ObjectStoreDao;
import eu.dnetlib.data.objectstore.rmi.Protocols;
import eu.dnetlib.enabling.resultset.client.ResultSetClientFactory;

/**
 * The Class DownloadServiceFeeder.
 */
public class DownloadServiceFeeder {

	/** The Constant log. */
	private static final Log log = LogFactory.getLog(DownloadServiceFeeder.class);

	/** The download plugin enumerator. */
	@Resource
	DownloadPluginEnumeratorImpl downloadPluginEnumerator;

	/** The result set client factory. */
	@Resource
	private ResultSetClientFactory resultSetClientFactory;

	/** The object store dao. */
	@Autowired
	private ObjectStoreDao objectStoreDao;

	/**
	 * Download and feed file into the objectStore .
	 *
	 * @param epr
	 *            the end-point reference of the result-set of Serialized DownloadItem
	 * @param plugin
	 *            the plugin used to retrieve the correct URL
	 * @param objectStoreID
	 *            the object store id to store the data
	 * @param protocol
	 *            the protocol used to download the file
	 * @param mimeType
	 *            the mime type of the Files
	 * @param numberOfThreads
	 *            the number of threads to use for download at the same time
	 * @throws DownloadServiceException
	 *             the download service exception
	 */
	public void Download(final String epr,
			final String plugin,
			final String objectStoreID,
			final String protocol,
			final String mimeType,
			final int numberOfThreads) throws DownloadServiceException {
		final DownloadPlugin downloadPlugin = downloadPluginEnumerator.get(plugin);

		Iterable<String> urlInfo = resultSetClientFactory.getClient(epr);
		BlockingQueue<DownloadItem> itmesQueue = new ArrayBlockingQueue<DownloadItem>(1024);

		Iterable<DownloadItem> convertedUrl = Iterables.transform(urlInfo, new Function<String, DownloadItem>() {

			@Override
			public DownloadItem apply(final String input) {
				DownloadItem di = DownloadItem.newObjectfromJSON(input);
				if (downloadPlugin.retrieveUrl(di) == null) {
					di.setUrl(null);
					di.setOriginalUrl(null);
				}
				return di;
			}
		});

		DownloadWorker workers[] = new DownloadWorker[numberOfThreads];

		ObjectStore objStore = objectStoreDao.getObjectStore(objectStoreID);

		for (int i = 0; i < numberOfThreads; i++) {
			workers[i] = new DownloadWorker(itmesQueue, objStore, Protocols.valueOf(protocol), mimeType);
			new Thread(workers[i]).start();
		}
		for (DownloadItem di : convertedUrl) {
			try {
				itmesQueue.put(di);
			} catch (InterruptedException e) {
				log.error(e);
			}
		}

		try {
			itmesQueue.put(DownloadServiceImpl.END_QUEUE);
		} catch (InterruptedException e) {
			log.error(e);
		}
	}

}
