/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.file;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.Closeable;
import java.io.File;
import java.io.FileDescriptor;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.zip.CRC32;
import org.apache.jackrabbit.oak.segment.file.FileStoreMonitor;
import org.apache.jackrabbit.oak.segment.file.TarEntry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class TarWriter
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(TarWriter.class);
    static final int INDEX_MAGIC = 170937098;
    static final int GRAPH_MAGIC = 170936074;
    static final int BINARY_REFERENCES_MAGIC = 170934794;
    static final int BLOCK_SIZE = 512;
    private static final byte[] ZERO_BYTES = new byte[512];
    private final int writeIndex;
    private final File file;
    private final FileStoreMonitor monitor;
    private RandomAccessFile access = null;
    private FileChannel channel = null;
    private boolean closed = false;
    private final Map<UUID, TarEntry> index = Maps.newLinkedHashMap();
    private final Map<Integer, Map<UUID, Set<String>>> binaryReferences = Maps.newHashMap();
    private final Map<UUID, Set<UUID>> graph = Maps.newHashMap();

    static final int getPaddingSize(int size) {
        int remainder = size % 512;
        if (remainder > 0) {
            return 512 - remainder;
        }
        return 0;
    }

    TarWriter(File file) {
        this.file = file;
        this.monitor = FileStoreMonitor.DEFAULT;
        this.writeIndex = -1;
    }

    TarWriter(File directory, FileStoreMonitor monitor, int writeIndex) {
        this.file = new File(directory, String.format("data%05d%s.tar", writeIndex, "a"));
        this.monitor = monitor;
        this.writeIndex = writeIndex;
    }

    synchronized int count() {
        return this.index.size();
    }

    synchronized Set<UUID> getUUIDs() {
        return Sets.newHashSet(this.index.keySet());
    }

    synchronized boolean containsEntry(long msb, long lsb) {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        return this.index.containsKey(new UUID(msb, lsb));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    ByteBuffer readEntry(long msb, long lsb) throws IOException {
        TarEntry entry;
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        TarWriter tarWriter = this;
        synchronized (tarWriter) {
            entry = this.index.get(new UUID(msb, lsb));
        }
        if (entry != null) {
            Preconditions.checkState((this.channel != null ? 1 : 0) != 0);
            ByteBuffer data = ByteBuffer.allocate(entry.size());
            this.channel.read(data, entry.offset());
            data.rewind();
            return data;
        }
        return null;
    }

    long writeEntry(long msb, long lsb, byte[] data, int offset, int size, int generation) throws IOException {
        Preconditions.checkNotNull((Object)data);
        Preconditions.checkPositionIndexes((int)offset, (int)(offset + size), (int)data.length);
        UUID uuid = new UUID(msb, lsb);
        CRC32 checksum = new CRC32();
        checksum.update(data, offset, size);
        String entryName = String.format("%s.%08x", uuid, checksum.getValue());
        byte[] header = TarWriter.newEntryHeader(entryName, size);
        log.debug("Writing segment {} to {}", (Object)uuid, (Object)this.file);
        return this.writeEntry(uuid, header, data, offset, size, generation);
    }

    private synchronized long writeEntry(UUID uuid, byte[] header, byte[] data, int offset, int size, int generation) throws IOException {
        long currentLength;
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        if (this.access == null) {
            this.access = new RandomAccessFile(this.file, "rw");
            this.channel = this.access.getChannel();
        }
        long initialLength = this.access.getFilePointer();
        this.access.write(header);
        this.access.write(data, offset, size);
        int padding = TarWriter.getPaddingSize(size);
        if (padding > 0) {
            this.access.write(ZERO_BYTES, 0, padding);
        }
        Preconditions.checkState(((currentLength = this.access.getFilePointer()) <= Integer.MAX_VALUE ? 1 : 0) != 0);
        TarEntry entry = new TarEntry(uuid.getMostSignificantBits(), uuid.getLeastSignificantBits(), (int)(currentLength - (long)size - (long)padding), size, generation);
        this.index.put(uuid, entry);
        this.monitor.written(currentLength - initialLength);
        return currentLength;
    }

    void addBinaryReference(int generation, UUID segmentId, String reference) {
        Set references;
        HashMap segmentToReferences = this.binaryReferences.get(generation);
        if (segmentToReferences == null) {
            segmentToReferences = Maps.newHashMap();
            this.binaryReferences.put(generation, segmentToReferences);
        }
        if ((references = (Set)segmentToReferences.get(segmentId)) == null) {
            references = Sets.newHashSet();
            segmentToReferences.put(segmentId, references);
        }
        references.add(reference);
    }

    void addGraphEdge(UUID from, UUID to) {
        HashSet adj = this.graph.get(from);
        if (adj == null) {
            adj = Sets.newHashSet();
            this.graph.put(from, adj);
        }
        adj.add(to);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void flush() throws IOException {
        File file = this.file;
        synchronized (file) {
            FileDescriptor descriptor = null;
            TarWriter tarWriter = this;
            synchronized (tarWriter) {
                if (this.access != null && !this.closed) {
                    descriptor = this.access.getFD();
                }
            }
            if (descriptor != null) {
                descriptor.sync();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        long currentPosition;
        long initialPosition;
        TarWriter tarWriter = this;
        synchronized (tarWriter) {
            Preconditions.checkState((!this.closed ? 1 : 0) != 0);
            this.closed = true;
        }
        if (this.access == null) {
            return;
        }
        File file = this.file;
        synchronized (file) {
            initialPosition = this.access.getFilePointer();
            this.writeBinaryReferences();
            this.writeGraph();
            this.writeIndex();
            this.access.write(ZERO_BYTES);
            this.access.write(ZERO_BYTES);
            currentPosition = this.access.getFilePointer();
            this.access.close();
        }
        this.monitor.written(currentPosition - initialPosition);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    TarWriter createNextGeneration() throws IOException {
        Preconditions.checkState((this.writeIndex >= 0 ? 1 : 0) != 0);
        TarWriter tarWriter = this;
        synchronized (tarWriter) {
            if (this.access == null) {
                return this;
            }
        }
        this.close();
        int newIndex = this.writeIndex + 1;
        return new TarWriter(this.file.getParentFile(), this.monitor, newIndex);
    }

    private void writeBinaryReferences() throws IOException {
        int binaryReferenceSize = 0;
        binaryReferenceSize += 4;
        binaryReferenceSize += 4;
        binaryReferenceSize += 4;
        binaryReferenceSize += 4;
        for (Map<UUID, Set<String>> map : this.binaryReferences.values()) {
            binaryReferenceSize += 4;
            binaryReferenceSize += 4;
            for (Set<String> references : map.values()) {
                binaryReferenceSize += 16;
                binaryReferenceSize += 4;
                for (String reference : references) {
                    binaryReferenceSize += 4;
                    binaryReferenceSize += reference.getBytes(Charsets.UTF_8).length;
                }
            }
        }
        ByteBuffer buffer = ByteBuffer.allocate(binaryReferenceSize);
        for (Map.Entry<Integer, Map<UUID, Set<String>>> entry : this.binaryReferences.entrySet()) {
            int generation = entry.getKey();
            Map<UUID, Set<String>> segmentToReferences = entry.getValue();
            buffer.putInt(generation);
            buffer.putInt(segmentToReferences.size());
            for (Map.Entry<UUID, Set<String>> se : segmentToReferences.entrySet()) {
                UUID segmentId = se.getKey();
                Set<String> references = se.getValue();
                buffer.putLong(segmentId.getMostSignificantBits());
                buffer.putLong(segmentId.getLeastSignificantBits());
                buffer.putInt(references.size());
                for (String reference : references) {
                    byte[] bytes = reference.getBytes(Charsets.UTF_8);
                    buffer.putInt(bytes.length);
                    buffer.put(bytes);
                }
            }
        }
        CRC32 cRC32 = new CRC32();
        cRC32.update(buffer.array(), 0, buffer.position());
        buffer.putInt((int)cRC32.getValue());
        buffer.putInt(this.binaryReferences.size());
        buffer.putInt(binaryReferenceSize);
        buffer.putInt(170934794);
        int n = TarWriter.getPaddingSize(binaryReferenceSize);
        byte[] header = TarWriter.newEntryHeader(this.file.getName() + ".brf", binaryReferenceSize + n);
        this.access.write(header);
        if (n > 0) {
            this.access.write(ZERO_BYTES, 0, n);
        }
        this.access.write(buffer.array());
    }

    private void writeGraph() throws IOException {
        int graphSize = 0;
        graphSize += 4;
        graphSize += 4;
        graphSize += 4;
        graphSize += 4;
        for (Map.Entry<UUID, Set<UUID>> entry : this.graph.entrySet()) {
            graphSize += 16;
            graphSize += 4;
            graphSize += 16 * entry.getValue().size();
        }
        ByteBuffer buffer = ByteBuffer.allocate(graphSize);
        for (Map.Entry<UUID, Set<UUID>> entry : this.graph.entrySet()) {
            UUID from = entry.getKey();
            buffer.putLong(from.getMostSignificantBits());
            buffer.putLong(from.getLeastSignificantBits());
            Set<UUID> adj = entry.getValue();
            buffer.putInt(adj.size());
            for (UUID to : adj) {
                buffer.putLong(to.getMostSignificantBits());
                buffer.putLong(to.getLeastSignificantBits());
            }
        }
        CRC32 cRC32 = new CRC32();
        cRC32.update(buffer.array(), 0, buffer.position());
        buffer.putInt((int)cRC32.getValue());
        buffer.putInt(this.graph.size());
        buffer.putInt(graphSize);
        buffer.putInt(170936074);
        int n = TarWriter.getPaddingSize(graphSize);
        this.access.write(TarWriter.newEntryHeader(this.file.getName() + ".gph", graphSize + n));
        if (n > 0) {
            this.access.write(ZERO_BYTES, 0, n);
        }
        this.access.write(buffer.array());
    }

    private void writeIndex() throws IOException {
        int indexSize = this.index.size() * 28 + 16;
        int padding = TarWriter.getPaddingSize(indexSize);
        String indexName = this.file.getName() + ".idx";
        byte[] header = TarWriter.newEntryHeader(indexName, indexSize + padding);
        ByteBuffer buffer = ByteBuffer.allocate(indexSize);
        TarEntry[] sorted = this.index.values().toArray(new TarEntry[this.index.size()]);
        Arrays.sort(sorted, TarEntry.IDENTIFIER_ORDER);
        for (TarEntry entry : sorted) {
            buffer.putLong(entry.msb());
            buffer.putLong(entry.lsb());
            buffer.putInt(entry.offset());
            buffer.putInt(entry.size());
            buffer.putInt(entry.generation());
        }
        CRC32 checksum = new CRC32();
        checksum.update(buffer.array(), 0, buffer.position());
        buffer.putInt((int)checksum.getValue());
        buffer.putInt(this.index.size());
        buffer.putInt(padding + indexSize);
        buffer.putInt(170937098);
        this.access.write(header);
        if (padding > 0) {
            this.access.write(ZERO_BYTES, 0, padding);
        }
        this.access.write(buffer.array());
    }

    private static byte[] newEntryHeader(String name, int size) {
        byte[] header = new byte[512];
        byte[] nameBytes = name.getBytes(Charsets.UTF_8);
        System.arraycopy(nameBytes, 0, header, 0, Math.min(nameBytes.length, 100));
        System.arraycopy(String.format("%07o", 256).getBytes(Charsets.UTF_8), 0, header, 100, 7);
        System.arraycopy(String.format("%07o", 0).getBytes(Charsets.UTF_8), 0, header, 108, 7);
        System.arraycopy(String.format("%07o", 0).getBytes(Charsets.UTF_8), 0, header, 116, 7);
        System.arraycopy(String.format("%011o", size).getBytes(Charsets.UTF_8), 0, header, 124, 11);
        long time = System.currentTimeMillis() / 1000L;
        System.arraycopy(String.format("%011o", time).getBytes(Charsets.UTF_8), 0, header, 136, 11);
        System.arraycopy(new byte[]{32, 32, 32, 32, 32, 32, 32, 32}, 0, header, 148, 8);
        header[156] = 48;
        int checksum = 0;
        for (int i = 0; i < header.length; ++i) {
            checksum += header[i] & 0xFF;
        }
        System.arraycopy(String.format("%06o\u0000 ", checksum).getBytes(Charsets.UTF_8), 0, header, 148, 8);
        return header;
    }

    synchronized void collectReferences(Set<UUID> referencedIds) {
        for (UUID uuid : Lists.reverse((List)Lists.newArrayList(this.index.keySet()))) {
            Set<UUID> refs;
            if (!referencedIds.remove(uuid) || (refs = this.graph.get(uuid)) == null) continue;
            referencedIds.addAll(refs);
        }
    }

    synchronized long fileLength() {
        return this.file.length();
    }

    synchronized File getFile() {
        return this.file;
    }

    synchronized boolean isClosed() {
        return this.closed;
    }

    public String toString() {
        return this.file.toString();
    }
}

