/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment;

import com.google.common.base.Predicate;
import com.google.common.base.Supplier;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.hash.Hashing;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.apache.jackrabbit.oak.api.Blob;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.commons.IOUtils;
import org.apache.jackrabbit.oak.commons.PathUtils;
import org.apache.jackrabbit.oak.plugins.memory.BinaryPropertyState;
import org.apache.jackrabbit.oak.plugins.memory.EmptyNodeState;
import org.apache.jackrabbit.oak.plugins.memory.MultiBinaryPropertyState;
import org.apache.jackrabbit.oak.plugins.memory.PropertyStates;
import org.apache.jackrabbit.oak.segment.CancelableDiff;
import org.apache.jackrabbit.oak.segment.RecordCache;
import org.apache.jackrabbit.oak.segment.RecordId;
import org.apache.jackrabbit.oak.segment.SegmentBlob;
import org.apache.jackrabbit.oak.segment.SegmentNodeBuilder;
import org.apache.jackrabbit.oak.segment.SegmentNodeState;
import org.apache.jackrabbit.oak.segment.SegmentReader;
import org.apache.jackrabbit.oak.segment.SegmentWriter;
import org.apache.jackrabbit.oak.segment.compaction.SegmentGCOptions;
import org.apache.jackrabbit.oak.spi.blob.BlobStore;
import org.apache.jackrabbit.oak.spi.state.ApplyDiff;
import org.apache.jackrabbit.oak.spi.state.NodeBuilder;
import org.apache.jackrabbit.oak.spi.state.NodeState;
import org.apache.jackrabbit.oak.spi.state.NodeStateDiff;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Compactor {
    private static final Logger log = LoggerFactory.getLogger(Compactor.class);
    private static boolean eagerFlush = Boolean.getBoolean("oak.compaction.eagerFlush");
    private final SegmentReader reader;
    private final BlobStore blobStore;
    private final SegmentWriter writer;
    private final Predicate<NodeState> includeInMap = new OfflineCompactionPredicate();
    private final ProgressTracker progress = new ProgressTracker();
    private final boolean binaryDedup;
    private final long binaryDedupMaxSize;
    private final Map<String, List<RecordId>> binaries = Maps.newHashMap();
    private boolean contentEqualityCheck;
    private final Supplier<Boolean> cancel;
    private static final int cacheSize;
    private final RecordCache<RecordId> cache = RecordCache.newRecordCache(cacheSize);

    public Compactor(SegmentReader reader, SegmentWriter writer, BlobStore blobStore, Supplier<Boolean> cancel, SegmentGCOptions gc) {
        this.reader = reader;
        this.writer = writer;
        this.blobStore = blobStore;
        this.cancel = cancel;
        this.binaryDedup = gc.isBinaryDeduplication();
        this.binaryDedupMaxSize = gc.getBinaryDeduplicationMaxSize();
    }

    private SegmentNodeBuilder process(NodeState before, NodeState after, NodeState onto) throws IOException {
        SegmentNodeBuilder builder = new SegmentNodeBuilder(this.writer.writeNode(onto), this.writer);
        new CompactDiff((NodeBuilder)builder).diff(before, after);
        return builder;
    }

    public SegmentNodeState compact(NodeState before, NodeState after, NodeState onto) throws IOException {
        this.progress.start();
        SegmentNodeState compacted = this.process(before, after, onto).getNodeState();
        this.writer.flush();
        this.progress.stop();
        return compacted;
    }

    private PropertyState compact(PropertyState property) {
        String name = property.getName();
        Type type = property.getType();
        if (type == Type.BINARY) {
            Blob blob = this.compact((Blob)property.getValue(Type.BINARY));
            return BinaryPropertyState.binaryProperty((String)name, (Blob)blob);
        }
        if (type == Type.BINARIES) {
            ArrayList<Blob> blobs = new ArrayList<Blob>();
            for (Blob blob : (Iterable)property.getValue(Type.BINARIES)) {
                blobs.add(this.compact(blob));
            }
            return MultiBinaryPropertyState.binaryPropertyFromBlob((String)name, blobs);
        }
        Object value = property.getValue(type);
        return PropertyStates.createProperty((String)name, (Object)value, (Type)type);
    }

    private Blob compact(Blob blob) {
        if (blob instanceof SegmentBlob) {
            SegmentBlob sb = (SegmentBlob)blob;
            try {
                boolean dedup;
                RecordId id = sb.getRecordId();
                RecordId compactedId = this.cache.get(id);
                if (compactedId != null) {
                    return new SegmentBlob(this.blobStore, compactedId);
                }
                this.progress.onBinary();
                if (sb.isExternal()) {
                    SegmentBlob clone = this.writer.writeBlob(sb);
                    this.cache.put(id, clone.getRecordId());
                    return clone;
                }
                if (sb.length() < 16512L) {
                    SegmentBlob clone = this.writer.writeBlob(blob);
                    this.cache.put(id, clone.getRecordId());
                    return clone;
                }
                ArrayList ids = null;
                String key = null;
                boolean bl = dedup = this.binaryDedup && blob.length() <= this.binaryDedupMaxSize;
                if (dedup && (ids = this.binaries.get(key = Compactor.getBlobKey(blob))) != null) {
                    for (RecordId duplicateId : ids) {
                        if (!new SegmentBlob(this.blobStore, duplicateId).equals(sb)) continue;
                        this.cache.put(id, duplicateId);
                        return new SegmentBlob(this.blobStore, duplicateId);
                    }
                }
                sb = this.writer.writeBlob(blob);
                this.cache.put(id, sb.getRecordId());
                if (dedup) {
                    if (ids == null) {
                        ids = Lists.newArrayList();
                        this.binaries.put(key, ids);
                    }
                    ids.add(sb.getRecordId());
                }
                return sb;
            }
            catch (IOException e) {
                log.warn("Failed to compact a blob", (Throwable)e);
            }
        }
        return blob;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static String getBlobKey(Blob blob) throws IOException {
        try (InputStream stream = blob.getNewStream();){
            byte[] buffer = new byte[4096];
            int n = IOUtils.readFully((InputStream)stream, (byte[])buffer, (int)0, (int)buffer.length);
            String string = blob.length() + ":" + Hashing.sha1().hashBytes(buffer, 0, n);
            return string;
        }
    }

    public void setContentEqualityCheck(boolean contentEqualityCheck) {
        this.contentEqualityCheck = contentEqualityCheck;
    }

    static {
        if (eagerFlush) {
            log.debug("Eager flush enabled.");
        }
        Integer ci = Integer.getInteger("compress-interval");
        Integer size = Integer.getInteger("oak.segment.compaction.cacheSize");
        if (size != null) {
            cacheSize = size;
        } else if (ci != null) {
            log.warn("Deprecated argument 'compress-interval', please use 'oak.segment.compaction.cacheSize' instead.");
            cacheSize = ci;
        } else {
            cacheSize = 100000;
        }
    }

    private static class OfflineCompactionPredicate
    implements Predicate<NodeState> {
        private static final long offlineThreshold = 65536L;

        private OfflineCompactionPredicate() {
        }

        public boolean apply(NodeState state) {
            if (state.getChildNodeCount(2L) > 1L) {
                return true;
            }
            long count = 0L;
            for (PropertyState ps : state.getProperties()) {
                Type type = ps.getType();
                for (int i = 0; i < ps.count(); ++i) {
                    long size = 0L;
                    if (type == Type.BINARY || type == Type.BINARIES) {
                        Blob blob = (Blob)ps.getValue(Type.BINARY, i);
                        if (blob instanceof SegmentBlob) {
                            if (!((SegmentBlob)blob).isExternal()) {
                                size += blob.length();
                            }
                        } else {
                            size += blob.length();
                        }
                    } else {
                        size = ps.size(i);
                    }
                    if (size < 65536L && (count += size) < 65536L) continue;
                    return true;
                }
            }
            return false;
        }
    }

    private static class ProgressTracker {
        private final long logAt = Long.getLong("compaction-progress-log", 150000L);
        private long start = 0L;
        private long nodes = 0L;
        private long properties = 0L;
        private long binaries = 0L;

        private ProgressTracker() {
        }

        void start() {
            this.nodes = 0L;
            this.properties = 0L;
            this.binaries = 0L;
            this.start = System.currentTimeMillis();
        }

        void onNode() {
            if (++this.nodes % this.logAt == 0L) {
                this.logProgress(this.start, false);
                this.start = System.currentTimeMillis();
            }
        }

        void onProperty() {
            ++this.properties;
        }

        void onBinary() {
            ++this.binaries;
        }

        void stop() {
            this.logProgress(this.start, true);
        }

        private void logProgress(long start, boolean done) {
            log.debug("Compacted {} nodes, {} properties, {} binaries in {} ms.", new Object[]{this.nodes, this.properties, this.binaries, System.currentTimeMillis() - start});
            if (done) {
                log.info("Finished compaction: {} nodes, {} properties, {} binaries.", new Object[]{this.nodes, this.properties, this.binaries});
            }
        }
    }

    private class CompactDiff
    extends ApplyDiff {
        private IOException exception;
        private final String path;

        CompactDiff(NodeBuilder builder) {
            super(builder);
            this.path = log.isTraceEnabled() ? "/" : null;
        }

        private CompactDiff(NodeBuilder builder, String path, String childName) {
            super(builder);
            this.path = path != null ? PathUtils.concat((String)path, (String)childName) : null;
        }

        boolean diff(NodeState before, NodeState after) throws IOException {
            boolean success = after.compareAgainstBaseState(before, (NodeStateDiff)new CancelableDiff((NodeStateDiff)this, (Supplier<Boolean>)Compactor.this.cancel));
            if (this.exception != null) {
                throw new IOException(this.exception);
            }
            return success;
        }

        public boolean propertyAdded(PropertyState after) {
            if (this.path != null) {
                log.trace("propertyAdded {}/{}", (Object)this.path, (Object)after.getName());
            }
            Compactor.this.progress.onProperty();
            return super.propertyAdded(Compactor.this.compact(after));
        }

        public boolean propertyChanged(PropertyState before, PropertyState after) {
            if (this.path != null) {
                log.trace("propertyChanged {}/{}", (Object)this.path, (Object)after.getName());
            }
            Compactor.this.progress.onProperty();
            return super.propertyChanged(before, Compactor.this.compact(after));
        }

        public boolean childNodeAdded(String name, NodeState after) {
            if (this.path != null) {
                log.trace("childNodeAdded {}/{}", (Object)this.path, (Object)name);
            }
            RecordId id = null;
            if (after instanceof SegmentNodeState) {
                id = ((SegmentNodeState)after).getRecordId();
                RecordId compactedId = Compactor.this.cache.get(id);
                if (compactedId != null) {
                    this.builder.setChildNode(name, (NodeState)new SegmentNodeState(Compactor.this.reader, Compactor.this.writer, compactedId));
                    return true;
                }
            }
            Compactor.this.progress.onNode();
            try {
                NodeBuilder child = eagerFlush ? this.builder.setChildNode(name) : EmptyNodeState.EMPTY_NODE.builder();
                boolean success = new CompactDiff(child, this.path, name).diff(EmptyNodeState.EMPTY_NODE, after);
                if (success) {
                    SegmentNodeState state = Compactor.this.writer.writeNode(child.getNodeState());
                    this.builder.setChildNode(name, (NodeState)state);
                    if (id != null && Compactor.this.includeInMap.apply((Object)after)) {
                        Compactor.this.cache.put(id, state.getRecordId());
                    }
                }
                return success;
            }
            catch (IOException e) {
                this.exception = e;
                return false;
            }
        }

        public boolean childNodeChanged(String name, NodeState before, NodeState after) {
            if (this.path != null) {
                log.trace("childNodeChanged {}/{}", (Object)this.path, (Object)name);
            }
            RecordId id = null;
            if (after instanceof SegmentNodeState) {
                id = ((SegmentNodeState)after).getRecordId();
                RecordId compactedId = Compactor.this.cache.get(id);
                if (compactedId != null) {
                    this.builder.setChildNode(name, (NodeState)new SegmentNodeState(Compactor.this.reader, Compactor.this.writer, compactedId));
                    return true;
                }
            }
            if (Compactor.this.contentEqualityCheck && before.equals(after)) {
                return true;
            }
            Compactor.this.progress.onNode();
            try {
                NodeBuilder child = this.builder.getChildNode(name);
                boolean success = new CompactDiff(child, this.path, name).diff(before, after);
                if (success) {
                    RecordId compactedId = Compactor.this.writer.writeNode(child.getNodeState()).getRecordId();
                    if (id != null) {
                        Compactor.this.cache.put(id, compactedId);
                    }
                }
                return success;
            }
            catch (IOException e) {
                this.exception = e;
                return false;
            }
        }
    }
}

