/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.file;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Stopwatch;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.commons.math3.stat.descriptive.DescriptiveStatistics;
import org.apache.jackrabbit.oak.api.jmx.CacheStatsMBean;
import org.apache.jackrabbit.oak.commons.IOUtils;
import org.apache.jackrabbit.oak.plugins.blob.ReferenceCollector;
import org.apache.jackrabbit.oak.plugins.memory.EmptyNodeState;
import org.apache.jackrabbit.oak.segment.BinaryReferenceConsumer;
import org.apache.jackrabbit.oak.segment.BinaryReferences;
import org.apache.jackrabbit.oak.segment.CachingSegmentReader;
import org.apache.jackrabbit.oak.segment.Compactor;
import org.apache.jackrabbit.oak.segment.RecordId;
import org.apache.jackrabbit.oak.segment.Revisions;
import org.apache.jackrabbit.oak.segment.Segment;
import org.apache.jackrabbit.oak.segment.SegmentBufferWriter;
import org.apache.jackrabbit.oak.segment.SegmentCache;
import org.apache.jackrabbit.oak.segment.SegmentGraph;
import org.apache.jackrabbit.oak.segment.SegmentId;
import org.apache.jackrabbit.oak.segment.SegmentIdFactory;
import org.apache.jackrabbit.oak.segment.SegmentNodeState;
import org.apache.jackrabbit.oak.segment.SegmentNotFoundException;
import org.apache.jackrabbit.oak.segment.SegmentReader;
import org.apache.jackrabbit.oak.segment.SegmentStore;
import org.apache.jackrabbit.oak.segment.SegmentTracker;
import org.apache.jackrabbit.oak.segment.SegmentWriter;
import org.apache.jackrabbit.oak.segment.SegmentWriterBuilder;
import org.apache.jackrabbit.oak.segment.WriterCacheManager;
import org.apache.jackrabbit.oak.segment.compaction.SegmentGCOptions;
import org.apache.jackrabbit.oak.segment.file.CompactionGainEstimate;
import org.apache.jackrabbit.oak.segment.file.FileReaper;
import org.apache.jackrabbit.oak.segment.file.FileStoreBuilder;
import org.apache.jackrabbit.oak.segment.file.FileStoreStats;
import org.apache.jackrabbit.oak.segment.file.GCEstimation;
import org.apache.jackrabbit.oak.segment.file.GCJournal;
import org.apache.jackrabbit.oak.segment.file.GCListener;
import org.apache.jackrabbit.oak.segment.file.InvalidFileStoreVersionException;
import org.apache.jackrabbit.oak.segment.file.Manifest;
import org.apache.jackrabbit.oak.segment.file.SafeRunnable;
import org.apache.jackrabbit.oak.segment.file.Scheduler;
import org.apache.jackrabbit.oak.segment.file.SizeDeltaGcEstimation;
import org.apache.jackrabbit.oak.segment.file.TarReader;
import org.apache.jackrabbit.oak.segment.file.TarRevisions;
import org.apache.jackrabbit.oak.segment.file.TarWriter;
import org.apache.jackrabbit.oak.spi.blob.BlobStore;
import org.apache.jackrabbit.oak.spi.state.NodeBuilder;
import org.apache.jackrabbit.oak.spi.state.NodeState;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileStore
implements SegmentStore,
Closeable {
    private static final Logger log = LoggerFactory.getLogger(FileStore.class);
    private static final int MB = 0x100000;
    private static final int INVALID_STORE_VERSION = 0;
    private static final int CURRENT_STORE_VERSION = 1;
    private static final Pattern FILE_NAME_PATTERN = Pattern.compile("(data|bulk)((0|[1-9][0-9]*)[0-9]{4})([a-z])?.tar");
    static final String FILE_NAME_FORMAT = "data%05d%s.tar";
    private static final String LOCK_FILE_NAME = "repo.lock";
    private static final String MANIFEST_FILE_NAME = "manifest";
    private static final AtomicLong GC_COUNT = new AtomicLong(0L);
    static final boolean MEMORY_MAPPING_DEFAULT = "64".equals(System.getProperty("sun.arch.data.model", "32"));
    @Nonnull
    private final SegmentTracker tracker;
    @Nonnull
    private final SegmentWriter segmentWriter;
    @Nonnull
    private final CachingSegmentReader segmentReader;
    @Nonnull
    private final BinaryReferenceConsumer binaryReferenceConsumer;
    private final File directory;
    private final BlobStore blobStore;
    private final int maxFileSize;
    private final boolean memoryMapping;
    private volatile List<TarReader> readers;
    private volatile TarWriter tarWriter;
    private final RandomAccessFile lockFile;
    private final FileLock lock;
    @Nonnull
    private final TarRevisions revisions;
    private final Scheduler fileStoreScheduler = new Scheduler("FileStore background tasks");
    private final SegmentGCOptions gcOptions;
    private final GCJournal gcJournal;
    private final Semaphore gcSemaphore = new Semaphore(1);
    private final FileReaper fileReaper = new FileReaper();
    private final GCListener gcListener;
    private final AtomicBoolean sufficientDiskSpace;
    private volatile boolean shutdown;
    private final ReadWriteLock fileStoreLock = new ReentrantReadWriteLock();
    private final FileStoreStats stats;
    @Nonnull
    private final SegmentCache segmentCache;
    private final SegmentIdFactory segmentIdFactory = new SegmentIdFactory(){

        @Override
        @Nonnull
        public SegmentId newSegmentId(long msb, long lsb) {
            return new SegmentId(FileStore.this, msb, lsb);
        }
    };

    FileStore(FileStoreBuilder builder, boolean readOnly) throws InvalidFileStoreVersionException, IOException {
        this.directory = builder.getDirectory();
        if (!readOnly) {
            this.lockFile = new RandomAccessFile(new File(this.directory, LOCK_FILE_NAME), "rw");
            try {
                this.lock = this.lockFile.getChannel().lock();
            }
            catch (OverlappingFileLockException ex) {
                throw new IllegalStateException(this.directory.getAbsolutePath() + " is in use by another store.", ex);
            }
        } else {
            this.lockFile = null;
            this.lock = null;
        }
        this.tracker = new SegmentTracker();
        this.revisions = builder.getRevisions();
        this.blobStore = builder.getBlobStore();
        this.segmentCache = new SegmentCache(builder.getSegmentCacheSize());
        this.segmentReader = new CachingSegmentReader(new Supplier<SegmentWriter>(){

            @Override
            public SegmentWriter get() {
                return FileStore.this.segmentWriter;
            }
        }, this.blobStore, builder.getStringCacheSize(), builder.getTemplateCacheSize());
        this.binaryReferenceConsumer = new BinaryReferenceConsumer(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void consume(int generation, UUID segmentId, String binaryReference) {
                FileStore.this.fileStoreLock.writeLock().lock();
                try {
                    FileStore.this.tarWriter.addBinaryReference(generation, segmentId, binaryReference);
                }
                finally {
                    FileStore.this.fileStoreLock.writeLock().unlock();
                }
            }
        };
        this.segmentWriter = SegmentWriterBuilder.segmentWriterBuilder("sys").withGeneration(new Supplier<Integer>(){

            @Override
            public Integer get() {
                return FileStore.this.getGcGeneration();
            }
        }).withWriterPool().with(builder.getCacheManager()).build(this);
        this.maxFileSize = builder.getMaxFileSize() * 0x100000;
        this.memoryMapping = builder.getMemoryMapping();
        this.gcListener = builder.getGcListener();
        this.gcOptions = builder.getGcOptions();
        this.gcJournal = new GCJournal(this.directory);
        Map<Integer, Map<Character, File>> map = FileStore.collectFiles(this.directory);
        Manifest manifest = Manifest.empty();
        if (map.size() > 0) {
            manifest = this.checkManifest(this.openManifest());
        }
        this.saveManifest(manifest);
        this.readers = Lists.newArrayListWithCapacity(map.size());
        Object[] indices = map.keySet().toArray(new Integer[map.size()]);
        Arrays.sort(indices);
        for (int i = indices.length - 1; i >= 0; --i) {
            if (!readOnly) {
                this.readers.add(TarReader.open(map.get(indices[i]), this.memoryMapping));
                continue;
            }
            boolean recover = i == indices.length - 1;
            this.readers.add(TarReader.openRO(map.get(indices[i]), this.memoryMapping, recover));
        }
        this.stats = new FileStoreStats(builder.getStatsProvider(), this, this.size());
        if (!readOnly) {
            int writeNumber = 0;
            if (indices.length > 0) {
                writeNumber = (Integer)indices[indices.length - 1] + 1;
            }
            this.tarWriter = new TarWriter(this.directory, this.stats, writeNumber);
        } else {
            this.tarWriter = null;
        }
        this.sufficientDiskSpace = new AtomicBoolean(true);
        if (!readOnly) {
            this.fileStoreScheduler.scheduleAtFixedRate(String.format("TarMK flush [%s]", this.directory), 5L, TimeUnit.SECONDS, new Runnable(){

                @Override
                public void run() {
                    try {
                        FileStore.this.flush();
                    }
                    catch (IOException e) {
                        log.warn("Failed to flush the TarMK at {}", (Object)FileStore.this.directory, (Object)e);
                    }
                }
            });
            this.fileStoreScheduler.scheduleAtFixedRate(String.format("TarMK filer reaper [%s]", this.directory), 5L, TimeUnit.SECONDS, new Runnable(){

                @Override
                public void run() {
                    FileStore.this.fileReaper.reap();
                }
            });
            this.fileStoreScheduler.scheduleAtFixedRate(String.format("TarMK disk space check [%s]", this.directory), 1L, TimeUnit.MINUTES, new Runnable(){

                @Override
                public void run() {
                    FileStore.this.checkDiskSpace();
                }
            });
        }
        if (readOnly) {
            log.info("TarMK ReadOnly opened: {} (mmap={})", (Object)this.directory, (Object)this.memoryMapping);
        } else {
            log.info("TarMK opened: {} (mmap={})", (Object)this.directory, (Object)this.memoryMapping);
        }
        log.debug("TarMK readers {}", (Object)this.readers);
    }

    FileStore bind(TarRevisions revisions) throws IOException {
        revisions.bind(this, this.initialNode());
        return this;
    }

    private File getManifestFile() {
        return new File(this.directory, MANIFEST_FILE_NAME);
    }

    private Manifest openManifest() throws IOException {
        File file = this.getManifestFile();
        if (file.exists()) {
            return Manifest.load(file);
        }
        return null;
    }

    private Manifest checkManifest(Manifest manifest) throws InvalidFileStoreVersionException {
        if (manifest == null) {
            throw new InvalidFileStoreVersionException("Using oak-segment-tar, but oak-segment should be used");
        }
        int storeVersion = manifest.getStoreVersion(0);
        if (storeVersion <= 0) {
            throw new IllegalStateException("Invalid store version");
        }
        if (storeVersion < 1) {
            throw new InvalidFileStoreVersionException("Using a too recent version of oak-segment-tar");
        }
        if (storeVersion > 1) {
            throw new InvalidFileStoreVersionException("Using a too old version of oak-segment tar");
        }
        return manifest;
    }

    private void saveManifest(Manifest manifest) throws IOException {
        manifest.setStoreVersion(1);
        manifest.save(this.getManifestFile());
    }

    @Nonnull
    private Supplier<RecordId> initialNode() {
        return new Supplier<RecordId>(){

            @Override
            public RecordId get() {
                try {
                    SegmentWriter writer = SegmentWriterBuilder.segmentWriterBuilder("init").build(FileStore.this);
                    NodeBuilder builder = EmptyNodeState.EMPTY_NODE.builder();
                    builder.setChildNode("root", EmptyNodeState.EMPTY_NODE);
                    SegmentNodeState node = writer.writeNode(builder.getNodeState());
                    writer.flush();
                    return node.getRecordId();
                }
                catch (IOException e) {
                    String msg = "Failed to write initial node";
                    log.error(msg, e);
                    throw new IllegalStateException(msg, e);
                }
            }
        };
    }

    private int getGcGeneration() {
        return this.revisions.getHead().getSegmentId().getGcGeneration();
    }

    @Nonnull
    public CacheStatsMBean getSegmentCacheStats() {
        return this.segmentCache.getCacheStats();
    }

    @Nonnull
    public CacheStatsMBean getStringCacheStats() {
        return this.segmentReader.getStringCacheStats();
    }

    @Nonnull
    public CacheStatsMBean getTemplateCacheStats() {
        return this.segmentReader.getTemplateCacheStats();
    }

    @CheckForNull
    public CacheStatsMBean getStringDeduplicationCacheStats() {
        return this.segmentWriter.getStringCacheStats();
    }

    @CheckForNull
    public CacheStatsMBean getTemplateDeduplicationCacheStats() {
        return this.segmentWriter.getTemplateCacheStats();
    }

    @CheckForNull
    public CacheStatsMBean getNodeDeduplicationCacheStats() {
        return this.segmentWriter.getNodeCacheStats();
    }

    public Runnable getGCRunner() {
        return new SafeRunnable(String.format("TarMK revision gc [%s]", this.directory), new Runnable(){

            @Override
            public void run() {
                try {
                    FileStore.this.gc();
                }
                catch (IOException e) {
                    log.error("Error running compaction", e);
                }
            }
        });
    }

    public void gc() throws IOException {
        this.gcListener.info("TarMK GC #{}: started", GC_COUNT.incrementAndGet());
        Stopwatch watch = Stopwatch.createStarted();
        int gainThreshold = this.gcOptions.getGainThreshold();
        boolean sufficientEstimatedGain = true;
        if (gainThreshold <= 0) {
            this.gcListener.info("TarMK GC #{}: estimation skipped because gain threshold value ({} <= 0)", GC_COUNT, gainThreshold);
        } else if (this.gcOptions.isPaused()) {
            this.gcListener.info("TarMK GC #{}: estimation skipped because compaction is paused", GC_COUNT);
        } else {
            this.gcListener.info("TarMK GC #{}: estimation started", GC_COUNT);
            Supplier<Boolean> cancel = this.newCancelCompactionCondition();
            GCEstimation estimate = this.estimateCompactionGain(cancel);
            if (cancel.get().booleanValue()) {
                this.gcListener.info("TarMK GC #{}: estimation interrupted: {}. Skipping compaction.", GC_COUNT, cancel);
            }
            sufficientEstimatedGain = estimate.gcNeeded();
            String gcLog = estimate.gcLog();
            if (sufficientEstimatedGain) {
                this.gcListener.info("TarMK GC #{}: estimation completed in {} ({} ms). {}", GC_COUNT, watch, watch.elapsed(TimeUnit.MILLISECONDS), gcLog);
            } else {
                this.gcListener.skipped("TarMK GC #{}: estimation completed in {} ({} ms). {}", GC_COUNT, watch, watch.elapsed(TimeUnit.MILLISECONDS), gcLog);
            }
        }
        if (sufficientEstimatedGain) {
            if (!this.gcOptions.isPaused()) {
                FileStore.logAndClear(this.segmentWriter.getNodeWriteTimeStats(), this.segmentWriter.getNodeCompactTimeStats());
                FileStore.log(this.segmentWriter.getNodeCacheOccupancyInfo());
                Runnable cleanupTask = this.compact();
                if (cleanupTask != null) {
                    cleanupTask.run();
                }
                FileStore.logAndClear(this.segmentWriter.getNodeWriteTimeStats(), this.segmentWriter.getNodeCompactTimeStats());
                FileStore.log(this.segmentWriter.getNodeCacheOccupancyInfo());
            } else {
                this.gcListener.skipped("TarMK GC #{}: compaction paused", GC_COUNT);
            }
        }
    }

    private static void logAndClear(@Nonnull DescriptiveStatistics nodeWriteTimeStats, @Nonnull DescriptiveStatistics nodeCompactTimeStats) {
        log.info("Node write time statistics (ns) {}", (Object)FileStore.toString(nodeWriteTimeStats));
        log.info("Node compact time statistics (ns) {}", (Object)FileStore.toString(nodeCompactTimeStats));
        nodeWriteTimeStats.clear();
        nodeCompactTimeStats.clear();
    }

    private static void log(@CheckForNull String nodeCacheOccupancyInfo) {
        if (nodeCacheOccupancyInfo != null) {
            log.info("NodeCache occupancy: {}", (Object)nodeCacheOccupancyInfo);
        }
    }

    private static String toString(DescriptiveStatistics statistics) {
        DecimalFormat sci = new DecimalFormat("##0.0E0");
        DecimalFormatSymbols symbols = sci.getDecimalFormatSymbols();
        symbols.setNaN("NaN");
        symbols.setInfinity("Inf");
        sci.setDecimalFormatSymbols(symbols);
        return "min=" + sci.format(statistics.getMin()) + ", 10%=" + sci.format(statistics.getPercentile(10.0)) + ", 50%=" + sci.format(statistics.getPercentile(50.0)) + ", 90%=" + sci.format(statistics.getPercentile(90.0)) + ", max=" + sci.format(statistics.getMax()) + ", mean=" + sci.format(statistics.getMean()) + ", stdev=" + sci.format(statistics.getStandardDeviation()) + ", N=" + sci.format(statistics.getN());
    }

    static Map<Integer, Map<Character, File>> collectFiles(File directory) {
        Map<Character, File> files;
        HashMap<Integer, Map<Character, File>> dataFiles = Maps.newHashMap();
        HashMap<Integer, File> bulkFiles = Maps.newHashMap();
        for (File file : directory.listFiles()) {
            Matcher matcher = FILE_NAME_PATTERN.matcher(file.getName());
            if (!matcher.matches()) continue;
            Object index = Integer.parseInt(matcher.group(2));
            if ("data".equals(matcher.group(1))) {
                files = (HashMap<Character, File>)dataFiles.get(index);
                if (files == null) {
                    files = Maps.newHashMap();
                    dataFiles.put((Integer)index, files);
                }
                Character generation = Character.valueOf('a');
                if (matcher.group(4) != null) {
                    generation = Character.valueOf(matcher.group(4).charAt(0));
                }
                Preconditions.checkState(files.put(generation, file) == null);
                continue;
            }
            Preconditions.checkState(bulkFiles.put((Integer)index, file) == null);
        }
        if (!bulkFiles.isEmpty()) {
            Integer newIndex;
            int position;
            Object[] indices;
            log.info("Upgrading TarMK file names in {}", (Object)directory);
            if (!dataFiles.isEmpty()) {
                indices = dataFiles.keySet().toArray(new Integer[dataFiles.size()]);
                Arrays.sort(indices);
                position = Math.max((Integer)indices[indices.length - 1] + 1, bulkFiles.size());
                for (Object index : indices) {
                    files = (Map)dataFiles.remove(index);
                    newIndex = position++;
                    for (Character generation : Sets.newHashSet(files.keySet())) {
                        File file = (File)files.get(generation);
                        File newFile = new File(directory, String.format(FILE_NAME_FORMAT, newIndex, generation));
                        log.info("Renaming {} to {}", (Object)file, (Object)newFile);
                        file.renameTo(newFile);
                        files.put(generation, newFile);
                    }
                    dataFiles.put(newIndex, files);
                }
            }
            indices = bulkFiles.keySet().toArray(new Integer[bulkFiles.size()]);
            Arrays.sort(indices);
            position = 0;
            for (Object index : indices) {
                File file = (File)bulkFiles.remove(index);
                newIndex = position++;
                File newFile = new File(directory, String.format(FILE_NAME_FORMAT, newIndex, "a"));
                log.info("Renaming {} to {}", (Object)file, (Object)newFile);
                file.renameTo(newFile);
                dataFiles.put(newIndex, Collections.singletonMap(Character.valueOf('a'), newFile));
            }
        }
        return dataFiles;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long size() {
        ImmutableList<TarReader> readersSnapshot = null;
        long writeFileSnapshotSize = 0L;
        this.fileStoreLock.readLock().lock();
        try {
            readersSnapshot = ImmutableList.copyOf(this.readers);
            writeFileSnapshotSize = this.tarWriter != null ? this.tarWriter.fileLength() : 0L;
        }
        finally {
            this.fileStoreLock.readLock().unlock();
        }
        long size = writeFileSnapshotSize;
        for (TarReader reader : readersSnapshot) {
            size += reader.size();
        }
        return size;
    }

    public int readerCount() {
        this.fileStoreLock.readLock().lock();
        try {
            int n = this.readers.size();
            return n;
        }
        finally {
            this.fileStoreLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int count() {
        this.fileStoreLock.readLock().lock();
        try {
            int count = 0;
            if (this.tarWriter != null) {
                count += this.tarWriter.count();
            }
            for (TarReader reader : this.readers) {
                count += reader.count();
            }
            int n = count;
            return n;
        }
        finally {
            this.fileStoreLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    GCEstimation estimateCompactionGain(Supplier<Boolean> stop) {
        if (this.gcOptions.isGcSizeDeltaEstimation()) {
            SizeDeltaGcEstimation e = new SizeDeltaGcEstimation(this.gcOptions, this.gcJournal, this.stats.getApproximateSize());
            return e;
        }
        CompactionGainEstimate estimate = new CompactionGainEstimate(this.getHead(), this.count(), stop, this.gcOptions.getGainThreshold());
        this.fileStoreLock.readLock().lock();
        try {
            for (TarReader reader : this.readers) {
                reader.accept(estimate);
                if (!stop.get().booleanValue()) continue;
                break;
            }
        }
        finally {
            this.fileStoreLock.readLock().unlock();
        }
        return estimate;
    }

    public FileStoreStats getStats() {
        return this.stats;
    }

    public void flush() throws IOException {
        this.revisions.flush(new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                FileStore.this.segmentWriter.flush();
                FileStore.this.tarWriter.flush();
                FileStore.this.stats.flushed();
                return null;
            }
        });
    }

    private static Closeable withSemaphore(final @Nonnull Semaphore semaphore) {
        if (!semaphore.tryAcquire()) {
            throw new IllegalStateException("Compaction or cleanup already in progress");
        }
        return new Closeable(){

            @Override
            public void close() {
                semaphore.release();
            }
        };
    }

    public void cleanup() throws IOException {
        this.fileReaper.add(this.cleanupOldGenerations(this.getGcGeneration()));
    }

    private List<File> cleanupOldGenerations(int gcGeneration) throws IOException {
        final int reclaimGeneration = gcGeneration - this.gcOptions.getRetainedGenerations();
        Predicate<Integer> reclaimPredicate = new Predicate<Integer>(){

            @Override
            public boolean apply(Integer generation) {
                return generation <= reclaimGeneration;
            }
        };
        return this.cleanup(reclaimPredicate, "gc-count=" + GC_COUNT + ",gc-status=success" + ",store-generation=" + gcGeneration + ",reclaim-predicate=(generation<=" + reclaimGeneration + ")");
    }

    private List<File> cleanupGeneration(final int gcGeneration) throws IOException {
        Predicate<Integer> cleanupPredicate = new Predicate<Integer>(){

            @Override
            public boolean apply(Integer generation) {
                return generation == gcGeneration;
            }
        };
        return this.cleanup(cleanupPredicate, "gc-count=" + GC_COUNT + ",gc-status=failed" + ",store-generation=" + (gcGeneration - 1) + ",reclaim-predicate=(generation==" + gcGeneration + ")");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<File> cleanup(@Nonnull Predicate<Integer> reclaimGeneration, @Nonnull String gcInfo) throws IOException {
        try (Closeable s = FileStore.withSemaphore(this.gcSemaphore);){
            Stopwatch watch = Stopwatch.createStarted();
            HashSet<UUID> bulkRefs = Sets.newHashSet();
            LinkedHashMap<TarReader, TarReader> cleaned = Maps.newLinkedHashMap();
            long initialSize = 0L;
            this.fileStoreLock.writeLock().lock();
            try {
                this.gcListener.info("TarMK GC #{}: cleanup started.", GC_COUNT);
                this.newWriter();
                this.segmentCache.clear();
                System.gc();
                this.collectBulkReferences(bulkRefs);
                for (TarReader tarReader : this.readers) {
                    cleaned.put(tarReader, tarReader);
                    initialSize += tarReader.size();
                }
            }
            finally {
                this.fileStoreLock.writeLock().unlock();
            }
            this.gcListener.info("TarMK GC #{}: current repository size is {} ({} bytes)", GC_COUNT, IOUtils.humanReadableByteCount(initialSize), initialSize);
            HashSet<UUID> reclaim = Sets.newHashSet();
            for (Object reader : cleaned.keySet()) {
                ((TarReader)reader).mark(bulkRefs, reclaim, reclaimGeneration);
                log.info("{}: size of bulk references/reclaim set {}/{}", reader, bulkRefs.size(), reclaim.size());
                if (!this.shutdown) continue;
                this.gcListener.info("TarMK GC #{}: cleanup interrupted", GC_COUNT);
                break;
            }
            HashSet<UUID> hashSet = Sets.newHashSet();
            for (TarReader reader : cleaned.keySet()) {
                cleaned.put(reader, reader.sweep(reclaim, hashSet));
                if (!this.shutdown) continue;
                this.gcListener.info("TarMK GC #{}: cleanup interrupted", GC_COUNT);
                break;
            }
            long afterCleanupSize = 0L;
            ArrayList<TarReader> oldReaders = Lists.newArrayList();
            this.fileStoreLock.writeLock().lock();
            try {
                ArrayList<TarReader> sweptReaders = Lists.newArrayList();
                for (TarReader reader : this.readers) {
                    if (cleaned.containsKey(reader)) {
                        TarReader newReader = (TarReader)cleaned.get(reader);
                        if (newReader != null) {
                            sweptReaders.add(newReader);
                            afterCleanupSize += newReader.size();
                        }
                        if (newReader == reader) continue;
                        oldReaders.add(reader);
                        continue;
                    }
                    sweptReaders.add(reader);
                }
                this.readers = sweptReaders;
            }
            finally {
                this.fileStoreLock.writeLock().unlock();
            }
            this.tracker.clearSegmentIdTables(hashSet, gcInfo);
            LinkedList<File> toRemove = Lists.newLinkedList();
            for (TarReader oldReader : oldReaders) {
                FileStore.closeAndLogOnFail(oldReader);
                File file = oldReader.getFile();
                this.gcListener.info("TarMK GC #{}: cleanup marking file for deletion: {}", GC_COUNT, file.getName());
                toRemove.addLast(file);
            }
            long finalSize = this.size();
            long reclaimedSize = initialSize - afterCleanupSize;
            this.stats.reclaimed(reclaimedSize);
            this.gcJournal.persist(reclaimedSize, finalSize);
            this.gcListener.cleaned(reclaimedSize, finalSize);
            this.gcListener.info("TarMK GC #{}: cleanup completed in {} ({} ms). Post cleanup size is {} ({} bytes) and space reclaimed {} ({} bytes).", GC_COUNT, watch, watch.elapsed(TimeUnit.MILLISECONDS), IOUtils.humanReadableByteCount(finalSize), finalSize, IOUtils.humanReadableByteCount(reclaimedSize), reclaimedSize);
            LinkedList<File> linkedList = toRemove;
            return linkedList;
        }
    }

    private void collectBulkReferences(Set<UUID> bulkRefs) {
        HashSet<UUID> refs = Sets.newHashSet();
        for (SegmentId id : this.tracker.getReferencedSegmentIds()) {
            refs.add(id.asUUID());
        }
        this.tarWriter.collectReferences(refs);
        for (UUID ref : refs) {
            if (SegmentId.isDataSegmentId(ref.getLeastSignificantBits())) continue;
            bulkRefs.add(ref);
        }
    }

    public void collectBlobReferences(ReferenceCollector collector) throws IOException {
        this.segmentWriter.flush();
        ArrayList<TarReader> tarReaders = Lists.newArrayList();
        this.fileStoreLock.writeLock().lock();
        try {
            this.newWriter();
            tarReaders.addAll(this.readers);
        }
        finally {
            this.fileStoreLock.writeLock().unlock();
        }
        int minGeneration = this.getGcGeneration() - this.gcOptions.getRetainedGenerations() + 1;
        for (TarReader tarReader : tarReaders) {
            tarReader.collectBlobReferences(collector, BinaryReferences.newReferenceReader(this), minGeneration);
        }
    }

    private Supplier<Boolean> newCancelCompactionCondition() {
        return new CancelCompactionSupplier(this);
    }

    private static Supplier<Boolean> timeOut(final long duration, final @Nonnull TimeUnit unit) {
        return new Supplier<Boolean>(){
            long deadline;
            {
                this.deadline = System.currentTimeMillis() + TimeUnit.MILLISECONDS.convert(duration, unit);
            }

            @Override
            public Boolean get() {
                return System.currentTimeMillis() > this.deadline;
            }
        };
    }

    private static Supplier<Boolean> or(@Nonnull Supplier<Boolean> supplier1, @Nonnull Supplier<Boolean> supplier2) {
        if (supplier1.get().booleanValue()) {
            return Suppliers.ofInstance(true);
        }
        return supplier2;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @CheckForNull
    public Runnable compact() throws IOException {
        try (Closeable s = FileStore.withSemaphore(this.gcSemaphore);){
            Stopwatch watch = Stopwatch.createStarted();
            this.gcListener.info("TarMK GC #{}: compaction started, gc options={}", GC_COUNT, this.gcOptions);
            SegmentNodeState before = this.getHead();
            final int newGeneration = this.getGcGeneration() + 1;
            SegmentBufferWriter bufferWriter = new SegmentBufferWriter(this, this.tracker, this.segmentReader, "c", newGeneration);
            Supplier<Boolean> cancel = this.newCancelCompactionCondition();
            SegmentNodeState after = this.compact(bufferWriter, before, cancel);
            if (after == null) {
                this.gcListener.info("TarMK GC #{}: compaction cancelled: {}.", GC_COUNT, cancel);
                Runnable runnable = null;
                return runnable;
            }
            this.gcListener.info("TarMK GC #{}: compacted {} to {}", GC_COUNT, before.getRecordId(), after.getRecordId());
            int cycles = 0;
            boolean success = false;
            while (cycles < this.gcOptions.getRetryCount() && !(success = this.revisions.setHead(before.getRecordId(), after.getRecordId(), TarRevisions.EXPEDITE_OPTION))) {
                this.gcListener.info("TarMK GC #{}: compaction detected concurrent commits while compacting. Compacting these commits. Cycle {} of {}", GC_COUNT, ++cycles, this.gcOptions.getRetryCount());
                SegmentNodeState head = this.getHead();
                after = this.compact(bufferWriter, head, cancel);
                if (after == null) {
                    this.gcListener.info("TarMK GC #{}: compaction cancelled: {}.", GC_COUNT, cancel);
                    Runnable runnable = null;
                    return runnable;
                }
                this.gcListener.info("TarMK GC #{}: compacted {} against {} to {}", GC_COUNT, head.getRecordId(), before.getRecordId(), after.getRecordId());
                before = head;
            }
            if (!success) {
                this.gcListener.info("TarMK GC #{}: compaction gave up compacting concurrent commits after {} cycles.", GC_COUNT, cycles);
                int forceTimeout = this.gcOptions.getForceTimeout();
                if (forceTimeout > 0) {
                    this.gcListener.info("TarMK GC #{}: trying to force compact remaining commits for {} seconds", GC_COUNT, forceTimeout);
                    ++cycles;
                    success = this.forceCompact(bufferWriter, FileStore.or(cancel, FileStore.timeOut(forceTimeout, TimeUnit.SECONDS)));
                    if (!success) {
                        if (cancel.get().booleanValue()) {
                            this.gcListener.warn("TarMK GC #{}: compaction failed to force compact remaining commits. Compaction was cancelled: {}.", GC_COUNT, cancel);
                        } else {
                            this.gcListener.warn("TarMK GC #{}: compaction failed to force compact remaining commits. Most likely compaction didn't get exclusive access to the store.", GC_COUNT);
                        }
                    }
                }
            }
            if (success) {
                this.gcListener.compacted(GCListener.Status.SUCCESS, newGeneration);
                this.gcListener.info("TarMK GC #{}: compaction succeeded in {} ({} ms), after {} cycles", GC_COUNT, watch, watch.elapsed(TimeUnit.MILLISECONDS), cycles);
                Runnable runnable = new Runnable(){

                    @Override
                    public void run() {
                        try {
                            FileStore.this.fileReaper.add(FileStore.this.cleanupOldGenerations(newGeneration));
                        }
                        catch (IOException e) {
                            FileStore.this.gcListener.error("TarMK GC #" + GC_COUNT + ": cleanup failed", e);
                        }
                    }
                };
                return runnable;
            }
            this.gcListener.compacted(GCListener.Status.FAILURE, newGeneration);
            this.gcListener.info("TarMK GC #{}: compaction failed after {} ({} ms), and {} cycles", GC_COUNT, watch, watch.elapsed(TimeUnit.MILLISECONDS), cycles);
            Runnable runnable = new Runnable(){

                @Override
                public void run() {
                    try {
                        FileStore.this.gcListener.info("TarMK GC #{}: cleaning up after failed compaction", GC_COUNT);
                        FileStore.this.fileReaper.add(FileStore.this.cleanupGeneration(newGeneration));
                    }
                    catch (IOException e) {
                        FileStore.this.gcListener.error("TarMK GC #" + GC_COUNT + ": cleanup failed", e);
                    }
                }
            };
            return runnable;
        }
        catch (InterruptedException e) {
            this.gcListener.error("TarMK GC #" + GC_COUNT + ": compaction interrupted", e);
            Thread.currentThread().interrupt();
            return null;
        }
        catch (Exception e) {
            this.gcListener.error("TarMK GC #" + GC_COUNT + ": compaction encountered an error", e);
            return null;
        }
    }

    private SegmentNodeState compact(SegmentBufferWriter bufferWriter, NodeState head, Supplier<Boolean> cancel) throws IOException {
        if (this.gcOptions.isOffline()) {
            SegmentWriter writer = new SegmentWriter(this, this.segmentReader, this.blobStore, new WriterCacheManager.Default(), bufferWriter, this.binaryReferenceConsumer);
            return new Compactor(this.segmentReader, writer, this.blobStore, cancel, this.gcOptions).compact(EmptyNodeState.EMPTY_NODE, head, EmptyNodeState.EMPTY_NODE);
        }
        return this.segmentWriter.writeNode(head, bufferWriter, cancel);
    }

    private boolean forceCompact(final @Nonnull SegmentBufferWriter bufferWriter, final @Nonnull Supplier<Boolean> cancel) throws InterruptedException {
        return this.revisions.setHead(new Function<RecordId, RecordId>(){

            @Override
            @Nullable
            public RecordId apply(RecordId base) {
                try {
                    long t0 = System.currentTimeMillis();
                    SegmentNodeState after = FileStore.this.compact(bufferWriter, FileStore.this.segmentReader.readNode(base), cancel);
                    if (after == null) {
                        FileStore.this.gcListener.info("TarMK GC #{}: compaction cancelled after {} seconds", GC_COUNT, (System.currentTimeMillis() - t0) / 1000L);
                        return null;
                    }
                    return after.getRecordId();
                }
                catch (IOException e) {
                    FileStore.this.gcListener.error("TarMK GC #{" + GC_COUNT + "}: Error during forced compaction.", e);
                    return null;
                }
            }
        }, TarRevisions.timeout(this.gcOptions.getForceTimeout(), TimeUnit.SECONDS));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Iterable<SegmentId> getSegmentIds() {
        this.fileStoreLock.readLock().lock();
        try {
            ArrayList<SegmentId> ids = Lists.newArrayList();
            if (this.tarWriter != null) {
                for (UUID uuid : this.tarWriter.getUUIDs()) {
                    long msb = uuid.getMostSignificantBits();
                    long lsb = uuid.getLeastSignificantBits();
                    ids.add(this.newSegmentId(msb, lsb));
                }
            }
            for (TarReader reader : this.readers) {
                for (UUID uuid : reader.getUUIDs()) {
                    long msb = uuid.getMostSignificantBits();
                    long lsb = uuid.getLeastSignificantBits();
                    ids.add(this.newSegmentId(msb, lsb));
                }
            }
            ArrayList<SegmentId> arrayList = ids;
            return arrayList;
        }
        finally {
            this.fileStoreLock.readLock().unlock();
        }
    }

    @Nonnull
    public SegmentTracker getTracker() {
        return this.tracker;
    }

    @Nonnull
    public SegmentWriter getWriter() {
        return this.segmentWriter;
    }

    @Nonnull
    public SegmentReader getReader() {
        return this.segmentReader;
    }

    @Nonnull
    public BinaryReferenceConsumer getBinaryReferenceConsumer() {
        return this.binaryReferenceConsumer;
    }

    @Nonnull
    public TarRevisions getRevisions() {
        return this.revisions;
    }

    @Nonnull
    public SegmentNodeState getHead() {
        return this.segmentReader.readHeadState(this.revisions);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        this.shutdown = true;
        this.fileStoreScheduler.close();
        try {
            this.flush();
            this.revisions.close();
            this.fileStoreLock.writeLock().lock();
            try {
                FileStore.closeAndLogOnFail(this.tarWriter);
                List<TarReader> list = this.readers;
                this.readers = Lists.newArrayList();
                for (TarReader reader : list) {
                    FileStore.closeAndLogOnFail(reader);
                }
                if (this.lock != null) {
                    this.lock.release();
                }
                FileStore.closeAndLogOnFail(this.lockFile);
            }
            finally {
                this.fileStoreLock.writeLock().unlock();
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to close the TarMK at " + this.directory, e);
        }
        this.fileReaper.reap();
        System.gc();
        log.info("TarMK closed: {}", (Object)this.directory);
    }

    @Override
    public boolean containsSegment(SegmentId id) {
        long msb = id.getMostSignificantBits();
        long lsb = id.getLeastSignificantBits();
        return this.containsSegment(msb, lsb);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean containsSegment(long msb, long lsb) {
        for (TarReader reader : this.readers) {
            if (!reader.containsEntry(msb, lsb)) continue;
            return true;
        }
        if (this.tarWriter != null) {
            this.fileStoreLock.readLock().lock();
            try {
                if (this.tarWriter.containsEntry(msb, lsb)) {
                    boolean bl = true;
                    return bl;
                }
            }
            finally {
                this.fileStoreLock.readLock().unlock();
            }
        }
        for (TarReader reader : this.readers) {
            if (!reader.containsEntry(msb, lsb)) continue;
            return true;
        }
        return false;
    }

    @Override
    @Nonnull
    public Segment readSegment(final SegmentId id) {
        try {
            return this.segmentCache.getSegment(id, new Callable<Segment>(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 * Enabled aggressive block sorting
                 * Enabled unnecessary exception pruning
                 * Enabled aggressive exception aggregation
                 */
                @Override
                public Segment call() throws Exception {
                    ByteBuffer buffer;
                    Object reader2;
                    long msb = id.getMostSignificantBits();
                    long lsb = id.getLeastSignificantBits();
                    for (Object reader2 : FileStore.this.readers) {
                        try {
                            if (((TarReader)reader2).isClosed()) {
                                log.debug("Skipping closed tar file {}", reader2);
                                continue;
                            }
                            buffer = ((TarReader)reader2).readEntry(msb, lsb);
                            if (buffer == null) continue;
                            return new Segment(FileStore.this, FileStore.this.segmentReader, id, buffer);
                        }
                        catch (IOException e) {
                            log.warn("Failed to read from tar file {}", reader2, (Object)e);
                        }
                    }
                    if (FileStore.this.tarWriter != null) {
                        FileStore.this.fileStoreLock.readLock().lock();
                        try {
                            ByteBuffer buffer2 = FileStore.this.tarWriter.readEntry(msb, lsb);
                            if (buffer2 != null) {
                                reader2 = new Segment(FileStore.this, FileStore.this.segmentReader, id, buffer2);
                                FileStore.this.fileStoreLock.readLock().unlock();
                                return reader2;
                            }
                        }
                        catch (IOException e) {
                            log.warn("Failed to read from tar file {}", (Object)FileStore.this.tarWriter, (Object)e);
                        }
                        catch (Throwable throwable) {
                            throw throwable;
                        }
                    }
                    Iterator iterator = FileStore.this.readers.iterator();
                    while (iterator.hasNext()) {
                        reader2 = (TarReader)iterator.next();
                        try {
                            if (((TarReader)reader2).isClosed()) {
                                log.info("Skipping closed tar file {}", reader2);
                                continue;
                            }
                            buffer = ((TarReader)reader2).readEntry(msb, lsb);
                            if (buffer == null) continue;
                            return new Segment(FileStore.this, FileStore.this.segmentReader, id, buffer);
                        }
                        catch (IOException e) {
                            log.warn("Failed to read from tar file {}", reader2, (Object)e);
                        }
                    }
                    throw new SegmentNotFoundException(id);
                }
            });
        }
        catch (ExecutionException e) {
            throw e.getCause() instanceof SegmentNotFoundException ? (SegmentNotFoundException)e.getCause() : new SegmentNotFoundException(id, (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void writeSegment(SegmentId id, byte[] buffer, int offset, int length) throws IOException {
        Segment segment = null;
        if (id.isDataSegmentId()) {
            ByteBuffer data;
            if (offset > 4096) {
                data = ByteBuffer.allocate(length);
                data.put(buffer, offset, length);
                data.rewind();
            } else {
                data = ByteBuffer.wrap(buffer, offset, length);
            }
            segment = new Segment(this, this.segmentReader, id, data);
        }
        this.fileStoreLock.writeLock().lock();
        try {
            int generation = Segment.getGcGeneration(ByteBuffer.wrap(buffer, offset, length), id.asUUID());
            long size = this.tarWriter.writeEntry(id.getMostSignificantBits(), id.getLeastSignificantBits(), buffer, offset, length, generation);
            if (segment != null) {
                UUID from = segment.getSegmentId().asUUID();
                for (int i = 0; i < segment.getReferencedSegmentIdCount(); ++i) {
                    this.tarWriter.addGraphEdge(from, segment.getReferencedSegmentId(i));
                }
            }
            if (size >= (long)this.maxFileSize) {
                this.newWriter();
            }
        }
        finally {
            this.fileStoreLock.writeLock().unlock();
        }
        if (segment != null) {
            this.segmentCache.putSegment(segment);
        }
    }

    @Override
    @Nonnull
    public SegmentId newSegmentId(long msb, long lsb) {
        return this.tracker.newSegmentId(msb, lsb, this.segmentIdFactory);
    }

    @Override
    @Nonnull
    public SegmentId newBulkSegmentId() {
        return this.tracker.newBulkSegmentId(this.segmentIdFactory);
    }

    @Override
    @Nonnull
    public SegmentId newDataSegmentId() {
        return this.tracker.newDataSegmentId(this.segmentIdFactory);
    }

    private void newWriter() throws IOException {
        TarWriter newWriter = this.tarWriter.createNextGeneration();
        if (newWriter != this.tarWriter) {
            File writeFile = this.tarWriter.getFile();
            ArrayList<TarReader> list = Lists.newArrayListWithCapacity(1 + this.readers.size());
            list.add(TarReader.open(writeFile, this.memoryMapping));
            list.addAll(this.readers);
            this.readers = list;
            this.tarWriter = newWriter;
        }
    }

    @CheckForNull
    public BlobStore getBlobStore() {
        return this.blobStore;
    }

    public Map<String, Set<UUID>> getTarReaderIndex() {
        HashMap<String, Set<UUID>> index = new HashMap<String, Set<UUID>>();
        for (TarReader reader : this.readers) {
            index.put(reader.getFile().getAbsolutePath(), reader.getUUIDs());
        }
        return index;
    }

    public Map<UUID, List<UUID>> getTarGraph(String fileName) throws IOException {
        for (TarReader reader : this.readers) {
            if (!fileName.equals(reader.getFile().getName())) continue;
            HashMap<UUID, List<UUID>> graph = Maps.newHashMap();
            for (UUID uuid : reader.getUUIDs()) {
                graph.put(uuid, null);
            }
            Map<UUID, List<UUID>> g = reader.getGraph(false);
            if (g != null) {
                graph.putAll(g);
            }
            return graph;
        }
        return Collections.emptyMap();
    }

    private void checkDiskSpace() {
        long availableDiskSpace;
        long repositoryDiskSpace = this.size();
        boolean updated = this.gcOptions.isDiskSpaceSufficient(repositoryDiskSpace, availableDiskSpace = this.directory.getFreeSpace());
        boolean previous = this.sufficientDiskSpace.getAndSet(updated);
        if (previous && !updated) {
            log.warn("Available disk space ({}) is too low, current repository size is approx. {}", (Object)IOUtils.humanReadableByteCount(availableDiskSpace), (Object)IOUtils.humanReadableByteCount(repositoryDiskSpace));
        }
        if (updated && !previous) {
            log.info("Available disk space ({}) is sufficient again for repository operations, current repository size is approx. {}", (Object)IOUtils.humanReadableByteCount(availableDiskSpace), (Object)IOUtils.humanReadableByteCount(repositoryDiskSpace));
        }
    }

    private static void closeAndLogOnFail(Closeable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
            }
            catch (IOException ioe) {
                log.error(ioe.getMessage(), ioe);
            }
        }
    }

    private static class CancelCompactionSupplier
    implements Supplier<Boolean> {
        private REASON reason = REASON.UNKNOWN;
        private final FileStore store;

        public CancelCompactionSupplier(FileStore store) {
            this.store = store;
            this.store.gcOptions.setStopCompaction(false);
        }

        @Override
        public Boolean get() {
            if (!this.store.sufficientDiskSpace.get()) {
                this.reason = REASON.DISK_SPACE;
                return true;
            }
            if (this.store.shutdown) {
                this.reason = REASON.SHUTDOWN;
                return true;
            }
            if (this.store.gcOptions.isStopCompaction()) {
                this.reason = REASON.MANUAL;
                return true;
            }
            return false;
        }

        public String toString() {
            switch (this.reason) {
                case DISK_SPACE: {
                    return "Not enough disk space available";
                }
                case SHUTDOWN: {
                    return "FileStore shutdown request received";
                }
                case MANUAL: {
                    return "GC stop request received";
                }
            }
            return "";
        }

        private static enum REASON {
            UNKNOWN,
            DISK_SPACE,
            SHUTDOWN,
            MANUAL;

        }
    }

    public static class ReadOnlyStore
    extends FileStore {
        private RecordId currentHead;

        ReadOnlyStore(FileStoreBuilder builder) throws InvalidFileStoreVersionException, IOException {
            super(builder, true);
        }

        @Override
        ReadOnlyStore bind(@Nonnull TarRevisions revisions) throws IOException {
            revisions.bind(this, new Supplier<RecordId>(){

                @Override
                public RecordId get() {
                    throw new IllegalStateException("Cannot start readonly store from empty journal");
                }
            });
            this.currentHead = revisions.getHead();
            return this;
        }

        public void setRevision(String revision) {
            RecordId newHead = RecordId.fromString(this, revision);
            if (((FileStore)this).revisions.setHead(this.currentHead, newHead, new Revisions.Option[0])) {
                this.currentHead = newHead;
            }
        }

        private static void includeForwardReferences(Iterable<TarReader> readers, Set<UUID> referencedIds) throws IOException {
            HashSet<UUID> fRefs = Sets.newHashSet(referencedIds);
            block0: do {
                for (TarReader reader : readers) {
                    reader.calculateForwardReferences(fRefs);
                    if (!fRefs.isEmpty()) continue;
                    continue block0;
                }
            } while (referencedIds.addAll(fRefs));
        }

        public void traverseSegmentGraph(@Nonnull Set<UUID> roots, @Nonnull SegmentGraph.SegmentGraphVisitor visitor) throws IOException {
            List readers = ((FileStore)this).readers;
            ReadOnlyStore.includeForwardReferences(readers, roots);
            for (TarReader reader : readers) {
                reader.traverseSegmentGraph(Preconditions.checkNotNull(roots), Preconditions.checkNotNull(visitor));
            }
        }

        @Override
        public void writeSegment(SegmentId id, byte[] data, int offset, int length) {
            throw new UnsupportedOperationException("Read Only Store");
        }

        @Override
        public void flush() {
        }

        @Override
        public void cleanup() {
            throw new UnsupportedOperationException("Read Only Store");
        }

        @Override
        public SafeRunnable getGCRunner() {
            throw new UnsupportedOperationException("Read Only Store");
        }

        @Override
        public Runnable compact() {
            throw new UnsupportedOperationException("Read Only Store");
        }

        @Override
        public void gc() {
            throw new UnsupportedOperationException("Read Only Store");
        }
    }
}

