/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import org.apache.jackrabbit.oak.segment.BinaryUtils;
import org.apache.jackrabbit.oak.segment.MutableRecordNumbers;
import org.apache.jackrabbit.oak.segment.MutableSegmentReferences;
import org.apache.jackrabbit.oak.segment.RecordId;
import org.apache.jackrabbit.oak.segment.RecordNumbers;
import org.apache.jackrabbit.oak.segment.RecordType;
import org.apache.jackrabbit.oak.segment.RecordWriters;
import org.apache.jackrabbit.oak.segment.Segment;
import org.apache.jackrabbit.oak.segment.SegmentId;
import org.apache.jackrabbit.oak.segment.SegmentNotFoundException;
import org.apache.jackrabbit.oak.segment.SegmentReader;
import org.apache.jackrabbit.oak.segment.SegmentStore;
import org.apache.jackrabbit.oak.segment.SegmentTracker;
import org.apache.jackrabbit.oak.segment.SegmentVersion;
import org.apache.jackrabbit.oak.segment.WriteOperationHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SegmentBufferWriter
implements WriteOperationHandler {
    private static final Logger LOG = LoggerFactory.getLogger(SegmentBufferWriter.class);
    private static final boolean ENABLE_GENERATION_CHECK = Boolean.getBoolean("enable-generation-check");
    private MutableRecordNumbers recordNumbers = new MutableRecordNumbers();
    private MutableSegmentReferences segmentReferences = new MutableSegmentReferences();
    @Nonnull
    private final SegmentStore store;
    @Nonnull
    private final SegmentTracker tracker;
    @Nonnull
    private final SegmentReader reader;
    @Nonnull
    private final String wid;
    private final int generation;
    private byte[] buffer;
    private Segment segment;
    private int length;
    private int position;
    private Statistics statistics;
    private boolean dirty;

    public SegmentBufferWriter(@Nonnull SegmentStore store, @Nonnull SegmentTracker tracker, @Nonnull SegmentReader reader, @CheckForNull String wid, int generation) {
        this.store = Preconditions.checkNotNull(store);
        this.tracker = Preconditions.checkNotNull(tracker);
        this.reader = Preconditions.checkNotNull(reader);
        this.wid = wid == null ? "w-" + System.identityHashCode(this) : wid;
        this.generation = generation;
        this.statistics = new Statistics();
        this.newSegment();
    }

    @Override
    public RecordId execute(WriteOperationHandler.WriteOperation writeOperation) throws IOException {
        return writeOperation.execute(this);
    }

    int getGeneration() {
        return this.generation;
    }

    private void newSegment() {
        this.buffer = new byte[262144];
        this.buffer[0] = 48;
        this.buffer[1] = 97;
        this.buffer[2] = 75;
        this.buffer[3] = SegmentVersion.asByte(SegmentVersion.LATEST_VERSION);
        this.buffer[4] = 0;
        this.buffer[5] = 0;
        this.buffer[10] = (byte)(this.generation >> 24);
        this.buffer[11] = (byte)(this.generation >> 16);
        this.buffer[12] = (byte)(this.generation >> 8);
        this.buffer[13] = (byte)this.generation;
        this.length = 0;
        this.position = this.buffer.length;
        this.recordNumbers = new MutableRecordNumbers();
        this.segmentReferences = new MutableSegmentReferences();
        String metaInfo = "{\"wid\":\"" + this.wid + '\"' + ",\"sno\":" + this.tracker.getSegmentCount() + ",\"t\":" + System.currentTimeMillis() + "}";
        try {
            this.segment = new Segment(this.store, this.reader, this.buffer, this.recordNumbers, this.segmentReferences, metaInfo);
            this.statistics = new Statistics();
            this.statistics.id = this.segment.getSegmentId();
            byte[] data = metaInfo.getBytes(Charsets.UTF_8);
            RecordWriters.newValueWriter(data.length, data).write(this);
        }
        catch (IOException e) {
            LOG.error("Unable to write meta info to segment {} {}", this.segment.getSegmentId(), metaInfo, e);
        }
        this.dirty = false;
    }

    public void writeByte(byte value) {
        this.position = BinaryUtils.writeByte(this.buffer, this.position, value);
        this.dirty = true;
    }

    public void writeShort(short value) {
        this.position = BinaryUtils.writeShort(this.buffer, this.position, value);
        this.dirty = true;
    }

    public void writeInt(int value) {
        this.position = BinaryUtils.writeInt(this.buffer, this.position, value);
        this.dirty = true;
    }

    public void writeLong(long value) {
        this.position = BinaryUtils.writeLong(this.buffer, this.position, value);
        this.dirty = true;
    }

    public void writeRecordId(RecordId recordId) {
        this.writeRecordId(recordId, true);
    }

    public void writeRecordId(RecordId recordId, boolean reference) {
        Preconditions.checkNotNull(recordId);
        Preconditions.checkState(this.segmentReferences.size() + 1 < 65535, "Segment cannot have more than 0xffff references");
        this.checkGCGeneration(recordId.getSegmentId());
        this.writeShort(SegmentBufferWriter.toShort(this.writeSegmentIdReference(recordId.getSegmentId())));
        this.writeInt(recordId.getRecordNumber());
        ++this.statistics.recordIdCount;
        this.dirty = true;
    }

    private static short toShort(int value) {
        return (short)value;
    }

    private int writeSegmentIdReference(SegmentId id) {
        if (id.equals(this.segment.getSegmentId())) {
            return 0;
        }
        return this.segmentReferences.addOrReference(id);
    }

    private void checkGCGeneration(SegmentId id) {
        if (ENABLE_GENERATION_CHECK) {
            try {
                if (SegmentId.isDataSegmentId(id.getLeastSignificantBits()) && id.getGcGeneration() < this.generation) {
                    LOG.warn("Detected reference from {} to segment {} from a previous gc generation.", SegmentBufferWriter.info(this.segment), SegmentBufferWriter.info(id.getSegment()), new Exception());
                }
            }
            catch (SegmentNotFoundException snfe) {
                LOG.warn("Detected reference from {} to non existing segment {}", SegmentBufferWriter.info(this.segment), id, snfe);
            }
        }
    }

    private static String info(Segment segment) {
        String info = segment.getSegmentId().toString();
        if (SegmentId.isDataSegmentId(segment.getSegmentId().getLeastSignificantBits())) {
            info = info + " " + segment.getSegmentInfo();
        }
        return info;
    }

    public void writeBytes(byte[] data, int offset, int length) {
        System.arraycopy(data, offset, this.buffer, this.position, length);
        this.position += length;
        this.dirty = true;
    }

    @Override
    public void flush() throws IOException {
        if (this.dirty) {
            int referencedSegmentIdCount = this.segmentReferences.size();
            BinaryUtils.writeInt(this.buffer, 14, referencedSegmentIdCount);
            this.statistics.segmentIdCount = referencedSegmentIdCount;
            int recordNumberCount = this.recordNumbers.size();
            BinaryUtils.writeInt(this.buffer, 18, recordNumberCount);
            int totalLength = Segment.align(32 + referencedSegmentIdCount * 16 + recordNumberCount * 9 + this.length, 16);
            if (totalLength > this.buffer.length) {
                throw new IllegalStateException("too much data for a segment");
            }
            this.statistics.size = this.length = totalLength;
            int pos = 32;
            if (pos + this.length <= this.buffer.length) {
                System.arraycopy(this.buffer, 0, this.buffer, this.buffer.length - this.length, pos);
                pos += this.buffer.length - this.length;
            } else {
                this.length = this.buffer.length;
            }
            for (SegmentId segmentId : this.segmentReferences) {
                pos = BinaryUtils.writeLong(this.buffer, pos, segmentId.getMostSignificantBits());
                pos = BinaryUtils.writeLong(this.buffer, pos, segmentId.getLeastSignificantBits());
            }
            for (RecordNumbers.Entry entry : this.recordNumbers) {
                pos = BinaryUtils.writeInt(this.buffer, pos, entry.getRecordNumber());
                pos = BinaryUtils.writeByte(this.buffer, pos, (byte)entry.getType().ordinal());
                pos = BinaryUtils.writeInt(this.buffer, pos, entry.getOffset());
            }
            SegmentId segmentId = this.segment.getSegmentId();
            LOG.debug("Writing data segment: {} ", (Object)this.statistics);
            this.store.writeSegment(segmentId, this.buffer, this.buffer.length - this.length, this.length);
            this.newSegment();
        }
    }

    public RecordId prepare(RecordType type, int size, Collection<RecordId> ids) throws IOException {
        Preconditions.checkArgument(size >= 0);
        Preconditions.checkNotNull(ids);
        int idCount = ids.size();
        int recordSize = Segment.align(size + idCount * 6, 4);
        int recordNumbersCount = this.recordNumbers.size() + 1;
        int referencedIdCount = this.segmentReferences.size() + ids.size();
        int headerSize = 32 + referencedIdCount * 16 + recordNumbersCount * 9;
        int segmentSize = Segment.align(headerSize + recordSize + this.length, 16);
        if (segmentSize > this.buffer.length) {
            HashSet<SegmentId> segmentIds = Sets.newHashSet();
            for (RecordId recordId : ids) {
                SegmentId segmentId = recordId.getSegmentId();
                if (this.segmentReferences.contains(segmentId)) continue;
                segmentIds.add(segmentId);
            }
            referencedIdCount = this.segmentReferences.size() + segmentIds.size();
            headerSize = 32 + referencedIdCount * 16 + recordNumbersCount * 9;
            segmentSize = Segment.align(headerSize + recordSize + this.length, 16);
        }
        if (segmentSize > this.buffer.length) {
            this.flush();
        }
        ++this.statistics.recordCount;
        this.length += recordSize;
        this.position = this.buffer.length - this.length;
        Preconditions.checkState(this.position >= 0);
        int recordNumber = this.recordNumbers.addRecord(type, this.position);
        return new RecordId(this.segment.getSegmentId(), recordNumber);
    }

    private static final class Statistics {
        int segmentIdCount;
        int recordIdCount;
        int recordCount;
        int size;
        SegmentId id;

        private Statistics() {
        }

        public String toString() {
            StringBuilder builder = new StringBuilder();
            builder.append("id=").append(this.id);
            builder.append(",");
            builder.append("size=").append(this.size);
            builder.append(",");
            builder.append("segmentIdCount=").append(this.segmentIdCount);
            builder.append(",");
            builder.append("recordIdCount=").append(this.recordIdCount);
            builder.append(",");
            builder.append("recordCount=").append(this.recordCount);
            return builder.toString();
        }
    }
}

