/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.apache.jackrabbit.oak.segment.MapEntry;
import org.apache.jackrabbit.oak.segment.MapRecord;
import org.apache.jackrabbit.oak.segment.RecordId;
import org.apache.jackrabbit.oak.segment.RecordType;
import org.apache.jackrabbit.oak.segment.SegmentBufferWriter;

final class RecordWriters {
    private RecordWriters() {
    }

    public static RecordWriter newMapLeafWriter(int level, Collection<MapEntry> entries) {
        return new MapLeafWriter(level, entries);
    }

    public static RecordWriter newMapLeafWriter() {
        return new MapLeafWriter();
    }

    public static RecordWriter newMapBranchWriter(int level, int entryCount, int bitmap, List<RecordId> ids) {
        return new MapBranchWriter(level, entryCount, bitmap, ids);
    }

    public static RecordWriter newMapBranchWriter(int bitmap, List<RecordId> ids) {
        return new MapBranchWriter(bitmap, ids);
    }

    public static RecordWriter newListWriter(int count, RecordId lid) {
        return new ListWriter(count, lid);
    }

    public static RecordWriter newListWriter() {
        return new ListWriter();
    }

    public static RecordWriter newListBucketWriter(List<RecordId> ids) {
        return new ListBucketWriter(ids);
    }

    public static RecordWriter newBlockWriter(byte[] bytes, int offset, int length) {
        return new BlockWriter(bytes, offset, length);
    }

    public static RecordWriter newValueWriter(RecordId rid, long len) {
        return new SingleValueWriter(rid, len);
    }

    public static RecordWriter newValueWriter(int length, byte[] data) {
        return new ArrayValueWriter(length, data);
    }

    public static RecordWriter newBlobIdWriter(RecordId rid) {
        return new LargeBlobIdWriter(rid);
    }

    public static RecordWriter newBlobIdWriter(byte[] blobId) {
        return new SmallBlobIdWriter(blobId);
    }

    public static RecordWriter newTemplateWriter(Collection<RecordId> ids, RecordId[] propertyNames, byte[] propertyTypes, int head, RecordId primaryId, List<RecordId> mixinIds, RecordId childNameId, RecordId propNamesId) {
        return new TemplateWriter(ids, propertyNames, propertyTypes, head, primaryId, mixinIds, childNameId, propNamesId);
    }

    public static RecordWriter newNodeStateWriter(RecordId stableId, List<RecordId> ids) {
        return new NodeStateWriter(stableId, ids);
    }

    private static class NodeStateWriter
    extends RecordWriter {
        private final RecordId stableId;

        private NodeStateWriter(RecordId stableId, List<RecordId> ids) {
            super(RecordType.NODE, 6, ids);
            this.stableId = stableId;
        }

        @Override
        protected RecordId writeRecordContent(RecordId id, SegmentBufferWriter writer) {
            if (this.stableId == null) {
                writer.writeRecordId(id, false);
            } else {
                writer.writeRecordId(this.stableId);
            }
            for (RecordId recordId : this.ids) {
                writer.writeRecordId(recordId);
            }
            return id;
        }
    }

    private static class TemplateWriter
    extends RecordWriter {
        private final RecordId[] propertyNames;
        private final byte[] propertyTypes;
        private final int head;
        private final RecordId primaryId;
        private final List<RecordId> mixinIds;
        private final RecordId childNameId;
        private final RecordId propNamesId;

        private TemplateWriter(Collection<RecordId> ids, RecordId[] propertyNames, byte[] propertyTypes, int head, RecordId primaryId, List<RecordId> mixinIds, RecordId childNameId, RecordId propNamesId) {
            super(RecordType.TEMPLATE, 4 + propertyTypes.length, ids);
            this.propertyNames = propertyNames;
            this.propertyTypes = propertyTypes;
            this.head = head;
            this.primaryId = primaryId;
            this.mixinIds = mixinIds;
            this.childNameId = childNameId;
            this.propNamesId = propNamesId;
        }

        @Override
        protected RecordId writeRecordContent(RecordId id, SegmentBufferWriter writer) {
            writer.writeInt(this.head);
            if (this.primaryId != null) {
                writer.writeRecordId(this.primaryId);
            }
            if (this.mixinIds != null) {
                for (RecordId mixinId : this.mixinIds) {
                    writer.writeRecordId(mixinId);
                }
            }
            if (this.childNameId != null) {
                writer.writeRecordId(this.childNameId);
            }
            if (this.propNamesId != null) {
                writer.writeRecordId(this.propNamesId);
            }
            for (int i = 0; i < this.propertyNames.length; ++i) {
                writer.writeByte(this.propertyTypes[i]);
            }
            return id;
        }
    }

    private static class SmallBlobIdWriter
    extends RecordWriter {
        private final byte[] blobId;

        private SmallBlobIdWriter(byte[] blobId) {
            super(RecordType.VALUE, 2 + blobId.length);
            Preconditions.checkArgument(blobId.length < 4096);
            this.blobId = blobId;
        }

        @Override
        protected RecordId writeRecordContent(RecordId id, SegmentBufferWriter writer) {
            int length = this.blobId.length;
            writer.writeShort((short)(length | 0xE000));
            writer.writeBytes(this.blobId, 0, length);
            return id;
        }
    }

    private static class LargeBlobIdWriter
    extends RecordWriter {
        private final RecordId stringRecord;

        private LargeBlobIdWriter(RecordId stringRecord) {
            super(RecordType.VALUE, 1, stringRecord);
            this.stringRecord = stringRecord;
        }

        @Override
        protected RecordId writeRecordContent(RecordId id, SegmentBufferWriter writer) {
            writer.writeByte((byte)-16);
            writer.writeRecordId(this.stringRecord);
            return id;
        }
    }

    private static class ArrayValueWriter
    extends RecordWriter {
        private final int length;
        private final byte[] data;

        private ArrayValueWriter(int length, byte[] data) {
            super(RecordType.VALUE, length + ArrayValueWriter.getSizeDelta(length));
            this.length = length;
            this.data = data;
        }

        private static boolean isSmallSize(int length) {
            return length < 128;
        }

        private static int getSizeDelta(int length) {
            if (ArrayValueWriter.isSmallSize(length)) {
                return 1;
            }
            return 2;
        }

        @Override
        protected RecordId writeRecordContent(RecordId id, SegmentBufferWriter writer) {
            if (ArrayValueWriter.isSmallSize(this.length)) {
                writer.writeByte((byte)this.length);
            } else {
                writer.writeShort((short)(this.length - 128 | 0x8000));
            }
            writer.writeBytes(this.data, 0, this.length);
            return id;
        }
    }

    private static class SingleValueWriter
    extends RecordWriter {
        private final RecordId rid;
        private final long len;

        private SingleValueWriter(RecordId rid, long len) {
            super(RecordType.VALUE, 8, rid);
            this.rid = rid;
            this.len = len;
        }

        @Override
        protected RecordId writeRecordContent(RecordId id, SegmentBufferWriter writer) {
            writer.writeLong(this.len);
            writer.writeRecordId(this.rid);
            return id;
        }
    }

    private static class BlockWriter
    extends RecordWriter {
        private final byte[] bytes;
        private final int offset;

        private BlockWriter(byte[] bytes, int offset, int length) {
            super(RecordType.BLOCK, length);
            this.bytes = bytes;
            this.offset = offset;
        }

        @Override
        protected RecordId writeRecordContent(RecordId id, SegmentBufferWriter writer) {
            writer.writeBytes(this.bytes, this.offset, this.size);
            return id;
        }
    }

    private static class ListBucketWriter
    extends RecordWriter {
        private ListBucketWriter(List<RecordId> ids) {
            super(RecordType.BUCKET, 0, ids);
        }

        @Override
        protected RecordId writeRecordContent(RecordId id, SegmentBufferWriter writer) {
            for (RecordId bucketId : this.ids) {
                writer.writeRecordId(bucketId);
            }
            return id;
        }
    }

    private static class ListWriter
    extends RecordWriter {
        private final int count;
        private final RecordId lid;

        private ListWriter() {
            super(RecordType.LIST, 4);
            this.count = 0;
            this.lid = null;
        }

        private ListWriter(int count, RecordId lid) {
            super(RecordType.LIST, 4, lid);
            this.count = count;
            this.lid = lid;
        }

        @Override
        protected RecordId writeRecordContent(RecordId id, SegmentBufferWriter writer) {
            writer.writeInt(this.count);
            if (this.lid != null) {
                writer.writeRecordId(this.lid);
            }
            return id;
        }
    }

    private static class MapBranchWriter
    extends RecordWriter {
        private final int level;
        private final int entryCount;
        private final int bitmap;

        private MapBranchWriter(int level, int entryCount, int bitmap, List<RecordId> ids) {
            super(RecordType.BRANCH, 8, ids);
            this.level = level;
            this.entryCount = entryCount;
            this.bitmap = bitmap;
        }

        private MapBranchWriter(int bitmap, List<RecordId> ids) {
            this(0, -1, bitmap, ids);
        }

        @Override
        protected RecordId writeRecordContent(RecordId id, SegmentBufferWriter writer) {
            writer.writeInt(this.level << MapRecord.SIZE_BITS | this.entryCount);
            writer.writeInt(this.bitmap);
            for (RecordId mapId : this.ids) {
                writer.writeRecordId(mapId);
            }
            return id;
        }
    }

    private static class MapLeafWriter
    extends RecordWriter {
        private final int level;
        private final Collection<MapEntry> entries;

        private MapLeafWriter() {
            super(RecordType.LEAF, 4);
            this.level = -1;
            this.entries = null;
        }

        private MapLeafWriter(int level, Collection<MapEntry> entries) {
            super(RecordType.LEAF, 4 + entries.size() * 4, MapLeafWriter.extractIds(entries));
            this.level = level;
            this.entries = entries;
        }

        private static List<RecordId> extractIds(Collection<MapEntry> entries) {
            ArrayList<RecordId> ids = Lists.newArrayListWithCapacity(2 * entries.size());
            for (MapEntry entry : entries) {
                ids.add(entry.getKey());
                ids.add(entry.getValue());
            }
            return ids;
        }

        @Override
        protected RecordId writeRecordContent(RecordId id, SegmentBufferWriter writer) {
            if (this.entries != null) {
                int size = this.entries.size();
                writer.writeInt(this.level << MapRecord.SIZE_BITS | size);
                Object[] array = this.entries.toArray(new MapEntry[size]);
                Arrays.sort(array);
                for (Object entry : array) {
                    writer.writeInt(((MapEntry)entry).getHash());
                }
                for (Object entry : array) {
                    writer.writeRecordId(((MapEntry)entry).getKey());
                    writer.writeRecordId(((MapEntry)entry).getValue());
                }
            } else {
                writer.writeInt(0);
            }
            return id;
        }
    }

    public static abstract class RecordWriter {
        private final RecordType type;
        protected final int size;
        protected final Collection<RecordId> ids;

        protected RecordWriter(RecordType type, int size, Collection<RecordId> ids) {
            this.type = type;
            this.size = size;
            this.ids = ids;
        }

        protected RecordWriter(RecordType type, int size, RecordId id) {
            this(type, size, Collections.singleton(id));
        }

        protected RecordWriter(RecordType type, int size) {
            this(type, size, Collections.emptyList());
        }

        public final RecordId write(SegmentBufferWriter writer) throws IOException {
            RecordId id = writer.prepare(this.type, this.size, this.ids);
            return this.writeRecordContent(id, writer);
        }

        protected abstract RecordId writeRecordContent(RecordId var1, SegmentBufferWriter var2);
    }
}

