/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.document.mongo;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicates;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.UncheckedExecutionException;
import com.mongodb.BasicDBObject;
import com.mongodb.BulkWriteError;
import com.mongodb.BulkWriteException;
import com.mongodb.BulkWriteOperation;
import com.mongodb.BulkWriteResult;
import com.mongodb.BulkWriteUpsert;
import com.mongodb.CommandResult;
import com.mongodb.DB;
import com.mongodb.DBCollection;
import com.mongodb.DBCursor;
import com.mongodb.DBObject;
import com.mongodb.MongoClientURI;
import com.mongodb.MongoException;
import com.mongodb.QueryBuilder;
import com.mongodb.ReadPreference;
import com.mongodb.WriteConcern;
import com.mongodb.WriteResult;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.jackrabbit.oak.cache.CacheStats;
import org.apache.jackrabbit.oak.cache.CacheValue;
import org.apache.jackrabbit.oak.plugins.document.Collection;
import org.apache.jackrabbit.oak.plugins.document.Document;
import org.apache.jackrabbit.oak.plugins.document.DocumentMK;
import org.apache.jackrabbit.oak.plugins.document.DocumentStore;
import org.apache.jackrabbit.oak.plugins.document.DocumentStoreException;
import org.apache.jackrabbit.oak.plugins.document.DocumentStoreStatsCollector;
import org.apache.jackrabbit.oak.plugins.document.NodeDocument;
import org.apache.jackrabbit.oak.plugins.document.Revision;
import org.apache.jackrabbit.oak.plugins.document.RevisionListener;
import org.apache.jackrabbit.oak.plugins.document.RevisionVector;
import org.apache.jackrabbit.oak.plugins.document.StableRevisionComparator;
import org.apache.jackrabbit.oak.plugins.document.UpdateOp;
import org.apache.jackrabbit.oak.plugins.document.UpdateUtils;
import org.apache.jackrabbit.oak.plugins.document.cache.CacheChangesTracker;
import org.apache.jackrabbit.oak.plugins.document.cache.CacheInvalidationStats;
import org.apache.jackrabbit.oak.plugins.document.cache.ModificationStamp;
import org.apache.jackrabbit.oak.plugins.document.cache.NodeDocumentCache;
import org.apache.jackrabbit.oak.plugins.document.locks.NodeDocumentLocks;
import org.apache.jackrabbit.oak.plugins.document.locks.StripedNodeDocumentLocks;
import org.apache.jackrabbit.oak.plugins.document.mongo.MongoUtils;
import org.apache.jackrabbit.oak.plugins.document.mongo.RevisionEntry;
import org.apache.jackrabbit.oak.plugins.document.mongo.replica.LocalChanges;
import org.apache.jackrabbit.oak.plugins.document.mongo.replica.ReplicaSetInfo;
import org.apache.jackrabbit.oak.plugins.document.util.Utils;
import org.apache.jackrabbit.oak.stats.Clock;
import org.apache.jackrabbit.oak.util.PerfLogger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MongoDocumentStore
implements DocumentStore,
RevisionListener {
    private static final Logger LOG = LoggerFactory.getLogger(MongoDocumentStore.class);
    private static final PerfLogger PERFLOG = new PerfLogger(LoggerFactory.getLogger(MongoDocumentStore.class.getName() + ".perf"));
    private static final DBObject BY_ID_ASC = new BasicDBObject("_id", (Object)1);
    public static final int IN_CLAUSE_BATCH_SIZE = 500;
    private static final ImmutableSet<String> SERVER_DETAIL_FIELD_NAMES = ((ImmutableSet.Builder)ImmutableSet.builder().add(new String[]{"host", "process", "connections", "repl", "storageEngine", "mem"})).build();
    private final DBCollection nodes;
    private final DBCollection clusterNodes;
    private final DBCollection settings;
    private final DBCollection journal;
    private final DB db;
    private final NodeDocumentCache nodesCache;
    private final NodeDocumentLocks nodeLocks;
    private Clock clock = Clock.SIMPLE;
    private ReplicaSetInfo replicaInfo;
    private RevisionVector mostRecentAccessedRevisions;
    final LocalChanges localChanges;
    private final long maxReplicationLagMillis;
    private final long maxDeltaForModTimeIdxSecs = Long.getLong("oak.mongo.maxDeltaForModTimeIdxSecs", 60L);
    private final boolean disableIndexHint = Boolean.getBoolean("oak.mongo.disableIndexHint");
    private final long maxQueryTimeMS = Long.getLong("oak.mongo.maxQueryTimeMS", TimeUnit.MINUTES.toMillis(1L));
    private long estimationPullFrequencyMS = Long.getLong("oak.mongo.estimationPullFrequencyMS", TimeUnit.SECONDS.toMillis(5L));
    private boolean fallbackSecondaryStrategy = Boolean.getBoolean("oak.mongo.fallbackSecondaryStrategy");
    private int bulkSize = Integer.getInteger("oak.mongo.bulkSize", 30);
    private int bulkRetries = Integer.getInteger("oak.mongo.bulkRetries", 0);
    private String lastReadWriteMode;
    private final Map<String, String> metadata;
    private DocumentStoreStatsCollector stats;
    private boolean hasModifiedIdCompoundIndex = true;

    public MongoDocumentStore(DB db, DocumentMK.Builder builder) {
        CommandResult serverStatus = db.command("serverStatus");
        String version = MongoDocumentStore.checkVersion(db, serverStatus);
        this.metadata = ImmutableMap.builder().put("type", "mongo").put("version", version).build();
        this.db = db;
        this.stats = builder.getDocumentStoreStatsCollector();
        this.nodes = db.getCollection(Collection.NODES.toString());
        this.clusterNodes = db.getCollection(Collection.CLUSTER_NODES.toString());
        this.settings = db.getCollection(Collection.SETTINGS.toString());
        this.journal = db.getCollection(Collection.JOURNAL.toString());
        this.maxReplicationLagMillis = builder.getMaxReplicationLagMillis();
        if (this.fallbackSecondaryStrategy) {
            this.replicaInfo = null;
            this.localChanges = null;
        } else {
            this.replicaInfo = new ReplicaSetInfo(this.clock, db, builder.getMongoUri(), this.estimationPullFrequencyMS, this.maxReplicationLagMillis, builder.getExecutor());
            Thread replicaInfoThread = new Thread((Runnable)this.replicaInfo, "MongoDocumentStore replica set info provider (" + builder.getClusterId() + ")");
            replicaInfoThread.setDaemon(true);
            replicaInfoThread.start();
            this.localChanges = new LocalChanges(builder.getClusterId());
            this.replicaInfo.addListener(this.localChanges);
        }
        if (this.nodes.count() == 0L) {
            MongoUtils.createIndex(this.nodes, new String[]{"_modified", "_id"}, new boolean[]{true, true}, false, false);
        } else if (!MongoUtils.hasIndex(this.nodes, "_modified", "_id")) {
            this.hasModifiedIdCompoundIndex = false;
            LOG.warn("Detected an upgrade from Oak version <= 1.2. For optimal performance it is recommended to create a compound index for the 'nodes' collection on {_modified:1, _id:1}.");
        }
        MongoUtils.createIndex(this.nodes, "_bin", true, false, true);
        MongoUtils.createIndex(this.nodes, "_deletedOnce", true, false, true);
        MongoUtils.createIndex(this.nodes, "_sdType", true, false, true);
        MongoUtils.createIndex(this.journal, "_modified", true, false, false);
        this.nodeLocks = new StripedNodeDocumentLocks();
        this.nodesCache = builder.buildNodeDocumentCache(this, this.nodeLocks);
        LOG.info("Connected to MongoDB {} with maxReplicationLagMillis {}, maxDeltaForModTimeIdxSecs {}, disableIndexHint {}, {}, serverStatus {}", version, this.maxReplicationLagMillis, this.maxDeltaForModTimeIdxSecs, this.disableIndexHint, db.getWriteConcern(), MongoDocumentStore.serverDetails(serverStatus));
    }

    @Nonnull
    private static String checkVersion(DB db, CommandResult serverStatus) {
        Matcher m;
        String version = serverStatus.getString("version");
        if (version == null) {
            version = db.command("buildInfo").getString("version");
        }
        if (!(m = Pattern.compile("^(\\d+)\\.(\\d+)\\..*").matcher(version)).matches()) {
            throw new IllegalArgumentException("Malformed MongoDB version: " + version);
        }
        int major = Integer.parseInt(m.group(1));
        int minor = Integer.parseInt(m.group(2));
        if (major > 2) {
            return version;
        }
        if (minor < 6) {
            String msg = "MongoDB version 2.6.0 or higher required. Currently connected to a MongoDB with version: " + version;
            throw new RuntimeException(msg);
        }
        return version;
    }

    @Nonnull
    private static String serverDetails(CommandResult serverStatus) {
        HashMap<String, Object> details = Maps.newHashMap();
        for (String key : SERVER_DETAIL_FIELD_NAMES) {
            Object value = serverStatus.get(key);
            if (value == null) continue;
            details.put(key, value);
        }
        return ((Object)details).toString();
    }

    public void finalize() throws Throwable {
        super.finalize();
        this.dispose();
    }

    @Override
    public CacheInvalidationStats invalidateCache() {
        InvalidationResult result = new InvalidationResult();
        for (CacheValue key : this.nodesCache.keys()) {
            ++result.invalidationCount;
            this.invalidateCache(Collection.NODES, key.toString());
        }
        return result;
    }

    @Override
    public CacheInvalidationStats invalidateCache(Iterable<String> keys) {
        LOG.debug("invalidateCache: start");
        InvalidationResult result = new InvalidationResult();
        int size = 0;
        Iterator<String> it = keys.iterator();
        while (it.hasNext()) {
            ArrayList<String> ids = new ArrayList<String>(500);
            while (it.hasNext() && ids.size() < 500) {
                String id = it.next();
                if (this.nodesCache.getIfPresent(id) == null) continue;
                ids.add(id);
            }
            size += ids.size();
            if (LOG.isTraceEnabled()) {
                LOG.trace("invalidateCache: batch size: {} of total so far {}", (Object)ids.size(), (Object)size);
            }
            Map<String, ModificationStamp> modStamps = this.getModStamps(ids);
            ++result.queryCount;
            int invalidated = this.nodesCache.invalidateOutdated(modStamps);
            for (String id : Iterables.filter(ids, Predicates.not(Predicates.in(modStamps.keySet())))) {
                this.nodesCache.invalidate(id);
                ++invalidated;
            }
            result.cacheEntriesProcessedCount += ids.size();
            result.invalidationCount += invalidated;
            result.upToDateCount += ids.size() - invalidated;
        }
        result.cacheSize = size;
        LOG.trace("invalidateCache: end. total: {}", (Object)size);
        return result;
    }

    @Override
    public <T extends Document> void invalidateCache(Collection<T> collection, String key) {
        if (collection == Collection.NODES) {
            this.nodesCache.invalidate(key);
        }
    }

    @Override
    public <T extends Document> T find(Collection<T> collection, String key) {
        long start = PERFLOG.start();
        T result = this.find(collection, key, true, -1);
        PERFLOG.end(start, 1L, "find: preferCached=true, key={}", (Object)key);
        return result;
    }

    @Override
    public <T extends Document> T find(Collection<T> collection, String key, int maxCacheAge) {
        long start = PERFLOG.start();
        T result = this.find(collection, key, false, maxCacheAge);
        PERFLOG.end(start, 1L, "find: preferCached=false, key={}", (Object)key);
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private <T extends Document> T find(Collection<T> collection, String key, boolean preferCached, int maxCacheAge) {
        Throwable t;
        NodeDocument doc;
        if (collection != Collection.NODES) {
            return this.findUncachedWithRetry(collection, key, DocumentReadPreference.PRIMARY, 2);
        }
        if ((maxCacheAge > 0 || preferCached) && (doc = this.nodesCache.getIfPresent(key)) != null && (preferCached || this.getTime() - doc.getCreated() < (long)maxCacheAge)) {
            this.stats.doneFindCached(collection, key);
            if (doc != NodeDocument.NULL) return (T)doc;
            return null;
        }
        try {
            Lock lock = this.nodeLocks.acquire(key);
            try {
                if ((maxCacheAge > 0 || preferCached) && (doc = this.nodesCache.getIfPresent(key)) != null && (preferCached || this.getTime() - doc.getCreated() < (long)maxCacheAge)) {
                    this.stats.doneFindCached(collection, key);
                    if (doc == NodeDocument.NULL) {
                        T t2 = null;
                        return t2;
                    }
                    NodeDocument nodeDocument = doc;
                    return (T)nodeDocument;
                }
                final NodeDocument d = (NodeDocument)this.findUncachedWithRetry(collection, key, this.getReadPreference(maxCacheAge), 2);
                this.invalidateCache(collection, key);
                doc = this.nodesCache.get(key, new Callable<NodeDocument>(){

                    @Override
                    public NodeDocument call() throws Exception {
                        return d == null ? NodeDocument.NULL : d;
                    }
                });
            }
            finally {
                lock.unlock();
            }
            if (doc != NodeDocument.NULL) return (T)doc;
            return null;
        }
        catch (UncheckedExecutionException e) {
            t = e.getCause();
            throw new DocumentStoreException("Failed to load document with " + key, t);
        }
        catch (ExecutionException e) {
            t = e.getCause();
            throw new DocumentStoreException("Failed to load document with " + key, t);
        }
        catch (RuntimeException e) {
            t = e;
        }
        throw new DocumentStoreException("Failed to load document with " + key, t);
    }

    @CheckForNull
    private <T extends Document> T findUncachedWithRetry(Collection<T> collection, String key, DocumentReadPreference docReadPref, int retries) {
        Preconditions.checkArgument(retries >= 0, "retries must not be negative");
        if (key.equals("0:/")) {
            LOG.trace("root node");
        }
        int numAttempts = retries + 1;
        MongoException ex = null;
        for (int i = 0; i < numAttempts; ++i) {
            if (i > 0) {
                LOG.warn("Retrying read of " + key);
            }
            try {
                return this.findUncached(collection, key, docReadPref);
            }
            catch (MongoException e) {
                ex = e;
                continue;
            }
        }
        if (ex != null) {
            throw ex;
        }
        throw new IllegalStateException();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @CheckForNull
    protected <T extends Document> T findUncached(Collection<T> collection, String key, DocumentReadPreference docReadPref) {
        MongoDocumentStore.log("findUncached", new Object[]{key, docReadPref});
        DBCollection dbCollection = this.getDBCollection(collection);
        Stopwatch watch = this.startWatch();
        boolean isSlaveOk = false;
        boolean docFound = true;
        try {
            DBObject obj;
            ReadPreference readPreference = this.getMongoReadPreference(collection, null, key, docReadPref);
            if (readPreference.isSlaveOk()) {
                LOG.trace("Routing call to secondary for fetching [{}]", (Object)key);
                isSlaveOk = true;
            }
            if ((obj = dbCollection.findOne(MongoDocumentStore.getByKeyQuery(key).get(), null, null, readPreference)) == null) {
                docFound = false;
                T t = null;
                return t;
            }
            T doc = this.convertFromDBObject(collection, obj);
            if (doc != null) {
                ((Document)doc).seal();
            }
            T t = doc;
            return t;
        }
        finally {
            this.stats.doneFindUncached(watch.elapsed(TimeUnit.NANOSECONDS), collection, key, docFound, isSlaveOk);
        }
    }

    @Override
    @Nonnull
    public <T extends Document> List<T> query(Collection<T> collection, String fromKey, String toKey, int limit) {
        return this.query(collection, fromKey, toKey, null, 0L, limit);
    }

    @Override
    @Nonnull
    public <T extends Document> List<T> query(Collection<T> collection, String fromKey, String toKey, String indexedProperty, long startValue, int limit) {
        return this.queryInternal(collection, fromKey, toKey, indexedProperty, startValue, limit, this.maxQueryTimeMS);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nonnull
    <T extends Document> List<T> queryInternal(Collection<T> collection, String fromKey, String toKey, String indexedProperty, long startValue, int limit, long maxQueryTime) {
        MongoDocumentStore.log("query", fromKey, toKey, indexedProperty, startValue, limit);
        DBCollection dbCollection = this.getDBCollection(collection);
        QueryBuilder queryBuilder = QueryBuilder.start("_id");
        queryBuilder.greaterThan(fromKey);
        queryBuilder.lessThan(toKey);
        BasicDBObject hint = new BasicDBObject("_id", (Object)1);
        if (indexedProperty != null) {
            if ("_deletedOnce".equals(indexedProperty)) {
                if (startValue != 1L) {
                    throw new DocumentStoreException("unsupported value for property _deletedOnce");
                }
                queryBuilder.and(indexedProperty);
                queryBuilder.is(true);
            } else {
                queryBuilder.and(indexedProperty);
                queryBuilder.greaterThanEquals(startValue);
                if ("_modified".equals(indexedProperty) && this.canUseModifiedTimeIdx(startValue)) {
                    hint = new BasicDBObject("_modified", (Object)-1);
                }
            }
        }
        DBObject query = queryBuilder.get();
        String parentId = Utils.getParentIdFromLowerLimit(fromKey);
        long lockTime = -1L;
        Stopwatch watch = this.startWatch();
        boolean isSlaveOk = false;
        int resultSize = 0;
        CacheChangesTracker cacheChangesTracker = null;
        if (parentId != null && collection == Collection.NODES) {
            cacheChangesTracker = this.nodesCache.registerTracker(fromKey, toKey);
        }
        try {
            ArrayList<NodeDocument> list;
            ReadPreference readPreference;
            DBCursor cursor = dbCollection.find(query).sort(BY_ID_ASC);
            if (!this.disableIndexHint && !this.hasModifiedIdCompoundIndex) {
                cursor.hint(hint);
            }
            if (maxQueryTime > 0L) {
                cursor.maxTime(maxQueryTime, TimeUnit.MILLISECONDS);
            }
            if ((readPreference = this.getMongoReadPreference(collection, parentId, null, this.getDefaultReadPreference(collection))).isSlaveOk()) {
                isSlaveOk = true;
                LOG.trace("Routing call to secondary for fetching children from [{}] to [{}]", (Object)fromKey, (Object)toKey);
            }
            cursor.setReadPreference(readPreference);
            try {
                list = new ArrayList<NodeDocument>();
                for (int i = 0; i < limit && cursor.hasNext(); ++i) {
                    DBObject o = cursor.next();
                    T doc = this.convertFromDBObject(collection, o);
                    list.add((NodeDocument)doc);
                }
                resultSize = list.size();
            }
            finally {
                cursor.close();
            }
            if (cacheChangesTracker != null) {
                this.nodesCache.putNonConflictingDocs(cacheChangesTracker, list);
            }
            ArrayList<NodeDocument> arrayList = list;
            if (cacheChangesTracker != null) {
                cacheChangesTracker.close();
            }
            this.stats.doneQuery(watch.elapsed(TimeUnit.NANOSECONDS), collection, fromKey, toKey, indexedProperty != null, resultSize, lockTime, isSlaveOk);
            return arrayList;
        }
        catch (Throwable throwable) {
            if (cacheChangesTracker != null) {
                cacheChangesTracker.close();
            }
            this.stats.doneQuery(watch.elapsed(TimeUnit.NANOSECONDS), collection, fromKey, toKey, indexedProperty != null, resultSize, lockTime, isSlaveOk);
            throw throwable;
        }
    }

    boolean canUseModifiedTimeIdx(long modifiedTimeInSecs) {
        if (this.maxDeltaForModTimeIdxSecs < 0L) {
            return false;
        }
        return NodeDocument.getModifiedInSecs(this.getTime()) - modifiedTimeInSecs <= this.maxDeltaForModTimeIdxSecs;
    }

    @Override
    public <T extends Document> void remove(Collection<T> collection, String key) {
        MongoDocumentStore.log("remove", key);
        DBCollection dbCollection = this.getDBCollection(collection);
        long start = PERFLOG.start();
        try {
            dbCollection.remove(MongoDocumentStore.getByKeyQuery(key).get());
        }
        catch (Exception e) {
            throw DocumentStoreException.convert(e, "Remove failed for " + key);
        }
        finally {
            this.invalidateCache(collection, key);
            PERFLOG.end(start, 1L, "remove key={}", (Object)key);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public <T extends Document> void remove(Collection<T> collection, List<String> keys) {
        MongoDocumentStore.log("remove", keys);
        DBCollection dbCollection = this.getDBCollection(collection);
        long start = PERFLOG.start();
        try {
            for (List<String> keyBatch : Lists.partition(keys, 500)) {
                DBObject query = QueryBuilder.start("_id").in(keyBatch).get();
                try {
                    dbCollection.remove(query);
                }
                catch (Exception e) {
                    throw DocumentStoreException.convert(e, "Remove failed for " + keyBatch);
                }
                finally {
                    if (collection != Collection.NODES) continue;
                    for (String key : keyBatch) {
                        this.invalidateCache(collection, key);
                    }
                }
            }
        }
        finally {
            PERFLOG.end(start, 1L, "remove keys={}", (Object)keys);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public <T extends Document> int remove(Collection<T> collection, Map<String, Map<UpdateOp.Key, UpdateOp.Condition>> toRemove) {
        MongoDocumentStore.log("remove", toRemove);
        int num = 0;
        DBCollection dbCollection = this.getDBCollection(collection);
        long start = PERFLOG.start();
        try {
            ArrayList<String> batchIds = Lists.newArrayList();
            ArrayList<DBObject> batch = Lists.newArrayList();
            Iterator<Map.Entry<String, Map<UpdateOp.Key, UpdateOp.Condition>>> it = toRemove.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry<String, Map<UpdateOp.Key, UpdateOp.Condition>> entry = it.next();
                QueryBuilder query = MongoDocumentStore.createQueryForUpdate(entry.getKey(), entry.getValue());
                batchIds.add(entry.getKey());
                batch.add(query.get());
                if (it.hasNext() && batch.size() != 500) continue;
                BasicDBObject q = new BasicDBObject();
                q.put("$or", batch);
                try {
                    num += dbCollection.remove(q).getN();
                }
                catch (Exception e) {
                    throw DocumentStoreException.convert(e, "Remove failed for " + batch);
                }
                finally {
                    if (collection == Collection.NODES) {
                        this.invalidateCache(batchIds);
                    }
                }
                batchIds.clear();
                batch.clear();
            }
        }
        finally {
            PERFLOG.end(start, 1L, "remove keys={}", (Object)toRemove);
        }
        return num;
    }

    @CheckForNull
    private <T extends Document> T findAndModify(Collection<T> collection, UpdateOp updateOp, boolean upsert, boolean checkConditions) {
        DBCollection dbCollection = this.getDBCollection(collection);
        updateOp = updateOp.copy();
        DBObject update = MongoDocumentStore.createUpdate(updateOp, false);
        Lock lock = null;
        if (collection == Collection.NODES) {
            lock = this.nodeLocks.acquire(updateOp.getId());
        }
        Stopwatch watch = this.startWatch();
        boolean newEntry = false;
        try {
            DBObject oldNode;
            QueryBuilder query;
            Long modCount = null;
            NodeDocument cachedDoc = null;
            if (collection == Collection.NODES && (cachedDoc = this.nodesCache.getIfPresent(updateOp.getId())) != null) {
                modCount = cachedDoc.getModCount();
            }
            if (modCount != null) {
                query = MongoDocumentStore.createQueryForUpdate(updateOp.getId(), updateOp.getConditions());
                query.and("_modCount").is(modCount);
                WriteResult result = dbCollection.update(query.get(), update);
                if (result.getN() > 0) {
                    NodeDocument newDoc;
                    if (collection == Collection.NODES) {
                        newDoc = this.applyChanges(collection, cachedDoc, updateOp);
                        this.nodesCache.put(newDoc);
                    }
                    newDoc = cachedDoc;
                    return (T)newDoc;
                }
            }
            if ((oldNode = dbCollection.findAndModify((query = MongoDocumentStore.createQueryForUpdate(updateOp.getId(), updateOp.getConditions())).get(), null, null, false, update, false, upsert)) == null) {
                newEntry = true;
            }
            if (checkConditions && oldNode == null) {
                T newDoc = null;
                return newDoc;
            }
            T oldDoc = this.convertFromDBObject(collection, oldNode);
            if (oldDoc != null) {
                if (collection == Collection.NODES) {
                    NodeDocument newDoc = (NodeDocument)this.applyChanges(collection, oldDoc, updateOp);
                    this.nodesCache.put(newDoc);
                    this.updateLocalChanges(newDoc);
                }
                ((Document)oldDoc).seal();
            } else if (upsert && collection == Collection.NODES) {
                NodeDocument doc = (NodeDocument)collection.newDocument(this);
                UpdateUtils.applyChanges(doc, updateOp);
                this.nodesCache.putIfAbsent(doc);
                this.updateLocalChanges(doc);
            }
            T t = oldDoc;
            return t;
        }
        catch (Exception e) {
            throw this.handleException(e, collection, updateOp.getId());
        }
        finally {
            if (lock != null) {
                lock.unlock();
            }
            this.stats.doneFindAndModify(watch.elapsed(TimeUnit.NANOSECONDS), collection, updateOp.getId(), newEntry, true, 0);
        }
    }

    @Override
    @CheckForNull
    public <T extends Document> T createOrUpdate(Collection<T> collection, UpdateOp update) throws DocumentStoreException {
        MongoDocumentStore.log("createOrUpdate", update);
        UpdateUtils.assertUnconditional(update);
        T doc = this.findAndModify(collection, update, true, false);
        MongoDocumentStore.log("createOrUpdate returns ", doc);
        return doc;
    }

    @Override
    @CheckForNull
    public <T extends Document> List<T> createOrUpdate(Collection<T> collection, List<UpdateOp> updateOps) {
        MongoDocumentStore.log("createOrUpdate", updateOps);
        LinkedHashMap<String, UpdateOp> operationsToCover = new LinkedHashMap<String, UpdateOp>();
        ArrayList<UpdateOp> duplicates = new ArrayList<UpdateOp>();
        LinkedHashMap<UpdateOp, T> results = new LinkedHashMap<UpdateOp, T>();
        Stopwatch watch = this.startWatch();
        try {
            for (UpdateOp updateOp : updateOps) {
                UpdateUtils.assertUnconditional(updateOp);
                UpdateOp clone = updateOp.copy();
                if (operationsToCover.containsKey(updateOp.getId())) {
                    duplicates.add(clone);
                } else {
                    operationsToCover.put(updateOp.getId(), clone);
                }
                results.put(clone, null);
            }
            HashMap<String, NodeDocument> oldDocs = new HashMap<String, NodeDocument>();
            if (collection == Collection.NODES) {
                oldDocs.putAll(this.getCachedNodes(operationsToCover.keySet()));
            }
            for (int i = 0; i <= this.bulkRetries && operationsToCover.size() > 2; ++i) {
                for (List<UpdateOp> list : Lists.partition(Lists.newArrayList(operationsToCover.values()), this.bulkSize)) {
                    Map<UpdateOp, T> successfulUpdates = this.bulkUpdate(collection, list, oldDocs);
                    results.putAll(successfulUpdates);
                    operationsToCover.values().removeAll(successfulUpdates.keySet());
                }
            }
            Iterator it = Iterators.concat(operationsToCover.values().iterator(), duplicates.iterator());
            while (it.hasNext()) {
                UpdateOp op = (UpdateOp)it.next();
                it.remove();
                T t = this.createOrUpdate(collection, op);
                if (t == null) continue;
                results.put(op, t);
            }
        }
        catch (MongoException e) {
            try {
                throw this.handleException((Exception)e, collection, Iterables.transform(updateOps, new Function<UpdateOp, String>(){

                    @Override
                    public String apply(UpdateOp input) {
                        return input.getId();
                    }
                }));
            }
            catch (Throwable throwable) {
                this.stats.doneCreateOrUpdate(watch.elapsed(TimeUnit.NANOSECONDS), collection, Lists.transform(updateOps, new Function<UpdateOp, String>(){

                    @Override
                    public String apply(UpdateOp input) {
                        return input.getId();
                    }
                }));
                throw throwable;
            }
        }
        this.stats.doneCreateOrUpdate(watch.elapsed(TimeUnit.NANOSECONDS), collection, Lists.transform(updateOps, new /* invalid duplicate definition of identical inner class */));
        ArrayList resultList = new ArrayList(results.values());
        MongoDocumentStore.log("createOrUpdate returns", resultList);
        return resultList;
    }

    private Map<String, NodeDocument> getCachedNodes(Set<String> keys) {
        HashMap<String, NodeDocument> nodes = new HashMap<String, NodeDocument>();
        for (String key : keys) {
            NodeDocument cached = this.nodesCache.getIfPresent(key);
            if (cached == null) continue;
            nodes.put(key, cached);
        }
        return nodes;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T extends Document> Map<UpdateOp, T> bulkUpdate(Collection<T> collection, List<UpdateOp> updateOperations, Map<String, T> oldDocs) {
        Map<String, UpdateOp> bulkOperations = MongoDocumentStore.createMap(updateOperations);
        Sets.SetView<String> lackingDocs = Sets.difference(bulkOperations.keySet(), oldDocs.keySet());
        oldDocs.putAll(this.findDocuments(collection, lackingDocs));
        CacheChangesTracker tracker = null;
        if (collection == Collection.NODES) {
            tracker = this.nodesCache.registerTracker(bulkOperations.keySet());
        }
        try {
            BulkUpdateResult bulkResult = this.sendBulkUpdate(collection, bulkOperations.values(), oldDocs);
            if (collection == Collection.NODES) {
                ArrayList<NodeDocument> docsToCache = new ArrayList<NodeDocument>();
                for (UpdateOp updateOp : Maps.filterKeys(bulkOperations, Predicates.in(bulkResult.upserts)).values()) {
                    NodeDocument doc = Collection.NODES.newDocument(this);
                    UpdateUtils.applyChanges(doc, updateOp);
                    docsToCache.add(doc);
                }
                for (String string : Sets.difference(bulkOperations.keySet(), bulkResult.failedUpdates)) {
                    Document oldDoc = (Document)oldDocs.get(string);
                    if (oldDoc == null || oldDoc == NodeDocument.NULL) continue;
                    NodeDocument newDoc = (NodeDocument)this.applyChanges(collection, oldDoc, bulkOperations.get(string));
                    docsToCache.add(newDoc);
                }
                for (NodeDocument nodeDocument : docsToCache) {
                    this.updateLocalChanges(nodeDocument);
                }
                this.nodesCache.putNonConflictingDocs(tracker, docsToCache);
            }
            oldDocs.keySet().removeAll(bulkResult.failedUpdates);
            HashMap result = new HashMap();
            for (Map.Entry entry : bulkOperations.entrySet()) {
                if (bulkResult.failedUpdates.contains(entry.getKey())) continue;
                if (bulkResult.upserts.contains(entry.getKey())) {
                    result.put(entry.getValue(), null);
                    continue;
                }
                result.put(entry.getValue(), oldDocs.get(entry.getKey()));
            }
            HashMap hashMap = result;
            return hashMap;
        }
        finally {
            if (tracker != null) {
                tracker.close();
            }
        }
    }

    private static Map<String, UpdateOp> createMap(List<UpdateOp> updateOps) {
        return Maps.uniqueIndex(updateOps, new Function<UpdateOp, String>(){

            @Override
            public String apply(UpdateOp input) {
                return input.getId();
            }
        });
    }

    private <T extends Document> Map<String, T> findDocuments(Collection<T> collection, Set<String> keys) {
        HashMap<String, T> docs = new HashMap<String, T>();
        if (!keys.isEmpty()) {
            DBObject[] conditions = new DBObject[keys.size()];
            int i = 0;
            for (String key : keys) {
                conditions[i++] = MongoDocumentStore.getByKeyQuery(key).get();
            }
            QueryBuilder builder = new QueryBuilder();
            builder.or(conditions);
            DBCursor cursor = this.getDBCollection(collection).find(builder.get());
            while (cursor.hasNext()) {
                T foundDoc = this.convertFromDBObject(collection, cursor.next());
                docs.put(((Document)foundDoc).getId(), foundDoc);
            }
        }
        return docs;
    }

    private <T extends Document> BulkUpdateResult sendBulkUpdate(Collection<T> collection, java.util.Collection<UpdateOp> updateOps, Map<String, T> oldDocs) {
        BulkWriteResult bulkResult;
        DBCollection dbCollection = this.getDBCollection(collection);
        BulkWriteOperation bulk = dbCollection.initializeUnorderedBulkOperation();
        String[] bulkIds = new String[updateOps.size()];
        int i = 0;
        for (UpdateOp updateOp : updateOps) {
            DBObject update;
            String id = updateOp.getId();
            QueryBuilder query = MongoDocumentStore.createQueryForUpdate(id, updateOp.getConditions());
            Document oldDoc = (Document)oldDocs.get(id);
            if (oldDoc == null || oldDoc == NodeDocument.NULL) {
                query.and("_modCount").exists(false);
                update = MongoDocumentStore.createUpdate(updateOp, true);
            } else {
                query.and("_modCount").is(oldDoc.getModCount());
                update = MongoDocumentStore.createUpdate(updateOp, false);
            }
            bulk.find(query.get()).upsert().updateOne(update);
            bulkIds[i++] = id;
        }
        HashSet<String> failedUpdates = new HashSet<String>();
        HashSet<String> upserts = new HashSet<String>();
        try {
            bulkResult = bulk.execute();
        }
        catch (BulkWriteException e) {
            bulkResult = e.getWriteResult();
            for (BulkWriteError err : e.getWriteErrors()) {
                failedUpdates.add(bulkIds[err.getIndex()]);
            }
        }
        for (BulkWriteUpsert upsert : bulkResult.getUpserts()) {
            upserts.add(bulkIds[upsert.getIndex()]);
        }
        return new BulkUpdateResult(failedUpdates, upserts);
    }

    @Override
    public <T extends Document> T findAndUpdate(Collection<T> collection, UpdateOp update) throws DocumentStoreException {
        MongoDocumentStore.log("findAndUpdate", update);
        T doc = this.findAndModify(collection, update, false, true);
        MongoDocumentStore.log("findAndUpdate returns ", doc);
        return doc;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public <T extends Document> boolean create(Collection<T> collection, List<UpdateOp> updateOps) {
        MongoDocumentStore.log("create", updateOps);
        ArrayList<T> docs = new ArrayList<T>();
        DBObject[] inserts = new DBObject[updateOps.size()];
        ArrayList<String> ids = Lists.newArrayListWithCapacity(updateOps.size());
        for (int i = 0; i < updateOps.size(); ++i) {
            inserts[i] = new BasicDBObject();
            UpdateOp update = updateOps.get(i);
            UpdateUtils.assertUnconditional(update);
            T target = collection.newDocument(this);
            UpdateUtils.applyChanges(target, update);
            docs.add(target);
            ids.add(updateOps.get(i).getId());
            for (Map.Entry entry : update.getChanges().entrySet()) {
                UpdateOp.Key k = (UpdateOp.Key)entry.getKey();
                UpdateOp.Operation op = (UpdateOp.Operation)entry.getValue();
                switch (op.type) {
                    case SET: 
                    case MAX: 
                    case INCREMENT: {
                        inserts[i].put(k.toString(), op.value);
                        break;
                    }
                    case SET_MAP_ENTRY: {
                        Revision r = k.getRevision();
                        if (r == null) {
                            throw new IllegalStateException("SET_MAP_ENTRY must not have null revision");
                        }
                        DBObject value = (DBObject)inserts[i].get(k.getName());
                        if (value == null) {
                            value = new RevisionEntry(r, op.value);
                            inserts[i].put(k.getName(), value);
                            break;
                        }
                        if (value.keySet().size() == 1) {
                            String key = value.keySet().iterator().next();
                            Object val = value.get(key);
                            value = new BasicDBObject(key, val);
                            value.put(r.toString(), op.value);
                            inserts[i].put(k.getName(), value);
                            break;
                        }
                        value.put(r.toString(), op.value);
                        break;
                    }
                }
            }
            if (inserts[i].containsField("_modCount")) continue;
            inserts[i].put("_modCount", 1L);
            ((Document)target).put("_modCount", 1L);
        }
        DBCollection dbCollection = this.getDBCollection(collection);
        Stopwatch watch = this.startWatch();
        boolean insertSuccess = false;
        try {
            dbCollection.insert(inserts);
            if (collection == Collection.NODES) {
                for (Document document : docs) {
                    this.nodesCache.putIfAbsent((NodeDocument)document);
                    this.updateLocalChanges((NodeDocument)document);
                }
            }
            insertSuccess = true;
            boolean i$ = true;
            return i$;
        }
        catch (MongoException e) {
            boolean bl = false;
            return bl;
        }
        finally {
            this.stats.doneCreate(watch.elapsed(TimeUnit.NANOSECONDS), collection, ids, insertSuccess);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public <T extends Document> void update(Collection<T> collection, List<String> keys, UpdateOp updateOp) {
        MongoDocumentStore.log("update", keys, updateOp);
        UpdateUtils.assertUnconditional(updateOp);
        DBCollection dbCollection = this.getDBCollection(collection);
        QueryBuilder query = QueryBuilder.start("_id").in(keys);
        updateOp = updateOp.copy();
        DBObject update = MongoDocumentStore.createUpdate(updateOp, false);
        Stopwatch watch = this.startWatch();
        try {
            Map<String, NodeDocument> cachedDocs = Collections.emptyMap();
            if (collection == Collection.NODES) {
                cachedDocs = Maps.newHashMap();
                for (String key : keys) {
                    cachedDocs.put(key, this.nodesCache.getIfPresent(key));
                }
            }
            try {
                dbCollection.update(query.get(), update, false, true);
                if (collection == Collection.NODES) {
                    Map<String, ModificationStamp> modCounts = this.getModStamps(Maps.filterValues(cachedDocs, Predicates.notNull()).keySet());
                    for (Map.Entry entry : cachedDocs.entrySet()) {
                        Lock lock = this.nodeLocks.acquire((String)entry.getKey());
                        try {
                            ModificationStamp postUpdateModStamp = modCounts.get(entry.getKey());
                            if (postUpdateModStamp != null && entry.getValue() != null && entry.getValue() != NodeDocument.NULL && Long.valueOf(postUpdateModStamp.modCount - 1L).equals(((NodeDocument)entry.getValue()).getModCount())) {
                                NodeDocument newDoc = (NodeDocument)this.applyChanges(Collection.NODES, (Document)entry.getValue(), updateOp.shallowCopy((String)entry.getKey()));
                                this.nodesCache.replaceCachedDocument((NodeDocument)entry.getValue(), newDoc);
                                continue;
                            }
                            this.nodesCache.invalidate((String)entry.getKey());
                        }
                        finally {
                            lock.unlock();
                        }
                    }
                }
            }
            catch (MongoException e) {
                throw this.handleException((Exception)e, collection, keys);
            }
        }
        finally {
            this.stats.doneUpdate(watch.elapsed(TimeUnit.NANOSECONDS), collection, keys.size());
        }
    }

    @Nonnull
    private Map<String, ModificationStamp> getModStamps(Iterable<String> keys) throws MongoException {
        QueryBuilder query = QueryBuilder.start("_id").in(keys);
        BasicDBObject fields = new BasicDBObject("_id", (Object)1);
        fields.put("_modCount", 1);
        fields.put("_modified", 1);
        DBCursor cursor = this.nodes.find(query.get(), fields);
        cursor.setReadPreference(ReadPreference.primary());
        HashMap<String, ModificationStamp> modCounts = Maps.newHashMap();
        for (DBObject obj : cursor) {
            Long modified;
            String id = (String)obj.get("_id");
            Long modCount = Utils.asLong((Number)obj.get("_modCount"));
            if (modCount == null) {
                modCount = -1L;
            }
            if ((modified = Utils.asLong((Number)obj.get("_modified"))) == null) {
                modified = -1L;
            }
            modCounts.put(id, new ModificationStamp(modCount, modified));
        }
        return modCounts;
    }

    DocumentReadPreference getReadPreference(int maxCacheAge) {
        long lag;
        long l = lag = this.fallbackSecondaryStrategy ? this.maxReplicationLagMillis : this.replicaInfo.getLag();
        if (maxCacheAge >= 0 && (long)maxCacheAge < lag) {
            return DocumentReadPreference.PRIMARY;
        }
        if (maxCacheAge == Integer.MAX_VALUE) {
            return DocumentReadPreference.PREFER_SECONDARY;
        }
        return DocumentReadPreference.PREFER_SECONDARY_IF_OLD_ENOUGH;
    }

    DocumentReadPreference getDefaultReadPreference(Collection col) {
        return col == Collection.NODES ? DocumentReadPreference.PREFER_SECONDARY_IF_OLD_ENOUGH : DocumentReadPreference.PRIMARY;
    }

    <T extends Document> ReadPreference getMongoReadPreference(@Nonnull Collection<T> collection, @Nullable String parentId, @Nullable String documentId, @Nonnull DocumentReadPreference preference) {
        switch (preference) {
            case PRIMARY: {
                return ReadPreference.primary();
            }
            case PREFER_PRIMARY: {
                return ReadPreference.primaryPreferred();
            }
            case PREFER_SECONDARY: {
                return this.getConfiguredReadPreference(collection);
            }
            case PREFER_SECONDARY_IF_OLD_ENOUGH: {
                boolean secondarySafe;
                if (collection != Collection.NODES) {
                    return ReadPreference.primary();
                }
                if (this.fallbackSecondaryStrategy) {
                    NodeDocument cachedDoc;
                    long replicationSafeLimit = this.getTime() - this.maxReplicationLagMillis;
                    secondarySafe = parentId == null ? false : (cachedDoc = this.nodesCache.getIfPresent(parentId)) != null && !cachedDoc.hasBeenModifiedSince(replicationSafeLimit);
                } else {
                    secondarySafe = true;
                    secondarySafe &= collection == Collection.NODES;
                    secondarySafe &= documentId == null || !this.localChanges.mayContain(documentId);
                    secondarySafe &= parentId == null || !this.localChanges.mayContainChildrenOf(parentId);
                    secondarySafe &= this.mostRecentAccessedRevisions == null || this.replicaInfo.isMoreRecentThan(this.mostRecentAccessedRevisions);
                }
                ReadPreference readPreference = secondarySafe ? this.getConfiguredReadPreference(collection) : ReadPreference.primary();
                return readPreference;
            }
        }
        throw new IllegalArgumentException("Unsupported usage " + (Object)((Object)preference));
    }

    ReadPreference getConfiguredReadPreference(Collection collection) {
        return this.getDBCollection(collection).getReadPreference();
    }

    @CheckForNull
    protected <T extends Document> T convertFromDBObject(@Nonnull Collection<T> collection, @Nullable DBObject n) {
        Document copy = null;
        if (n != null) {
            copy = (Document)collection.newDocument(this);
            for (String key : n.keySet()) {
                Object o = n.get(key);
                if (o instanceof String) {
                    copy.put(key, o);
                    continue;
                }
                if (o instanceof Number && ("_modified".equals(key) || "_modCount".equals(key))) {
                    copy.put(key, Utils.asLong((Number)o));
                    continue;
                }
                if (o instanceof Long) {
                    copy.put(key, o);
                    continue;
                }
                if (o instanceof Integer) {
                    copy.put(key, o);
                    continue;
                }
                if (o instanceof Boolean) {
                    copy.put(key, o);
                    continue;
                }
                if (!(o instanceof BasicDBObject)) continue;
                copy.put(key, this.convertMongoMap((BasicDBObject)o));
            }
        }
        return (T)copy;
    }

    @Nonnull
    private Map<Revision, Object> convertMongoMap(@Nonnull BasicDBObject obj) {
        TreeMap<Revision, Object> map = new TreeMap<Revision, Object>(StableRevisionComparator.REVERSE);
        for (Map.Entry entry : obj.entrySet()) {
            map.put(Revision.fromString((String)entry.getKey()), entry.getValue());
        }
        return map;
    }

    <T extends Document> DBCollection getDBCollection(Collection<T> collection) {
        if (collection == Collection.NODES) {
            return this.nodes;
        }
        if (collection == Collection.CLUSTER_NODES) {
            return this.clusterNodes;
        }
        if (collection == Collection.SETTINGS) {
            return this.settings;
        }
        if (collection == Collection.JOURNAL) {
            return this.journal;
        }
        throw new IllegalArgumentException("Unknown collection: " + collection.toString());
    }

    private static QueryBuilder getByKeyQuery(String key) {
        return QueryBuilder.start("_id").is(key);
    }

    @Override
    public void dispose() {
        if (this.replicaInfo != null) {
            this.replicaInfo.stop();
        }
        this.nodes.getDB().getMongo().close();
        try {
            this.nodesCache.close();
        }
        catch (IOException e) {
            LOG.warn("Error occurred while closing nodes cache", e);
        }
    }

    @Override
    public Iterable<CacheStats> getCacheStats() {
        return this.nodesCache.getCacheStats();
    }

    @Override
    public Map<String, String> getMetadata() {
        return this.metadata;
    }

    long getMaxDeltaForModTimeIdxSecs() {
        return this.maxDeltaForModTimeIdxSecs;
    }

    boolean getDisableIndexHint() {
        return this.disableIndexHint;
    }

    private static void log(String message, Object ... args) {
        if (LOG.isDebugEnabled()) {
            String argList = Arrays.toString(args);
            if (argList.length() > 10000) {
                argList = argList.length() + ": " + argList;
            }
            LOG.debug(message + argList);
        }
    }

    @Override
    public <T extends Document> T getIfCached(Collection<T> collection, String key) {
        if (collection != Collection.NODES) {
            return null;
        }
        NodeDocument doc = this.nodesCache.getIfPresent(key);
        if (doc == NodeDocument.NULL) {
            doc = null;
        }
        return (T)doc;
    }

    @Nonnull
    private static QueryBuilder createQueryForUpdate(String key, Map<UpdateOp.Key, UpdateOp.Condition> conditions) {
        QueryBuilder query = MongoDocumentStore.getByKeyQuery(key);
        for (Map.Entry<UpdateOp.Key, UpdateOp.Condition> entry : conditions.entrySet()) {
            UpdateOp.Key k = entry.getKey();
            UpdateOp.Condition c = entry.getValue();
            switch (c.type) {
                case EXISTS: {
                    query.and(k.toString()).exists(c.value);
                    break;
                }
                case EQUALS: {
                    query.and(k.toString()).is(c.value);
                    break;
                }
                case NOTEQUALS: {
                    query.and(k.toString()).notEquals(c.value);
                }
            }
        }
        return query;
    }

    @Nonnull
    private static DBObject createUpdate(UpdateOp updateOp, boolean includeId) {
        BasicDBObject setUpdates = new BasicDBObject();
        BasicDBObject maxUpdates = new BasicDBObject();
        BasicDBObject incUpdates = new BasicDBObject();
        BasicDBObject unsetUpdates = new BasicDBObject();
        updateOp.increment("_modCount", 1L);
        for (Map.Entry<UpdateOp.Key, UpdateOp.Operation> entry : updateOp.getChanges().entrySet()) {
            UpdateOp.Key k = entry.getKey();
            if (!includeId && k.getName().equals("_id")) continue;
            UpdateOp.Operation op = entry.getValue();
            switch (op.type) {
                case SET: 
                case SET_MAP_ENTRY: {
                    setUpdates.append(k.toString(), op.value);
                    break;
                }
                case MAX: {
                    maxUpdates.append(k.toString(), op.value);
                    break;
                }
                case INCREMENT: {
                    incUpdates.append(k.toString(), op.value);
                    break;
                }
                case REMOVE_MAP_ENTRY: {
                    unsetUpdates.append(k.toString(), "1");
                }
            }
        }
        BasicDBObject update = new BasicDBObject();
        if (!setUpdates.isEmpty()) {
            update.append("$set", setUpdates);
        }
        if (!maxUpdates.isEmpty()) {
            update.append("$max", maxUpdates);
        }
        if (!incUpdates.isEmpty()) {
            update.append("$inc", incUpdates);
        }
        if (!unsetUpdates.isEmpty()) {
            update.append("$unset", unsetUpdates);
        }
        return update;
    }

    @Nonnull
    private <T extends Document> T applyChanges(Collection<T> collection, T oldDoc, UpdateOp update) {
        T doc = collection.newDocument(this);
        oldDoc.deepCopy((Document)doc);
        UpdateUtils.applyChanges(doc, update);
        ((Document)doc).seal();
        return doc;
    }

    private Stopwatch startWatch() {
        return Stopwatch.createStarted();
    }

    @Override
    public void setReadWriteMode(String readWriteMode) {
        if (readWriteMode == null || readWriteMode.equals(this.lastReadWriteMode)) {
            return;
        }
        this.lastReadWriteMode = readWriteMode;
        try {
            WriteConcern writeConcern;
            MongoClientURI uri;
            ReadPreference readPref;
            String rwModeUri = readWriteMode;
            if (!readWriteMode.startsWith("mongodb://")) {
                rwModeUri = String.format("mongodb://localhost/?%s", readWriteMode);
            }
            if (!(readPref = (uri = new MongoClientURI(rwModeUri)).getOptions().getReadPreference()).equals(this.nodes.getReadPreference())) {
                this.nodes.setReadPreference(readPref);
                LOG.info("Using ReadPreference {} ", (Object)readPref);
            }
            if (!(writeConcern = uri.getOptions().getWriteConcern()).equals(this.nodes.getWriteConcern())) {
                this.nodes.setWriteConcern(writeConcern);
                LOG.info("Using WriteConcern " + writeConcern);
            }
        }
        catch (Exception e) {
            LOG.error("Error setting readWriteMode " + readWriteMode, e);
        }
    }

    private long getTime() {
        return this.clock.getTime();
    }

    void setClock(Clock clock) {
        this.clock = clock;
    }

    NodeDocumentCache getNodeDocumentCache() {
        return this.nodesCache;
    }

    public void setStatsCollector(DocumentStoreStatsCollector stats) {
        this.stats = stats;
    }

    void setReplicaInfo(ReplicaSetInfo replicaInfo) {
        if (this.replicaInfo != null) {
            this.replicaInfo.stop();
        }
        this.replicaInfo = replicaInfo;
        this.replicaInfo.addListener(this.localChanges);
    }

    @Override
    public long determineServerTimeDifferenceMillis() {
        long start = System.currentTimeMillis();
        CommandResult serverStatus = this.db.command("serverStatus");
        if (serverStatus == null) {
            LOG.warn("determineServerTimeDifferenceMillis: db.serverStatus returned null - cannot determine time difference - assuming 0ms.");
            return 0L;
        }
        Date serverLocalTime = serverStatus.getDate("localTime");
        if (serverLocalTime == null) {
            LOG.warn("determineServerTimeDifferenceMillis: db.serverStatus.localTime returned null - cannot determine time difference - assuming 0ms. (Result details: server exception=" + serverStatus.getException() + ", server error message=" + serverStatus.getErrorMessage() + ")", serverStatus.getException());
            return 0L;
        }
        long end = System.currentTimeMillis();
        long midPoint = (start + end) / 2L;
        long serverLocalTimeMillis = serverLocalTime.getTime();
        long diff = midPoint - serverLocalTimeMillis;
        return diff;
    }

    @Override
    public synchronized void updateAccessedRevision(RevisionVector revisions) {
        RevisionVector previousValue = this.mostRecentAccessedRevisions;
        this.mostRecentAccessedRevisions = this.mostRecentAccessedRevisions == null ? revisions : this.mostRecentAccessedRevisions.pmax(revisions);
        if (LOG.isDebugEnabled() && !this.mostRecentAccessedRevisions.equals(previousValue)) {
            LOG.debug("Most recent accessed revisions: {}", (Object)this.mostRecentAccessedRevisions);
        }
    }

    private void updateLocalChanges(NodeDocument doc) {
        if (this.localChanges != null) {
            this.localChanges.add(doc.getId(), Revision.getCurrentTimestamp());
        }
    }

    private <T extends Document> DocumentStoreException handleException(Exception ex, Collection<T> collection, Iterable<String> ids) {
        if (collection == Collection.NODES) {
            for (String id : ids) {
                this.invalidateCache(collection, id);
            }
        }
        return DocumentStoreException.convert(ex);
    }

    private <T extends Document> DocumentStoreException handleException(Exception ex, Collection<T> collection, String id) {
        return this.handleException(ex, collection, Collections.singleton(id));
    }

    private static class InvalidationResult
    implements CacheInvalidationStats {
        int invalidationCount;
        int upToDateCount;
        int cacheSize;
        int queryCount;
        int cacheEntriesProcessedCount;

        private InvalidationResult() {
        }

        public String toString() {
            return "InvalidationResult{invalidationCount=" + this.invalidationCount + ", upToDateCount=" + this.upToDateCount + ", cacheSize=" + this.cacheSize + ", queryCount=" + this.queryCount + ", cacheEntriesProcessedCount=" + this.cacheEntriesProcessedCount + '}';
        }

        @Override
        public String summaryReport() {
            return this.toString();
        }
    }

    private static class BulkUpdateResult {
        private final Set<String> failedUpdates;
        private final Set<String> upserts;

        private BulkUpdateResult(Set<String> failedUpdates, Set<String> upserts) {
            this.failedUpdates = failedUpdates;
            this.upserts = upserts;
        }
    }

    static enum DocumentReadPreference {
        PRIMARY,
        PREFER_PRIMARY,
        PREFER_SECONDARY,
        PREFER_SECONDARY_IF_OLD_ENOUGH;

    }
}

