/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.document;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.SortedMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.CheckForNull;
import org.apache.jackrabbit.oak.commons.PathUtils;
import org.apache.jackrabbit.oak.plugins.document.ClusterNodeInfo;
import org.apache.jackrabbit.oak.plugins.document.ClusterNodeInfoDocument;
import org.apache.jackrabbit.oak.plugins.document.Collection;
import org.apache.jackrabbit.oak.plugins.document.DocumentNodeStore;
import org.apache.jackrabbit.oak.plugins.document.DocumentStore;
import org.apache.jackrabbit.oak.plugins.document.DocumentStoreException;
import org.apache.jackrabbit.oak.plugins.document.JournalEntry;
import org.apache.jackrabbit.oak.plugins.document.MissingLastRevSeeker;
import org.apache.jackrabbit.oak.plugins.document.NodeDocument;
import org.apache.jackrabbit.oak.plugins.document.Revision;
import org.apache.jackrabbit.oak.plugins.document.UnsavedModifications;
import org.apache.jackrabbit.oak.plugins.document.util.MapFactory;
import org.apache.jackrabbit.oak.plugins.document.util.Utils;
import org.apache.jackrabbit.oak.stats.Clock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LastRevRecoveryAgent {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final DocumentNodeStore nodeStore;
    private final MissingLastRevSeeker missingLastRevUtil;

    public LastRevRecoveryAgent(DocumentNodeStore nodeStore, MissingLastRevSeeker seeker) {
        this.nodeStore = nodeStore;
        this.missingLastRevUtil = seeker;
    }

    public LastRevRecoveryAgent(DocumentNodeStore nodeStore) {
        this(nodeStore, new MissingLastRevSeeker(nodeStore.getDocumentStore(), nodeStore.getClock()));
    }

    public int recover(int clusterId, long waitUntil) throws DocumentStoreException {
        ClusterNodeInfoDocument nodeInfo = this.missingLastRevUtil.getClusterNodeInfo(clusterId);
        long leaseTime = ClusterNodeInfo.DEFAULT_LEASE_DURATION_MILLIS;
        long asyncDelay = this.nodeStore.getAsyncDelay();
        if (nodeInfo != null && this.missingLastRevUtil.isRecoveryNeeded(nodeInfo)) {
            String reason;
            long startTime;
            long leaseEnd = nodeInfo.getLeaseEndTime();
            NodeDocument root = this.missingLastRevUtil.getRoot();
            Revision lastRev = root.getLastRev().get(clusterId);
            if (lastRev != null) {
                startTime = lastRev.getTimestamp();
                reason = "lastRev: " + lastRev.toString();
            } else {
                startTime = leaseEnd - leaseTime - asyncDelay;
                reason = String.format("no lastRev for root, using timestamp based on leaseEnd %d - leaseTime %d - asyncDelay %d", leaseEnd, leaseTime, asyncDelay);
            }
            return this.recoverCandidates(nodeInfo, startTime, waitUntil, reason);
        }
        this.log.debug("No recovery needed for clusterId {}", (Object)clusterId);
        return 0;
    }

    public int recover(int clusterId) {
        return this.recover(clusterId, 0L);
    }

    public int recover(Iterator<NodeDocument> suspects, int clusterId) {
        return this.recover(suspects, clusterId, false);
    }

    public int recover(Iterator<NodeDocument> suspects, int clusterId, boolean dryRun) {
        UnsavedModifications unsaved = new UnsavedModifications();
        UnsavedModifications unsavedParents = new UnsavedModifications();
        ConcurrentMap<String, Revision> knownLastRevOrModification = MapFactory.getInstance().create();
        final DocumentStore docStore = this.nodeStore.getDocumentStore();
        final JournalEntry changes = Collection.JOURNAL.newDocument(docStore);
        long count = 0L;
        while (suspects.hasNext()) {
            NodeDocument doc = suspects.next();
            if (++count % 100000L == 0L) {
                this.log.info("Scanned {} suspects so far...", (Object)count);
            }
            Revision currentLastRev = doc.getLastRev().get(clusterId);
            Revision lastModifiedRev = this.determineLastModification(doc, clusterId);
            Revision lastRevForParents = Utils.max(lastModifiedRev, currentLastRev);
            if (lastRevForParents != null) {
                knownLastRevOrModification.put(doc.getPath(), lastRevForParents);
            }
            if (lastRevForParents == null) continue;
            String path = doc.getPath();
            changes.modified(path);
            while (!PathUtils.denotesRoot(path)) {
                path = PathUtils.getParentPath(path);
                unsavedParents.put(path, lastRevForParents);
            }
        }
        for (String parentPath : unsavedParents.getPaths()) {
            Revision calcLastRev = unsavedParents.get(parentPath);
            Revision knownLastRev = (Revision)knownLastRevOrModification.get(parentPath);
            if (knownLastRev == null) {
                String id = Utils.getIdFromPath(parentPath);
                NodeDocument doc = docStore.find(Collection.NODES, id);
                if (doc != null) {
                    Revision lastRev = doc.getLastRev().get(clusterId);
                    Revision lastMod = this.determineLastModification(doc, clusterId);
                    knownLastRev = Utils.max(lastRev, lastMod);
                } else {
                    this.log.warn("Unable to find document: {}", (Object)id);
                    continue;
                }
            }
            if (knownLastRev != null && calcLastRev.compareRevisionTime(knownLastRev) <= 0) continue;
            unsaved.put(parentPath, calcLastRev);
        }
        final Revision lastRootRev = unsaved.get("/");
        int size = unsaved.getPaths().size();
        String updates = unsaved.toString();
        if (dryRun) {
            this.log.info("Dry run of lastRev recovery identified [{}] documents for cluster node [{}]: {}", size, clusterId, updates);
        } else {
            unsaved.persist(this.nodeStore, new UnsavedModifications.Snapshot(){

                @Override
                public void acquiring(Revision mostRecent) {
                    if (lastRootRev == null) {
                        return;
                    }
                    String id = JournalEntry.asId(lastRootRev);
                    JournalEntry existingEntry = docStore.find(Collection.JOURNAL, id);
                    if (existingEntry != null) {
                        return;
                    }
                    docStore.create(Collection.JOURNAL, Collections.singletonList(changes.asUpdateOp(lastRootRev)));
                }
            }, new ReentrantLock());
            this.log.info("Updated lastRev of [{}] documents while performing lastRev recovery for cluster node [{}]: {}", size, clusterId, updates);
        }
        return size;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int recoverCandidates(ClusterNodeInfoDocument nodeInfo, long startTime, long waitUntil, String info) {
        ClusterNodeInfoDocument infoDoc = nodeInfo;
        int clusterId = infoDoc.getClusterId();
        while (!this.missingLastRevUtil.acquireRecoveryLock(clusterId, this.nodeStore.getClusterId())) {
            Clock clock = this.nodeStore.getClock();
            long remaining = waitUntil - clock.getTime();
            if (remaining < 0L) {
                return -1;
            }
            this.log.info("Last revision recovery already being performed by cluster node {}. Waiting at most until {} for recovery to finish ({} seconds remaining).", infoDoc.getRecoveryBy(), Utils.timestampToString(waitUntil), remaining / 1000L);
            long time = Math.min(waitUntil, clock.getTime() + 5000L);
            try {
                clock.waitUntil(time);
            }
            catch (InterruptedException e) {
                Thread.interrupted();
                String msg = "Interrupted while waiting for _lastRev recovery to finish.";
                throw new DocumentStoreException(msg, e);
            }
            infoDoc = this.missingLastRevUtil.getClusterNodeInfo(clusterId);
            if (this.missingLastRevUtil.isRecoveryNeeded(infoDoc)) continue;
            return 0;
        }
        boolean success = false;
        try {
            int n;
            this.log.info("Recovering candidates modified after: [{}] for clusterId [{}] [{}]", Utils.timestampToString(startTime), clusterId, info);
            Iterable<NodeDocument> suspects = this.missingLastRevUtil.getCandidates(startTime);
            try {
                this.log.info("Performing Last Revision Recovery for clusterNodeId {}", (Object)clusterId);
                int num = this.recover(suspects.iterator(), clusterId);
                success = true;
                n = num;
            }
            catch (Throwable throwable) {
                Utils.closeIfCloseable(suspects);
                throw throwable;
            }
            Utils.closeIfCloseable(suspects);
            return n;
        }
        finally {
            this.missingLastRevUtil.releaseRecoveryLock(clusterId, success);
            this.nodeStore.signalClusterStateChange();
        }
    }

    @CheckForNull
    private Revision determineLastModification(NodeDocument doc, int clusterId) {
        ClusterPredicate cp = new ClusterPredicate(clusterId);
        Revision lastModified = null;
        block0: for (String property : Sets.filter(doc.keySet(), Utils.PROPERTY_OR_DELETED)) {
            SortedMap<Revision, String> valueMap = doc.getLocalMap(property);
            for (Map.Entry<Revision, String> entry : Maps.filterKeys(valueMap, cp).entrySet()) {
                Revision rev = entry.getKey();
                if (!doc.isCommitted(rev)) continue;
                lastModified = Utils.max(lastModified, doc.getCommitRevision(rev));
                continue block0;
            }
        }
        return lastModified;
    }

    public boolean isRecoveryNeeded() {
        return this.missingLastRevUtil.isRecoveryNeeded();
    }

    public void performRecoveryIfNeeded() {
        if (this.isRecoveryNeeded()) {
            Iterable<Integer> clusterIds = this.getRecoveryCandidateNodes();
            this.log.info("ClusterNodeId [{}] starting Last Revision Recovery for clusterNodeId(s) {}", (Object)this.nodeStore.getClusterId(), (Object)clusterIds);
            for (int clusterId : clusterIds) {
                if (this.recover(clusterId) != -1) continue;
                this.log.info("Last Revision Recovery for cluster node {} ongoing by other cluster node.", (Object)clusterId);
            }
        }
    }

    public Iterable<Integer> getRecoveryCandidateNodes() {
        return Iterables.transform(Iterables.filter(this.missingLastRevUtil.getAllClusters(), new Predicate<ClusterNodeInfoDocument>(){

            @Override
            public boolean apply(ClusterNodeInfoDocument input) {
                return LastRevRecoveryAgent.this.missingLastRevUtil.isRecoveryNeeded(input);
            }
        }), new Function<ClusterNodeInfoDocument, Integer>(){

            @Override
            public Integer apply(ClusterNodeInfoDocument input) {
                return input.getClusterId();
            }
        });
    }

    private static class ClusterPredicate
    implements Predicate<Revision> {
        private final int clusterId;

        private ClusterPredicate(int clusterId) {
            this.clusterId = clusterId;
        }

        @Override
        public boolean apply(Revision input) {
            return this.clusterId == input.getClusterId();
        }
    }
}

