package com.finconsgroup.itserr.marketplace.userprofile.bs.component;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.userprofile.bs.bean.UserProfileApplicationEvent;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.UserProfileMessagingAdditionalDataDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.mapper.UserProfileMapper;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.EventProducer;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.ResourceProducer;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import com.finconsgroup.itserr.messaging.dto.MessagingEventUserDto;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationListener;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

/**
 * The implementation for ApplicationListener to listen to User Profile application events
 * and produce relevant messages.
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class MessagingUserProfileApplicationEventListener implements ApplicationListener<UserProfileApplicationEvent> {

    private final EventProducer eventProducer;
    private final ResourceProducer resourceProducer;
    private final UserProfileMapper userProfileMapper;

    @Override
    public void onApplicationEvent(@NonNull UserProfileApplicationEvent event) {
        MessagingEventDto<UserProfileMessagingAdditionalDataDto> messagingEventDto =
                userProfileMapper.toMessagingEventDto(event.getUserProfileDto());

        MessagingEventUserDto user = MessagingEventUserDto.builder()
                .id(JwtTokenHolder.getUserId().orElse(null))
                .name(JwtTokenHolder.getName().orElse(null))
                .username(JwtTokenHolder.getPreferredUsername().orElse(null))
                .build();
        messagingEventDto.setUser(user);

        updateEventForType(event, messagingEventDto);
        publishEventForType(event, messagingEventDto);
    }

    private void updateEventForType(@NonNull UserProfileApplicationEvent event,
                                    @NonNull MessagingEventDto<UserProfileMessagingAdditionalDataDto> messagingEventDto) {

        OutputUserProfileDto userProfileDto = event.getUserProfileDto();
        UserProfileMessagingAdditionalDataDto additionalData = new UserProfileMessagingAdditionalDataDto();
        additionalData.setImageUrl(userProfileDto.getImageUrl());

        switch (event.getEventType()) {
            case CREATED -> {
                messagingEventDto.setTimestamp(userProfileDto.getCreationTime().toInstant());
            }
            case UPDATED, CV_UPDATED, IMAGE_UPDATED, LANGUAGE_UPDATED -> {
                messagingEventDto.setTimestamp(userProfileDto.getUpdateTime().toInstant());
            }
            case DELETED -> {
                messagingEventDto.setTimestamp(event.getEventTimestamp());
            }
        }

        messagingEventDto.setAdditionalData(additionalData);
    }

    private void publishEventForType(@NonNull UserProfileApplicationEvent event,
                                     @NonNull MessagingEventDto<UserProfileMessagingAdditionalDataDto> messagingEventDto) {

        switch (event.getEventType()) {
            case CREATED -> {
                resourceProducer.publishCreatedResource(event.getUserProfileDto());
                eventProducer.publishCreateEvent(messagingEventDto);
            }
            case UPDATED -> {
                resourceProducer.publishUpdatedResource(event.getUserProfileDto());
                eventProducer.publishUpdateEvent(messagingEventDto);
            }
            case DELETED -> {
                resourceProducer.publishDeletedResource(event.getUserProfileDto());
                eventProducer.publishDeleteEvent(messagingEventDto);
            }
            case CV_UPDATED -> {
                resourceProducer.publishUpdatedResource(event.getUserProfileDto());
                eventProducer.publishUpdateCVEvent(messagingEventDto);
            }
            case IMAGE_UPDATED -> {
                resourceProducer.publishUpdatedResource(event.getUserProfileDto());
                eventProducer.publishUpdateImageEvent(messagingEventDto);
            }
            case LANGUAGE_UPDATED -> {
                resourceProducer.publishUpdatedResource(event.getUserProfileDto());
                eventProducer.publishUpdateLanguageEvent(messagingEventDto);
            }
        }
    }
}
