package com.finconsgroup.itserr.marketplace.search.dm.controller;

import com.finconsgroup.itserr.marketplace.search.dm.api.InstitutionalPageApi;
import com.finconsgroup.itserr.marketplace.search.dm.dto.InputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputInstitutionalPageLocalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.service.InstitutionalPageService;
import com.finconsgroup.itserr.marketplace.search.dm.util.SortUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.propertyeditors.StringTrimmerEditor;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Map;

/**
 * Spring Boot REST Controller implementation for the {@link InstitutionalPageApi} endpoints.
 */
@RestController
@RequiredArgsConstructor
@Slf4j
public class InstitutionalPageController implements InstitutionalPageApi {
    private final InstitutionalPageService institutionalPageService;

    @InitBinder
    public void initBinder(WebDataBinder binder) {
        // trim strings before triggering validation, but do not convert to null
        // as otherwise it throws - MissingServletRequestParameterException
        binder.registerCustomEditor(String.class, new StringTrimmerEditor(false));
    }

    @NonNull
    @Override
    public OutputInstitutionalPageDto upsertInstitutionalPage(String id, InputInstitutionalPageDto dto) {
        return institutionalPageService.upsertDocument(dto);
    }

    @NonNull
    @Override
    public OutputInstitutionalPageDto getInstitutionalPage(String id) {
        return institutionalPageService.getDocument(id);
    }

    @Override
    public void deleteInstitutionalPage(String id) {
        institutionalPageService.deleteDocument(id);
    }

    @Override
    public void deleteAll() {
        institutionalPageService.deleteAll();
    }

    @NonNull
    @Override
    public List<OutputGlobalSearchAutoCompleteDto> getAutoCompletions(@NonNull String terms) {
        return institutionalPageService.getAutoCompletions(terms);
    }

    @NonNull
    @Override
    public Page<OutputInstitutionalPageLocalSearchDto> getLocalSearch(@Nullable String terms,
                                                                      @Nullable String filters,
                                                                      int pageNumber,
                                                                      int pageSize,
                                                                      String sort,
                                                                      Sort.Direction direction,
                                                                      String multiSort) {

        Sort sortToApply = SortUtils.buildSort(sort, direction, multiSort, ":", Map.of());
        Pageable pageable = PageRequest.of(pageNumber, pageSize, sortToApply);
        return institutionalPageService.getLocalSearch(terms, filters, pageable);
    }
}
