package com.finconsgroup.itserr.marketplace.search.dm.dto;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.finconsgroup.itserr.marketplace.core.web.dto.PutIdParameterBody;
import io.swagger.v3.oas.annotations.Hidden;
import io.swagger.v3.oas.annotations.media.Schema;
import jakarta.validation.constraints.NotBlank;
import lombok.Builder;
import lombok.Data;

import java.time.Instant;
import java.util.LinkedHashMap;
import java.util.List;

/**
 * It represents the payload to create/update an Institutional Page search document.
 */
@Builder
@Schema(title = "Create/Update Institutional Page search document DTO",
    description = "Represents the payload to create/update a Institutional Page search document")
@Data
public class InputInstitutionalPageDto implements PutIdParameterBody {

    @Schema(description = "ID of the institutional page", example = "11111111-aaaa-0000-0000-000000000000")
    @NotBlank
    private String id;

    @Schema(description = "The name of the institutional page", example = "Criterion")
    private String name;

    @Schema(
            description = "ID of the workspace folder related to the institutional page",
            example = "22222222-aaaa-0000-0000-000000000000"
    )
    private String workspaceFolderId;

    @Schema(description = "The parent node of the institutional page")
    private RelatedInstitutionalPageDto parentInstitutionalPage;

    @Schema(description = "The child nodes of the institutional page")
    private List<RelatedInstitutionalPageDto> childInstitutionalPages;

    @Schema(description = "ID of the root institutional page")
    private String rootInstitutionalPageId;

    @Schema(description = "Publication Time", example = "2025-05-26T14:19:46.606883Z")
    private Instant publicationTime;

    @Schema(description = "Rejection message of moderation", example = "Not correct")
    private String rejectionMessage;

    @Schema(description = "Indicates whether the institutional page is published", example = "true")
    private Boolean published;

    @Schema(
            description = "The category of the institutional page",
            allowableValues = {"tool", "project"},
            example = "project"
    )
    private String category;

    @Schema(
            description = "The moderation status of the institutional page",
            allowableValues = {"Draft", "Pending", "Approved", "Rejected"},
            example = "Pending"
    )
    private String moderationStatus;

    @Schema(
            description = "The moderation status of the parallel version",
            allowableValues = {"Draft", "Pending", "Approved", "Rejected"},
            example = "Pending"
    )
    private String parallelVersionStatus;

    @Schema(
            description = "The abstract of the institutional page",
            example = "WP3, the Toolkit for Religious Studies (T-ReS), aims to improve the research experience."
    )
    private String abstractContent;

    @Schema(description = "Copyright info for the institutional page")
    private InstitutionalPageCopyrightDto copyright;

    @Schema(description = "Profile of the WP leads of the institutional page")
    private List<UserProfileMinimalDto> wpLeads;

    @Schema(description = "Profile of members of the institutional page")
    private List<UserProfileMinimalDto> members;

    @Schema(description = "Profile of the maintainer of the institutional page")
    private UserProfileMinimalDto maintainer;

    @Schema(description = "Profile of the user who last modified the institutional page")
    private UserProfileMinimalDto lastModifiedBy;

    @Schema(description = "The paragraphs related to the institutional page")
    private List<InstitutionalPageParagraphDto> paragraphs;

    @Schema(description = "List of tags", example = "[\"aramaic\",\"english\",\"italian\"]")
    private List<String> tags;

    @Schema(description = "List of languages", example = "[\"English\",\"Italian\",\"German\"]")
    private List<String> languages;

    @Schema(description = "The image URL", example = "https://picsum.photos/200")
    private String imageUrl;

    @Schema(description = "The user manual URL", example = "https://www.usermanual.com")
    private String userManualUrl;

    @Schema(description = "The FAQs page URL", example = "https://www.faqs.com")
    private String faqsUrl;

    @Schema(description = "The application version", example = "v1.3.0")
    private String applicationVersion;

    @Schema(description = "The application type", example = "web application")
    private String applicationType;

    @Schema(description = "The license name", example = "MIT License")
    private String license;

    @Schema(description = "The supported operating systems for the institutional page")
    private String supportedOs;

    @Schema(description = "Output DTO of repository links for the institutional page")
    private InstitutionalPageRepositoriesDto repositories;

    @Schema(
            description = "Additional arbitrary fields in key-value format",
            example = "{\"developer\": \"John Doe\", \"releaseNotes\": \"Initial beta release\"}"
    )
    private LinkedHashMap<String, String> additionalFields;

    @Schema(description = "Buttons related to the institutional page")
    private List<InstitutionalPageButtonDto> buttons;

    @Schema(description = "The publications associated to the institutional page")
    private List<InstitutionalPagePublicationDto> publications;

    @Schema(description = "External links")
    private LinkedHashMap<String, String> externalLinks;

    @Schema(description = "Created At", example = "2025-05-26T14:19:46.606883Z")
    private Instant creationTime;

    @Schema(description = "Last Updated At", example = "2025-05-27T14:19:46.606883Z")
    private Instant updateTime;

    @Schema(description = "Indicates whether the institutional page is waiting to be deleted", example = "true")
    private Boolean toDelete;

    @Schema(description = "Indicates whether the institutional page has an updated version", example = "true")
    private Boolean hasUpdatedVersion;

    @Schema(hidden = true)
    @JsonIgnore
    @Hidden
    @Override
    public String getPutId() {
        return id;
    }


}
