package com.finconsgroup.itserr.marketplace.notification.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.InputCreateUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.InputPatchUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.InputPatchUserNotificationsDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputPatchUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.UserDto;
import com.finconsgroup.itserr.marketplace.notification.dm.entity.ArchivedUserNotificationEntity;
import com.finconsgroup.itserr.marketplace.notification.dm.entity.UserNotificationEntity;
import com.finconsgroup.itserr.marketplace.notification.dm.mapper.ArchivedUserNotificationMapper;
import com.finconsgroup.itserr.marketplace.notification.dm.mapper.UserNotificationMapper;
import com.finconsgroup.itserr.marketplace.notification.dm.repository.ArchivedUserNotificationRepository;
import com.finconsgroup.itserr.marketplace.notification.dm.repository.UserNotificationRepository;
import com.finconsgroup.itserr.marketplace.notification.dm.service.LocalizedUserNotificationService;
import com.finconsgroup.itserr.marketplace.notification.dm.service.UserNotificationService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.UUID;

/**
 * Default implementation of {@link UserNotificationService} to perform operations related to notification resources
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultUserNotificationService implements UserNotificationService {

    private final UserNotificationRepository userNotificationRepository;
    private final UserNotificationMapper userNotificationMapper;

    private final ArchivedUserNotificationRepository archivedUserNotificationRepository;
    private final ArchivedUserNotificationMapper archivedUserNotificationMapper;

    private final LocalizedUserNotificationService localizedUserNotificationService;

    @Override
    @NonNull
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public List<OutputUserNotificationDto> create(
            @NotNull final InputCreateUserNotificationDto request) {

        final List<UserNotificationEntity> entities = userNotificationMapper.toEntities(request);
        for (final UserNotificationEntity entity : entities) {
            entity.setRead(Boolean.FALSE);
            entity.setArchived(Boolean.FALSE);
        }
        final List<UserNotificationEntity> savedEntities = userNotificationRepository.saveAllAndFlush(entities);

        return savedEntities.stream()
                .map(userNotificationMapper::toDto)
                .toList();

    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public @NotNull OutputPatchUserNotificationDto patchById(
            @NonNull UserDto user,
            @NotNull UUID notificationId,
            @NotNull InputPatchUserNotificationDto inputPatchUserNotificationDto) {
        UserNotificationEntity userNotificationEntity = findByIdOrThrow(
                user,
                notificationId);
        userNotificationMapper.patchEntity(inputPatchUserNotificationDto, userNotificationEntity);
        UserNotificationEntity savedUserNotificationEntity = userNotificationRepository.save(userNotificationEntity);
        return userNotificationMapper.toPatchDto(savedUserNotificationEntity);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void deleteById(
            @NonNull UserDto user,
            @NotNull UUID notificationId) {
        UserNotificationEntity userNotificationEntity = findByIdOrThrow(
                user,
                notificationId);
        ArchivedUserNotificationEntity archivedUserNotificationEntity = archivedUserNotificationMapper.toArchivedEntity(userNotificationEntity);
        archivedUserNotificationRepository.save(archivedUserNotificationEntity);
        userNotificationRepository.deleteById(notificationId);
    }

    private UserNotificationEntity findByIdOrThrow(
            @NonNull UserDto user,
            @NonNull UUID notificationId) {
        return userNotificationRepository
                .findByUserAndId(user.id().toString(), user.username(), user.email(), notificationId)
                .orElseThrow(() -> new WP2ResourceNotFoundException(notificationId));
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void patchAll(
            @NonNull final UserDto user,
            @NonNull final InputPatchUserNotificationsDto request) {

        if (request.getRead() != null) {
            userNotificationRepository.updateRead(
                    user.id().toString(),
                    user.username(),
                    user.email(),
                    request.getRead());
        }

    }

}
