package com.finconsgroup.itserr.marketplace.notification.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

import java.time.Instant;

/**
 * Entity class representing a notification email template for a specific email type and language.
 */
@Entity
@Table(name = "notification_email_template")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class NotificationEmailTemplateEntity extends AbstractUUIDEntity {

    /**
     * The language code for the notification template (e.g., "en", "it", "fr"). Uses ISO 639-1 two-letter language codes.
     */
    @Column(name = "language_code", nullable = false, length = 5)
    private String languageCode;

    /**
     * Default Email subject text
     */
    @Column(name = "subject_text", nullable = false)
    private String subjectText;

    /**
     * Default Email body text
     */
    @Column(name = "body_text", nullable = false)
    private String bodyText;

    /**
     * The timestamp when the notification was created.
     */
    @Column(name = "creation_time", nullable = false)
    private Instant creationTime;

    /**
     * The timestamp when the notification was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates. Each time the entity is updated, the version is incremented. If two transactions
     * try to update the same entity simultaneously, JPA will detect the conflict based on this version and throw an
     * {@link jakarta.persistence.OptimisticLockException}.
     */
    @Version
    @Column(name = "version", nullable = false)
    private long version;

    @Override
    public void prePersist() {
        super.prePersist();
        Instant now = Instant.now();
        if (creationTime == null) {
            creationTime = now;
        }
        if (updateTime == null) {
            updateTime = now;
        }
    }

}
