package com.finconsgroup.itserr.marketplace.notification.bs.api;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.InputCreateUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.InputPatchUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.InputPatchUserNotificationsDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.OutputPatchUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.OutputUserNotificationDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.List;
import java.util.UUID;

/**
 * This interface defines the contract for REST API endpoints related to Notification Management.
 *
 * <p>
 * It provides endpoints for retrieving, creating, updating, and deleting Notifications.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * POST notification/notifications - Create a new notification.
 * PATCH  notification/notifications/{id} - Update an existing notification.
 * DELETE notification/notifications/{id} - Delete a notification.
 * </pre>
 * </p>
 */
@Tag(
        name = "UserNotificationApi",
        description = "The UserNotificationApi API: it provides endpoints for " +
                "retrieving, creating, updating, and deleting Notifications for a user."
)
@SecurityRequirement(name = "BearerAuth")
public interface UserNotificationApi {

    /**
     * Creates a new user notification.
     *
     * @param inputCreateUserNotificationDto the input data transfer object containing user notification details
     * @param acceptLanguage Accept-Language header value.
     * @return the created notification and HTTP status 201 (Created)
     */
    @Operation(
            summary = "create users notifications",
            responses = {
                    @ApiResponse(responseCode = "201", description = "Created"),
                    @ApiResponse(responseCode = "503", description = "Service unavailable", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
            }
    )
    @PostMapping(
            value = "/notification/notifications",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.CREATED)
    List<OutputUserNotificationDto> createNotification(
            @Valid @RequestBody InputCreateUserNotificationDto inputCreateUserNotificationDto,
            @RequestHeader(name = "Accept-Language", required = false) String acceptLanguage);

    /**
     * Patches all existing user notifications.
     *
     * @param request the patching request
     */
    @Operation(
            summary = "patch all user notifications, mark notification as read/unread",
            responses = {
                    @ApiResponse(responseCode = "204", description = "OK"),
            }
    )
    @PatchMapping(
            value = "/notification/notifications",
            consumes = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.NO_CONTENT)
    void patchAll(
            @Valid @RequestBody InputPatchUserNotificationsDto request
    );

    /**
     * Patches an existing user notification by id.
     *
     * @param id the id of the notification to patch
     * @param inputPatchUserNotificationDto the input data transfer object containing patched notification details
     * @return the patched {@link OutputPatchUserNotificationDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "patch notification by id, mark notification as read/unRead or archive/un-archive",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
                    @ApiResponse(responseCode = "503", description = "Service unavailable", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
            }
    )
    @PatchMapping(
            value = "/notification/notifications/{id}",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputPatchUserNotificationDto patchUserNotificationById(
            @PathVariable("id") UUID id,
            @Valid @RequestBody InputPatchUserNotificationDto inputPatchUserNotificationDto
    );

    /**
     * Deletes a user notification by id.
     *
     * @param id the id of the notification to delete
     */
    @Operation(
            summary = "delete a user notification by id",
            responses = {
                    @ApiResponse(responseCode = "204", description = "No Content"),
                    @ApiResponse(responseCode = "404", description = "Not Found", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
                    @ApiResponse(responseCode = "503", description = "Service unavailable", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
            }
    )
    @ResponseStatus(HttpStatus.NO_CONTENT)
    @DeleteMapping(value = "/notification/notifications/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    void deleteUserNotificationById(@PathVariable("id") UUID id);

}
