package com.finconsgroup.itserr.marketplace.metrics.feeder.messaging;

import com.fasterxml.jackson.core.type.TypeReference;
import com.finconsgroup.itserr.marketplace.metrics.feeder.client.MetricsBsClient;
import com.finconsgroup.itserr.marketplace.metrics.feeder.config.properties.MetricsFeederConfigurationProperties;
import com.finconsgroup.itserr.marketplace.metrics.feeder.dto.favouriteuserbs.FavouriteItemMessageBodyDto;
import com.finconsgroup.itserr.marketplace.metrics.feeder.dto.metricbs.InputCreateMetricEventDto;
import com.finconsgroup.itserr.marketplace.metrics.feeder.dto.metricbs.MetricDtoType;
import io.cloudevents.CloudEvent;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

/**
 * Subscriber class that handles favorite user resource events and converts them into metric events.
 */
@Slf4j
@Component("favouriteUserEventConsumer")
public class FavouriteUserEventConsumer
        extends AbstractMetricEventConsumer<FavouriteItemMessageBodyDto> {

    /**
     * Constructor.
     *
     * @param config Configuration properties for the metrics feeder
     * @param metricsBsClient Client for interacting with the metrics business service
     */
    public FavouriteUserEventConsumer(
            final MetricsFeederConfigurationProperties config,
            final MetricsBsClient metricsBsClient) {

        super(
                metricsBsClient,
                MetricDtoType.CATALOG_ITEM_FAVOURITE,
                config.getMessaging().getFavouriteUser().getResourceCreatedEventType(),
                new TypeReference<>() {
                });

    }

    @Override
    protected void fillMetricEvent(
            @NonNull final InputCreateMetricEventDto.InputCreateMetricEventDtoBuilder metricEventBuilder,
            @NonNull final FavouriteItemMessageBodyDto resourcePayload,
            @NonNull final CloudEvent event) throws ValidationException {

        if (StringUtils.isBlank(resourcePayload.getCategory())) {
            throw new ValidationException("no category");
        } else if (!StringUtils.equalsIgnoreCase(resourcePayload.getCategory(), "CATALOG")) {
            throw new ValidationException("not a catalog item favourite (type=" + resourcePayload.getCategory() + ")", false);
        } else if (resourcePayload.getAdditionalData() == null) {
            throw new ValidationException("no additional data");
        } else if (StringUtils.isBlank(resourcePayload.getAdditionalData().getItemId())) {
            throw new ValidationException("no item id");
        } else {

            final String itemId = resourcePayload.getAdditionalData().getItemId();

            metricEventBuilder
                    .resourceId(itemId);

        }
    }

}
