package com.finconsgroup.itserr.marketplace.metrics.dm.api;

import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputReportMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputResourcesDailyMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputResourcesSummaryMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.ReportGranularity;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.constraints.NotEmpty;
import jakarta.validation.constraints.NotNull;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.time.LocalDate;
import java.util.List;
import java.util.Set;

/**
 * This interface defines the contract for REST API endpoints related to metrics.
 */
@Tag(
        name = "Metric",
        description = "The Metric API: it provides endpoints for managing metrics"
)
@SecurityRequirement(name = "BearerAuth")
public interface MetricApi {

    /**
     * Retrieves the daily metrics for a given list of resources within a specified date range.
     *
     * @param resourcesIds the list of resource IDs for which metrics are to be retrieved
     * @param fromDate the start date of the date range (inclusive)
     * @param toDate the end date of the date range (inclusive)
     * @return an {@link OutputResourcesDailyMetricsDto} containing the daily metrics for the specified resources within the date range
     */
    @Operation(
            summary = "Find resources daily metrics for a dates range",
            responses = { @ApiResponse(responseCode = "200", description = "OK") }
    )
    @GetMapping(value = "/metrics/resources/{resourcesIds}/metrics/{fromDate}/{toDate}", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputResourcesDailyMetricsDto findResourcesDailyMetrics(
            @Schema(example = "f47ac10b-58cc-4372-a567-0e02b2c3d479") @PathVariable("resourcesIds") Set<String> resourcesIds,
            @Schema(example = "2025-08-08") @PathVariable("fromDate") LocalDate fromDate,
            @Schema(example = "2025-08-10") @PathVariable("toDate") LocalDate toDate);

    /**
     * Summarizes the total metrics of the specified resources, combining their metrics into a single summary.
     *
     * @param resourcesIds the set of resource IDs for which the metrics summary is to be retrieved; must not be empty
     * @param toDate the optional end date for calculating the metrics summary; if null, the current date is assumed
     * @return an {@link OutputResourcesSummaryMetricsDto} containing the aggregated metrics for the specified resources
     */
    @Operation(
            summary = "Summarize the total metrics of the given resources (summed together)",
            responses = { @ApiResponse(responseCode = "200", description = "OK") }
    )
    @GetMapping(value = "/metrics/metrics-summaries", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputResourcesSummaryMetricsDto findResourcesSummaryMetrics(
            @RequestParam("resourceId") @NotEmpty Set<String> resourcesIds,
            @RequestParam(value = "toDate", required = false) LocalDate toDate);

    /**
     * Summarizes the total metrics of the specified resources, combining their metrics into a single summary based
     * on the report granularity.
     *
     * @param resourcesIds the set of resource IDs for which the metrics summary is to be retrieved; must not be empty
     * @param granularity  the report granularity
     * @param fromDate the start date of the date range (inclusive)
     * @param toDate the end date of the date range (inclusive)
     * @return a list of {@link OutputReportMetricsDto} containing the aggregated metrics for the specified resources
     */
    @Operation(
            summary = "Summarize the total metrics of the given resources (summed together) aggregated by resource id",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(value = "/metrics/report/metrics-summaries", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    List<OutputReportMetricsDto> findReportingMetrics(
            @Schema(example = "f47ac10b-58cc-4372-a567-0e02b2c3d479")
            @RequestParam("resourceId") @NotEmpty Set<String> resourcesIds,
            @Schema(example = "USER")
            @RequestParam("granularity") @NotNull ReportGranularity granularity,
            @Schema(example = "2025-08-08")
            @RequestParam(value = "fromDate", required = false) LocalDate fromDate,
            @Schema(example = "2025-08-10")
            @RequestParam(value = "toDate", required = false) LocalDate toDate);

}
