package com.finconsgroup.itserr.marketplace.metrics.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ExecutionException;
import com.finconsgroup.itserr.marketplace.metrics.bs.bean.MetricsReportRequest;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.ExportFileFormat;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.ReportType;
import com.finconsgroup.itserr.marketplace.metrics.bs.service.MetricReportService;
import jakarta.annotation.PostConstruct;
import lombok.extern.slf4j.Slf4j;
import net.sf.jasperreports.engine.JRDataSource;
import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JRTextElement;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.JasperReportsContext;
import net.sf.jasperreports.engine.SimpleJasperReportsContext;
import net.sf.jasperreports.engine.data.JRBeanCollectionDataSource;
import net.sf.jasperreports.engine.export.JRCsvExporter;
import net.sf.jasperreports.engine.export.ooxml.JRXlsxExporter;
import net.sf.jasperreports.engine.util.JRLoader;
import net.sf.jasperreports.export.SimpleCsvExporterConfiguration;
import net.sf.jasperreports.export.SimpleCsvReportConfiguration;
import net.sf.jasperreports.export.SimpleExporterInput;
import net.sf.jasperreports.export.SimpleOutputStreamExporterOutput;
import net.sf.jasperreports.export.SimpleWriterExporterOutput;
import net.sf.jasperreports.export.SimpleXlsxExporterConfiguration;
import net.sf.jasperreports.export.SimpleXlsxReportConfiguration;
import net.sf.jasperreports.pdf.JRPdfExporter;
import net.sf.jasperreports.pdf.SimplePdfExporterConfiguration;
import net.sf.jasperreports.pdf.SimplePdfReportConfiguration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.core.io.InputStreamResource;
import org.springframework.lang.NonNull;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.stereotype.Service;

import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;

import static com.finconsgroup.itserr.marketplace.metrics.bs.config.ReportConfiguration.REPORT_SCHEDULER;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.Constants.MESSAGE_SOURCE_BEAN_NAME;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.ReportConstants.KEY_HEADER_COMMENTS;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.ReportConstants.KEY_HEADER_DOWNLOADS;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.ReportConstants.KEY_HEADER_FAVOURITES;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.ReportConstants.KEY_HEADER_INTEREST;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.ReportConstants.KEY_HEADER_RESOURCE_ID;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.ReportConstants.KEY_HEADER_RESOURCE_TITLE;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.ReportConstants.KEY_HEADER_VIEWS;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.ReportConstants.MESSAGE_KEY_HEADER_COMMENTS;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.ReportConstants.MESSAGE_KEY_HEADER_DOWNLOADS;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.ReportConstants.MESSAGE_KEY_HEADER_FAVOURITES;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.ReportConstants.MESSAGE_KEY_HEADER_INTEREST;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.ReportConstants.MESSAGE_KEY_HEADER_RESOURCE_ID;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.ReportConstants.MESSAGE_KEY_HEADER_RESOURCE_TITLE;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.ReportConstants.MESSAGE_KEY_HEADER_VIEWS;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.ReportConstants.MESSAGE_KEY_SHEET_DETAIL;
import static com.finconsgroup.itserr.marketplace.metrics.bs.util.ReportConstants.MESSAGE_KEY_SHEET_SUMMARY;

/**
 * Default implementation of {@link MetricReportService} to perform operations related to metrices related reports.
 */
@Service
@Slf4j
public class DefaultMetricReportService implements MetricReportService {

    private final MessageSource messageSource;
    private final ThreadPoolTaskScheduler reportScheduler;

    private final Map<ExportFileFormat, List<JasperReport>> jasperReportsByFormat = new EnumMap<>(ExportFileFormat.class);
    private final Map<ExportFileFormat, Map<Integer, ReportType>> reportTypeMapByFormat = new EnumMap<>(ExportFileFormat.class);

    public DefaultMetricReportService(@Qualifier(MESSAGE_SOURCE_BEAN_NAME) MessageSource messageSource,
                                      @Qualifier(REPORT_SCHEDULER) ThreadPoolTaskScheduler reportScheduler) {
        this.messageSource = messageSource;
        this.reportScheduler = reportScheduler;
    }

    @PostConstruct
    public void init() {
        int index;
        for (ExportFileFormat format : ExportFileFormat.values()) {
            index = 0;
            List<JasperReport> jasperReports = new ArrayList<>();
            Map<Integer, ReportType> reportTypeMap = new HashMap<>();

            if (addReport(jasperReports, format, ReportType.SUMMARY)) {
                reportTypeMap.put(index++, ReportType.SUMMARY);
            }

            if (addReport(jasperReports, format, ReportType.DETAIL)) {
                reportTypeMap.put(index, ReportType.DETAIL);
            }

            if (!jasperReports.isEmpty()) {
                jasperReportsByFormat.put(format, jasperReports);
                reportTypeMapByFormat.put(format, reportTypeMap);
            }
        }
    }

    @Override
    public InputStreamResource downloadReport(final @NonNull ExportFileFormat format,
                                              final @NonNull MetricsReportRequest metricsReportRequest) {
        if (!jasperReportsByFormat.containsKey(format)) {
            throw new WP2BusinessException("User metrics report could not be loaded for format: " + format);
        }
        try {

            Map<String, Object> parameters = new HashMap<>();
            addHeaderLabels(parameters, metricsReportRequest.locale());

            List<JasperReport> jasperReports = jasperReportsByFormat.get(format);
            Map<Integer, ReportType> reportTypeMap = reportTypeMapByFormat.get(format);
            List<JasperPrint> jasperPrints = new ArrayList<>();
            for (int i = 0; i < jasperReports.size(); i++) {
                JasperReport jasperReport = jasperReports.get(i);
                JRDataSource jrDataSource;
                if (Objects.requireNonNull(reportTypeMap.get(i)) == ReportType.SUMMARY) {
                    jrDataSource = new JRBeanCollectionDataSource(metricsReportRequest.summaryMetrics(), false);
                } else {
                    jrDataSource = new JRBeanCollectionDataSource(metricsReportRequest.resourcesMetrics(), false);
                }
                JasperPrint jasperPrint = JasperFillManager.fillReport(jasperReport, parameters, jrDataSource);
                jasperPrints.add(jasperPrint);
            }

            JasperReportsContext jasperReportsContext = new SimpleJasperReportsContext();
            jasperReportsContext.setProperty(JRTextElement.PROPERTY_PRINT_KEEP_FULL_TEXT, Boolean.TRUE.toString());


            PipedInputStream in = new PipedInputStream();
            PipedOutputStream out = new PipedOutputStream(in);

            // need to submit the task to the write the report in a new thread
            // as otherwise the input stream wrapped over the report's output stream
            // cannot be returned
            reportScheduler.submit(() -> {
                try (out) {
                    switch (format) {
                        case CSV -> exportToCsv(jasperReportsContext, jasperPrints, out);
                        case PDF -> exportToPdf(jasperReportsContext, jasperPrints, out);
                        default ->
                                exportToExcel(jasperReportsContext, jasperPrints, out, metricsReportRequest.locale());
                    }
                } catch (Exception ex) {
                    log.error("Error exporting user metrics report", ex);
                }
            });

            return new InputStreamResource(in);
        } catch (Exception ex) {
            throw new WP2ExecutionException(ex);
        }
    }

    private boolean addReport(List<JasperReport> jasperReports, ExportFileFormat format, ReportType reportType) {
        try {
            String reportFileName = "userMetrics%sReport_%s".formatted(reportType.getValue(),
                    format.name().toUpperCase());
            String compiledReportFilePath = "/jasperreports/compiled/%s.jasper".formatted(reportFileName);
            InputStream compiledReportStream = getClass().getResourceAsStream(compiledReportFilePath);
            JasperReport jasperReport = (JasperReport) JRLoader.loadObject(compiledReportStream);
            jasperReports.add(jasperReport);
            log.info("Loaded user metrics report for format: {} and report type: {} from path: {}",
                    format, reportType, compiledReportFilePath);
            return true;
        } catch (Exception e) {
            log.error("Error loading user metrics report for format: {} and report type: {}", format, reportType, e);
            return false;
        }
    }

    private void addHeaderLabels(Map<String, Object> parameters, Locale locale) {
        parameters.put(KEY_HEADER_RESOURCE_ID,
                messageSource.getMessage(MESSAGE_KEY_HEADER_RESOURCE_ID, null, locale));
        parameters.put(KEY_HEADER_RESOURCE_TITLE,
                messageSource.getMessage(MESSAGE_KEY_HEADER_RESOURCE_TITLE, null, locale));
        parameters.put(KEY_HEADER_INTEREST,
                messageSource.getMessage(MESSAGE_KEY_HEADER_INTEREST, null, locale));
        parameters.put(KEY_HEADER_DOWNLOADS,
                messageSource.getMessage(MESSAGE_KEY_HEADER_DOWNLOADS, null, locale));
        parameters.put(KEY_HEADER_VIEWS,
                messageSource.getMessage(MESSAGE_KEY_HEADER_VIEWS, null, locale));
        parameters.put(KEY_HEADER_FAVOURITES,
                messageSource.getMessage(MESSAGE_KEY_HEADER_FAVOURITES, null, locale));
        parameters.put(KEY_HEADER_COMMENTS,
                messageSource.getMessage(MESSAGE_KEY_HEADER_COMMENTS, null, locale));
    }

    private void exportToPdf(JasperReportsContext jasperReportsContext,
                             List<JasperPrint> jasperPrints,
                             OutputStream outputStream) throws JRException {
        JRPdfExporter exporter = new JRPdfExporter(jasperReportsContext);
        exporter.setExporterInput(SimpleExporterInput.getInstance(jasperPrints));
        exporter.setExporterOutput(new SimpleOutputStreamExporterOutput(outputStream));

        // Optional: Configure additional settings
        SimplePdfExporterConfiguration configuration = new SimplePdfExporterConfiguration();
        exporter.setConfiguration(configuration);

        SimplePdfReportConfiguration reportConfiguration = new SimplePdfReportConfiguration();
        exporter.setConfiguration(reportConfiguration);

        exporter.exportReport();
    }

    private void exportToCsv(JasperReportsContext jasperReportsContext,
                             List<JasperPrint> jasperPrints,
                             OutputStream outputStream) throws JRException {
        JRCsvExporter exporter = new JRCsvExporter(jasperReportsContext);
        exporter.setExporterInput(SimpleExporterInput.getInstance(jasperPrints));
        exporter.setExporterOutput(new SimpleWriterExporterOutput(outputStream));

        // Optional: Configure additional settings
        SimpleCsvExporterConfiguration configuration = new SimpleCsvExporterConfiguration();
        exporter.setConfiguration(configuration);

        SimpleCsvReportConfiguration reportConfiguration = new SimpleCsvReportConfiguration();
        exporter.setConfiguration(reportConfiguration);

        exporter.exportReport();
    }

    private void exportToExcel(JasperReportsContext jasperReportsContext,
                               List<JasperPrint> jasperPrints,
                               OutputStream outputStream,
                               Locale locale) throws JRException {
        JRXlsxExporter exporter = new JRXlsxExporter(jasperReportsContext);
        exporter.setExporterInput(SimpleExporterInput.getInstance(jasperPrints));
        exporter.setExporterOutput(new SimpleOutputStreamExporterOutput(outputStream));

        // Optional: Configure additional settings
        SimpleXlsxExporterConfiguration configuration = new SimpleXlsxExporterConfiguration();
        exporter.setConfiguration(configuration);

        SimpleXlsxReportConfiguration reportConfiguration = new SimpleXlsxReportConfiguration();
        reportConfiguration.setOnePagePerSheet(true);
        reportConfiguration.setRemoveEmptySpaceBetweenColumns(true);
        reportConfiguration.setRemoveEmptySpaceBetweenRows(true);
        reportConfiguration.setShowGridLines(true);
        reportConfiguration.setWhitePageBackground(false);
        reportConfiguration.setDetectCellType(true);
        reportConfiguration.setSheetNames(new String[]{
                messageSource.getMessage(MESSAGE_KEY_SHEET_SUMMARY, null, locale),
                messageSource.getMessage(MESSAGE_KEY_SHEET_DETAIL, null, locale),
        });
        exporter.setConfiguration(reportConfiguration);

        exporter.exportReport();
    }
}
