package com.finconsgroup.itserr.marketplace.metadata.bs.controller;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.metadata.bs.api.DiagnosticsApi;
import com.finconsgroup.itserr.marketplace.metadata.bs.bean.MetadataEvent;
import com.finconsgroup.itserr.marketplace.metadata.bs.bean.MetadataModerationRequestedEvent;
import com.finconsgroup.itserr.marketplace.metadata.bs.bean.MetadataStatusEvent;
import com.finconsgroup.itserr.marketplace.metadata.bs.config.MetadataBsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.MetadataStatus;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.enums.EventType;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.appender.RollingFileAppender;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RestController;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

/**
 * REST controller class for handling debugging/analysis related API requests.
 *
 * <p>Implements the {@link DiagnosticsApi} interface.</p>
 */
@Slf4j
@RestController
@RequiredArgsConstructor
public class DiagnosticsController implements DiagnosticsApi {

    private final MetadataBsConfigurationProperties metadataBsConfigurationProperties;
    private final ApplicationEventPublisher applicationEventPublisher;

    // For storing rabbitmq test messages
    static Map<String, OutputMetadataDto> stringOutputMetadataDtoHashMap = new HashMap<>();

    // The name of the RollingFile Log4j2 component
    // It is not the name of the log file
    // We use this to dynamically retrieve the name of the log file.
    @Value("${log.log4j2-rolling-file-name}")
    private String log4j2RollingFileName;

    @Override
    public ResponseEntity<Resource> downloadLogs() {
        log.debug("call to DiagnosticsController - downloadLogs");
        Path filePath = Paths.get(getLogFilePathFromLog4j2());
        log.debug("Trying to retrieve log file from: {}", filePath);
        File logFile = filePath.toFile();
        if (!logFile.exists() || !logFile.isFile()) {
            throw new WP2ResourceNotFoundException("Log file not found: %s".formatted(logFile.getAbsolutePath()));
        }
        Resource resource = new FileSystemResource(logFile);
        return ResponseEntity.ok()
                .header(HttpHeaders.CONTENT_DISPOSITION, getContentDispositionHeaderValue(logFile.getName()))
                .contentType(MediaType.APPLICATION_OCTET_STREAM)
                .body(resource);
    }

    // private

    public String getLogFilePathFromLog4j2() {
        LoggerContext context = (LoggerContext) LogManager.getContext(false);
        RollingFileAppender appender = context.getConfiguration().getAppender(log4j2RollingFileName);
        return appender.getFileName();
    }

    private static String getContentDispositionHeaderValue(String fileName) {
        return "attachment; filename=\"%s\"".formatted(fileName);
    }

    @Override
    public ResponseEntity<String> publishMessage(EventType eventType, String name) {
        if (!metadataBsConfigurationProperties.getDiagnostics().isPublishMessageEnabled()) {
            log.warn("Attempt to access disabled diagnostics endpoint");
            return ResponseEntity.status(HttpStatus.FORBIDDEN)
                    .body("This endpoint is disabled in the current environment");
        }

        log.debug("call to DiagnosticsController - publishMessage, name: %s".formatted(name));

        OutputMetadataDto outputMetadataDto;
        switch (eventType) {
            case CREATED:
                outputMetadataDto = buildTestOutputMetadataDto(name, MetadataStatus.DRAFT);
                log.info("sending created message, outputMetadataDto: %s".formatted(outputMetadataDto));
                stringOutputMetadataDtoHashMap.put(outputMetadataDto.getName(), outputMetadataDto);
                applicationEventPublisher.publishEvent(
                        new MetadataEvent(outputMetadataDto, EventType.CREATED)
                );
                break;
            case UPDATED:
                outputMetadataDto = buildTestOutputMetadataDto(name, MetadataStatus.DRAFT);
                log.info("sending updated message, outputMetadataDto: %s".formatted(outputMetadataDto));
                stringOutputMetadataDtoHashMap.put(outputMetadataDto.getName(), outputMetadataDto);
                applicationEventPublisher.publishEvent(
                        new MetadataEvent(outputMetadataDto, EventType.UPDATED)
                );
                break;
            case APPROVED:
                outputMetadataDto = buildTestOutputMetadataDto(name, MetadataStatus.APPROVED);
                log.info("sending approved message, outputMetadataDto: %s".formatted(outputMetadataDto));
                stringOutputMetadataDtoHashMap.put(outputMetadataDto.getName(), outputMetadataDto);
                applicationEventPublisher.publishEvent(
                        new MetadataStatusEvent(outputMetadataDto, EventType.APPROVED)
                );
                break;
            case REJECTED:
                outputMetadataDto = buildTestOutputMetadataDto(name, MetadataStatus.REJECTED);
                log.info("sending rejected message, outputMetadataDto: %s".formatted(outputMetadataDto));
                stringOutputMetadataDtoHashMap.put(outputMetadataDto.getName(), outputMetadataDto);
                applicationEventPublisher.publishEvent(
                        new MetadataStatusEvent(outputMetadataDto, EventType.REJECTED)
                );
                break;
            case MODERATION_REQUESTED:
                outputMetadataDto = buildTestOutputMetadataDto(name, MetadataStatus.PENDING);
                log.info("sending message for moderation, outputMetadataDto: %s".formatted(outputMetadataDto));
                stringOutputMetadataDtoHashMap.put(outputMetadataDto.getName(), outputMetadataDto);
                applicationEventPublisher.publishEvent(
                        new MetadataModerationRequestedEvent(outputMetadataDto, EventType.MODERATION_REQUESTED)
                );
                break;
            case DELETED:
                outputMetadataDto = buildTestOutputMetadataDto(name, MetadataStatus.DRAFT);
                if (outputMetadataDto == null) {
                    return ResponseEntity.badRequest()
                            .body("no test item created for name: %s".formatted(name));
                }
                log.info("sending deleted message, outputMetadataDto: %s".formatted(outputMetadataDto));
                stringOutputMetadataDtoHashMap.remove(outputMetadataDto.getName());
                applicationEventPublisher.publishEvent(
                        new MetadataEvent(outputMetadataDto, EventType.DELETED)
                );
                break;
            case null, default:
                return ResponseEntity.badRequest()
                        .body("Event type must be one of: " + Arrays.toString(EventType.values()));
        }
        return ResponseEntity.ok("message sent correctly, outputMetadataDto: %s".formatted(outputMetadataDto));
    }

    private OutputMetadataDto buildTestOutputMetadataDto(String name, MetadataStatus status) {
        return OutputMetadataDto.builder()
                .id(UUID.randomUUID())
                .name(name)
                .status(status)
                .creator(OutputUserProfileDto.builder().id(UUID.randomUUID()).build())
                .build();
    }
}