package com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonValue;
import io.swagger.v3.oas.annotations.media.Schema;
import org.jetbrains.annotations.NotNull;
import org.springframework.core.convert.converter.Converter;
import org.springframework.stereotype.Component;

/**
 * Enumeration for filtering metadata by moderation status.
 * Only PENDING and REJECTED statuses are allowed for moderation filtering.
 */
@Schema(description = "Filter for moderation status (PENDING, REJECTED)")
public enum ModerationDmStatus {
    /**
     * Metadata pending moderation review.
     */
    PENDING("pending"),

    /**
     * Metadata that moderators have rejected.
     */
    REJECTED("rejected");

    private final String value;

    ModerationDmStatus(String value) {
        this.value = value;
    }

    @JsonValue
    public String getValue() {
        return value;
    }

    @Override
    public String toString() {
        return value;
    }

    /**
     * Converts this filter to its corresponding MetadataStatus.
     *
     * @return the corresponding MetadataStatus
     */
    public ModerationDmStatus toModerationDmStatus() {
        return switch (this) {
            case PENDING -> ModerationDmStatus.PENDING;
            case REJECTED -> ModerationDmStatus.REJECTED;
        };
    }

    @JsonCreator
    public static ModerationDmStatus fromString(String value) {
        if (value == null) {
            return null;
        }
        for (ModerationDmStatus filter : ModerationDmStatus.values()) {
            if (filter.value.equalsIgnoreCase(value)) {
                return filter;
            }
        }
        throw new IllegalArgumentException("Unknown ModerationStatus: " + value + ". Allowed values are: pending, rejected");
    }

    @Component
    public static class ModerationStatusConverter implements Converter<String, ModerationDmStatus> {
        @Override
        public ModerationDmStatus convert(@NotNull String source) {
            return ModerationDmStatus.fromString(source);
        }
    }
}