package com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.FetchType;
import jakarta.persistence.JoinColumn;
import jakarta.persistence.ManyToOne;
import jakarta.persistence.PostPersist;
import jakarta.persistence.PreUpdate;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.institutionalpage.dm.util.Constants.MAX_LONG_TEXT_LENGTH;

/**
 * Entity class representing a paragraph in the institutional-page service.
 * This entity is mapped to the "paragraph" table in the database
 * and stores information such as
 * the paragraph name, category, wpLead, creation time, and update time.
 *
 * <p>Example usage:
 * <pre>
 * ParagraphEntity paragraphEntity = ParagraphEntity.builder()
 *     .title("Outcomes")
 *     .content("A shared library of reusable software components")
 *     .build();
 * </pre>
 * </p>
 */
@Entity
@Table(name = "paragraph")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class ParagraphEntity extends AbstractUUIDEntity {

    /**
     * Association of the related institutional page for navigation.
     */
    @ManyToOne(fetch = FetchType.LAZY, optional = false)
    @JoinColumn(name = "institutional_page_id", updatable = false)
    private InstitutionalPageEntity institutionalPage;

    /**
     * ID of the related institutional page.
     * <p></p>
     * Had to add this column to avoid joins for findByInstitutionalPageId* queries.
     * <a href="https://github.com/spring-projects/spring-data-jpa/issues/1575">JPA Github Issue</a>
     */
    @Column(name = "institutional_page_id", insertable = false, updatable = false)
    private UUID institutionalPageId;

    /**
     * The title of the paragraph.
     */
    @Column(name = "title", nullable = false)
    private String title;

    /**
     * The content of the paragraph.
     */
    @Column(name = "content", length = MAX_LONG_TEXT_LENGTH)
    private String content;

    /**
     * The image url of the paragraph.
     */
    @Column(name = "image_url")
    private String imageUrl;

    /**
     * The resource url of the paragraph.
     */
    @Column(name = "resource_url")
    private String resourceUrl;

    /**
     * list of labels.
     */
    @Column(name = "labels")
    private List<String> labels;

    /**
     * The timestamp when the institutionalPage was created.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private Instant creationTime;

    /**
     * The timestamp when the institutionalPage was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     * </p>
     */
    @Version
    private long version;

    /**
     * Sets both {@code creationTime} and {@code updateTime} before the entity is persisted.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @Override
    public void prePersist() {
        super.prePersist();
        Instant currentTime = Instant.now().truncatedTo(ChronoUnit.MICROS);
        if (creationTime == null) {
            creationTime = currentTime;
        }
        updateTime = currentTime;
    }

    /**
     * Sets the {@code institutionalPageId} field after the entity has been persisted.
     * <p>
     * This is required during the initial creation of the entity, as Hibernate does not
     * populate fields when persisting.
     * </p>
     */
    @PostPersist
    public void postPersist() {
        institutionalPageId = institutionalPage.getId();
    }

    /**
     * Updates {@code updateTime} just before the entity is updated.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PreUpdate
    public void onUpdate() {
        updateTime = Instant.now().truncatedTo(ChronoUnit.MICROS);
    }

}
