package com.finconsgroup.itserr.marketplace.institutionalpage.dm.api;

import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputSearchPublishedInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.validation.ValidAssociationToLoad;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.Set;
import java.util.UUID;

/**
 * This interface defines the contract for REST API endpoints related to published InstitutionalPages.
 *
 * <p>
 * It provides endpoints for retrieving and searching published InstitutionalPages,
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * GET    /api/v1/dm/institutional-page/institutional-pages/published                              - Retrieve a paginated list of all Published InstitutionalPages (paginated).
 * POST   /api/v1/dm/institutional-page/institutional-pages/published/search                       - Retrieves a paginated list of all Published InstitutionalPages, matching the search criteria.
 * GET    /api/v1/dm/institutional-page/institutional-pages/published/{institutionalPageId}        - Retrieve a specific Published InstitutionalPage by id.
 * </pre>
 * </p>
 */
@Tag(
        name = "Publication",
        description = "The Publication API: it provides endpoints for " +
                "retrieving and searching published InstitutionalPages."
)
@SecurityRequirement(name = "BearerAuth")
public interface PublicationApi {

    /**
     * Retrieves a paginated list of all Published InstitutionalPages, matching the search criteria.
     *
     * @param inputSearchPublishedInstitutionalPageDto the dto containing the filters to be applied
     * @param associationsToLoad                       comma separated list of the associations to be returned (default is "all").
     * @param pageNumber                               the page number to retrieve (default is 0)
     * @param pageSize                                 the number of InstitutionalPages per page (default is 10)
     * @param sort                                     the field to sort by (default is "name")
     * @param direction                                the direction of sorting (default is ascending)
     * @return a page of {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieves a paginated list of all Published InstitutionalPages, matching the search criteria",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @PostMapping(
            value = "/institutional-page/institutional-pages/published/search",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    Page<OutputInstitutionalPageDto> searchPublishedInstitutionalPages(
            @Valid @RequestBody InputSearchPublishedInstitutionalPageDto inputSearchPublishedInstitutionalPageDto,
            @RequestParam(name = "associationsToLoad", defaultValue = "all", required = false) Set<@ValidAssociationToLoad String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves a paginated list of all Published InstitutionalPages.
     *
     * @param associationsToLoad comma separated list of the associations to be returned (default is "all").
     * @param pageNumber         the page number to retrieve (default is 0)
     * @param pageSize           the number of InstitutionalPages per page (default is 10)
     * @param sort               the field to sort by (default is "name")
     * @param direction          the direction of sorting (default is ascending)
     * @return a page of {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieves a paginated list of all Published InstitutionalPages",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(value = "/institutional-page/institutional-pages/published", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    Page<OutputInstitutionalPageDto> findAllPublishedInstitutionalPages(
            @RequestParam(name = "associationsToLoad", defaultValue = "all", required = false) Set<@ValidAssociationToLoad String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves a specific Published InstitutionalPage by id.
     *
     * @param institutionalPageId the id of the InstitutionalPage to retrieve
     * @return the found {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieves a specific Published InstitutionalPage by id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @GetMapping(
            value = "/institutional-page/institutional-pages/published/{institutionalPageId}",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputInstitutionalPageDto findPublishedInstitutionalPageById(
            @PathVariable("institutionalPageId") UUID institutionalPageId
    );

}
