package com.finconsgroup.itserr.marketplace.favouritesearch.bs.api;


import com.finconsgroup.itserr.marketplace.favouritesearch.bs.dto.InputCreateFavouriteSearchDto;
import com.finconsgroup.itserr.marketplace.favouritesearch.bs.dto.OutputFavouriteSearchDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;

import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;

import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.UUID;


/**
 * This interface defines the contract for REST API endpoints related to favourite searches management.
 *
 * <p>
 * It provides endpoints for creating favourite searches.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * POST     /api/v1/bs/favourite-searches - Create a new favouriteSearch
 * </pre>
 * </p>
 */
@Tag(
        name = "FavouriteSearch",
        description = "The FavouriteSearch API: it provides endpoints for " +
                "retrieving, creating, and deleting FavouriteSearch."
)
@SecurityRequirement(name = "BearerAuth")
public interface FavouriteSearchApi {

    /**
     * Creates a new favourite search for the authenticated user.
     *
     * @param inputCreateFavouriteSearchDto the input data transfer object containing the favourite search details
     * @return the created {@link OutputFavouriteSearchDto} and HTTP status 201 (Created)
     */
    @Operation(
            summary = "create a favouriteSearch for the authenticated user",
            responses = {@ApiResponse(responseCode = "201", description = "Created")}
    )
    @PostMapping(
            value = "/favourite-searches",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.CREATED)
    OutputFavouriteSearchDto create(
            @Valid @RequestBody InputCreateFavouriteSearchDto inputCreateFavouriteSearchDto
    );

    /**
     * Deletes a favourite search by id for the authenticated user.
     *
     * @param favouriteSearchId the id of the favouriteSearch to delete
     */
    @Operation(
            summary = "delete favouriteSearch by id for the authenticated user",
            responses = {
                    @ApiResponse(responseCode = "204", description = "No Content"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @ResponseStatus(HttpStatus.NO_CONTENT)
    @DeleteMapping(value = "/favourite-searches/{favouriteSearchId}", produces = MediaType.APPLICATION_JSON_VALUE)
    void deleteById(
            @PathVariable("favouriteSearchId") UUID favouriteSearchId
    );
}
