package com.finconsgroup.itserr.marketplace.echo.dm.api;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.Map;

/**
 * This interface defines the contract for REST API endpoints related to Orders Management.
 */
@Tag(
        name = "Orders",
        description = "The Orders API: it provides endpoints for " +
                "retrieving Orders."
)
@SecurityRequirement(name = "BearerAuth")
public interface OrderApi {

    /**
     * Find all users.
     *
     * @return a string response
     */
    @Operation(
            summary = "Find all users",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(value = "/users", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    String findAllUsers(@RequestHeader Map<String, String> headers);

    /**
     * Delete all users.
     *
     * @return a string response
     */
    @Operation(
            summary = "delete all users",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @DeleteMapping(value = "/users", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    String deleteAllUsers(@RequestHeader Map<String, String> headers);

    /**
     * Find user by user id.
     *
     * @return a string response
     */
    @Operation(
            summary = "Find user by user id",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(value = "/users/{userId}", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    String findUserByUserId(
            @PathVariable String userId,
            @RequestHeader Map<String, String> headers
    );

    /**
     * Delete user by user id.
     *
     * @return a string response
     */
    @Operation(
            summary = "Delete user by user id",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @DeleteMapping(value = "/users/{userId}", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    String deleteUserByUserId(
            @PathVariable String userId,
            @RequestHeader Map<String, String> headers
    );

    /**
     * Find all orders by user id.
     *
     * @return a string response
     */
    @Operation(
            summary = "Find all orders by user id",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(value = "/users/{userId}/orders", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    String findAllOrdersByUserId(
            @PathVariable String userId,
            @RequestHeader Map<String, String> headers
    );

    /**
     * Delete all orders by user id.
     *
     * @return a string response
     */
    @Operation(
            summary = "Delete all orders by user id",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @DeleteMapping(value = "/users/{userId}/orders", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    String deleteAllOrdersByUserId(
            @PathVariable String userId,
            @RequestHeader Map<String, String> headers
    );

    /**
     * Find order by user id and order id.
     *
     * @return a string response
     */
    @Operation(
            summary = "Find order by user id and order id",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(value = "/users/{userId}/orders/{orderId}", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    String findOrderByUserIdAndOrderId(
            @PathVariable String userId,
            @PathVariable String orderId,
            @RequestHeader Map<String, String> headers
    );

    /**
     * Delete order by user id and order id.
     *
     * @return a string response
     */
    @Operation(
            summary = "Delete order by user id and order id",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @DeleteMapping(value = "/users/{userId}/orders/{orderId}", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    String deleteOrderByUserIdAndOrderId(
            @PathVariable String userId,
            @PathVariable String orderId,
            @RequestHeader Map<String, String> headers
    );

}
