package com.finconsgroup.itserr.marketplace.discussion.bs.service;

import com.finconsgroup.itserr.marketplace.discussion.bs.dto.DiscussionDTO;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.InputUpdateDiscussionDto;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.ThreadDTO;
import com.finconsgroup.itserr.marketplace.discussion.bs.enums.ReactionType;
import com.finconsgroup.itserr.marketplace.discussion.bs.enums.ResourceType;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.ResponseEntity;

import java.time.Instant;

/**
 * Service contract for discussion-related operations.
 */
public interface DiscussionService {

    /**
     * Retrieves a paginated list of discussions.
     *
     * @param page page number (0-based)
     * @param size page size
     * @param sortBy sorting field
     * @param direction sort direction (asc|desc)
     * @param visibility optional visibility filter
     * @param includeThreads whether to include discussion threads
     * @param resourceType optional filter by the type of the resource
     * @param createdAfter optional filter for created after
     * @return page of discussions
     */
    Page<DiscussionDTO> getDiscussions(int page,
                                       int size,
                                       String sortBy,
                                       String direction,
                                       String visibility,
                                       boolean includeThreads,
                                       String resourceType,
                                       Instant createdAfter);

    Page<ThreadDTO> getThreadsByDiscussionId(String discussionId, int page, int size, String sortBy, Sort.Direction direction);

    /**
     * Creates a new discussion using the provided payload.
     *
     * @param discussionDTO discussion payload
     * @return the created {@link DiscussionDTO}
     */
    DiscussionDTO createDiscussion(DiscussionDTO discussionDTO);

    /**
     * Creates a new discussion using the provided payload for the event.
     *
     * @param resourceId the resource identifier
     * @param eventData  the event payload
     * @param eventType  the event type
     */
    void createDiscussionFromEvent(String resourceId, Object eventData, String eventType);

    /**
     * Updates an existing discussion with the provided data.
     *
     * @param id the unique identifier of the discussion to be updated
     * @param discussionDTO the payload containing the updated discussion fields
     * @return the updated discussion data wrapped in a response entity
     */
    DiscussionDTO updateDiscussion(String id, InputUpdateDiscussionDto discussionDTO);

    /**
     * Deletes a discussion by its identifier.
     *
     * @param id the discussion identifier
     * @return response entity indicating the outcome (typically 204 No Content)
     */
    ResponseEntity<Void> deleteDiscussion(String id);

    /**
     * Fetches a discussion by its identifier.
     *
     * @param id the discussion identifier
     * @return discussion data response
     */
    DiscussionDTO getDiscussion(String id);

    /**
     * Adds a new thread to an existing discussion.
     *
     * @param discussionId the discussion identifier
     * @param threadDTO    the thread payload to add
     * @return response entity indicating the outcome (typically 201 Created)
     */
    ResponseEntity<ThreadDTO> addThread(String discussionId, ThreadDTO threadDTO);

    /**
     * Deletes a thread from a discussion.
     *
     * @param discussionId the discussion identifier
     * @param threadId     the thread identifier
     * @return response entity indicating the outcome (typically 204 No Content)
     */
    ResponseEntity<Void> deleteThread(String discussionId, String threadId);

    /**
     * Updates a specific thread within a discussion.
     *
     * @param discussionId the discussion identifier
     * @param threadId     the thread identifier
     * @param threadDTO    the updated thread payload
     * @return response entity indicating the outcome (e.g., 200 OK or 204 No Content)
     */
    ResponseEntity<ThreadDTO> updateThread(String discussionId, String threadId, ThreadDTO threadDTO);

    /**
     * Adds a reaction to a discussion by a user.
     *
     * @param discussionId the discussion identifier
     * @param userId       the user applying the reaction
     * @param reactionType the reaction to apply
     * @return response entity indicating the outcome (e.g., 200 OK or 201 Created)
     */
    ResponseEntity<Void> addReactionToDiscussion(String discussionId, String userId, ReactionType reactionType);

    /**
     * Removes a user's reaction from a discussion.
     *
     * @param discussionId the discussion identifier
     * @param userId       the user whose reaction should be removed
     * @return response entity indicating the outcome (typically 204 No Content)
     */
    ResponseEntity<Void> removeReactionFromDiscussion(String discussionId, String userId);

    /**
     * Adds a reaction to a specific thread within a discussion.
     *
     * @param discussionId the discussion identifier
     * @param threadId     the thread identifier
     * @param userId       the user applying the reaction
     * @param reactionType the reaction to apply
     * @return response entity indicating the outcome (e.g., 200 OK or 201 Created)
     */
    ResponseEntity<Void> addReactionToThread(String discussionId, String threadId, String userId, ReactionType reactionType);

    /**
     * Removes a user's reaction from a thread.
     *
     * @param discussionId the discussion identifier
     * @param threadId     the thread identifier
     * @param userId       the user whose reaction should be removed
     * @return response entity indicating the outcome (typically 204 No Content)
     */
    ResponseEntity<Void> removeReactionFromThread(String discussionId, String threadId, String userId);

    DiscussionDTO getDiscussionForResource(String resourceId, ResourceType resourceType);
}
