package com.finconsgroup.itserr.marketplace.userprofile.bs.client;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputAddProjectToUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputAdminPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputFindUserProfilesByInterestsDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputFindUserProfilesByPrincipalsDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputPatchUserProfileProjectDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputRemoveProjectFromUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.OutputAdminPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.OutputUserProfileFolderDetails;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.InputPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.InputUpdateUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.InputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputEndorsementAcknowledgementDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputUserProfileDto;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import jakarta.validation.Valid;
import org.springframework.data.domain.Page;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.List;
import java.util.UUID;

/**
 * Feign client for communication with the UserProfile DM service.
 */
public interface UserProfileDmClient {

    /**
     * Retrieves the info of the user profile folder.
     *
     * @return the info of the profile folder
     * @throws feign.FeignException.FeignClientException.InternalServerError if an internal error occurs in the DM service
     */
    @GetMapping("/profile/folder")
    OutputUserProfileFolderDetails getProfileFolder();

    /**
     * Patch the user profile info.
     *
     * @return the user profile patched info
     * @throws feign.FeignException.FeignClientException.InternalServerError if an internal error occurs in the DM service
     */
    @PatchMapping("/profile")
    OutputPatchUserProfileDto patchUserProfile(InputPatchUserProfileDto inputPatchUserProfileDto);

    /**
     * Creates a new userprofile.
     *
     * @param inputUserProfileDto the input data transfer object containing userprofile details
     * @return the created {@link OutputUserProfileDto} and HTTP status 201 (Created)
     */
    @PostMapping("/profile")
    OutputUserProfileDto create(InputUserProfileDto inputUserProfileDto);

    /**
     * Updates an existing userprofile by id.
     *
     * @param inputUpdateUserProfileDto the input data transfer object containing updated userprofile details
     * @return the updated {@link OutputUserProfileDto} and HTTP status 200 (OK)
     */
    @PutMapping("/profile")
    OutputUserProfileDto updateById(InputUpdateUserProfileDto inputUpdateUserProfileDto);

    /**
     * Retrieves an userprofile by id using JWT token.
     *
     * @return the found {@link OutputUserProfileDto} and HTTP status 200 (OK)
     */
    @GetMapping(value = "/profile", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputUserProfileDto findById();

    /**
     * Retrieves a paginated list of all user profiles for provided principals.
     *
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of UserProfiles per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @return the list of found {@link OutputUserProfileDto}s and HTTP status 200 (OK)
     */
    @PostMapping(value = "/profiles-by-principals", consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    Page<OutputUserProfileDto> findAllByPrincipals(
            @RequestBody InputFindUserProfilesByPrincipalsDto inputFindUserProfilesByPrincipalsDto,
            @RequestParam(name = "pageNumber") int pageNumber,
            @RequestParam(name = "pageSize") int pageSize,
            @RequestParam(name = "sort") String sort,
            @RequestParam(name = "direction") SortDirection direction);

    /**
     * add projects to existing userprofile for given ids.
     *
     * @param inputAddProjectToUserProfilesDto contains the list of userIds and object containing updated userprofile project details
     * @return the updated {@link List<OutputUserProfileDto>} and HTTP status 200 (OK)
     */
    @PostMapping(
            value = "/profiles/project",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    List<OutputUserProfileDto> addProjectToUserProfiles(
            @RequestBody InputAddProjectToUserProfilesDto inputAddProjectToUserProfilesDto
    );

    /**
     * removes projects for existing userprofile for given ids.
     *
     * @param inputRemoveProjectFromUserProfilesDto contains the list of userIds and object containing updated userprofile project details
     * @return the updated {@link List<OutputUserProfileDto>} and HTTP status 200 (OK)
     */
    @DeleteMapping(
            value = "profiles/project",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    List<OutputUserProfileDto> removeProjectFromUserProfiles(
            @RequestBody InputRemoveProjectFromUserProfilesDto inputRemoveProjectFromUserProfilesDto
    );

    /**
     * Retrieves an userprofile by profileId.
     *
     * @param profileId of the userprofile to fetch
     * @return the found {@link OutputUserProfileDto} and HTTP status 200 (OK)
     */
    @GetMapping(value = "/profiles/{profileId}", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputUserProfileDto getById(@PathVariable("profileId") UUID profileId);

    /**
     * update project for existing userprofile for given ids.
     *
     * @param inputPatchUserProfileProjectDto contains the list of userIds and patch updates for userprofile project details
     * @return the updated {@link List<OutputUserProfileDto>} and HTTP status 200 (OK)
     */
    @PatchMapping(
            value = "/profiles/projects",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    List<OutputUserProfileDto> patchUserProfileProject(
            @Valid @RequestBody InputPatchUserProfileProjectDto inputPatchUserProfileProjectDto
    );

    /**
     * Updates the public visibility settings of a user profile by admin
     *
     * @param profileId                     id of the user profile
     * @param inputAdminPatchUserProfileDto which contains information that needs to be updated for an existing entity
     * @return the found {@link OutputAdminPatchUserProfileDto} and HTTP status 200 (OK)
     */
    @PatchMapping(value = "/admin/profile/{profileId}", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputAdminPatchUserProfileDto patchUserProfile(@PathVariable("profileId") UUID profileId,
                                                    @RequestBody InputAdminPatchUserProfileDto inputAdminPatchUserProfileDto);

    /**
     * Retrieves a paginated list of all UserProfiles.
     *
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of UserProfiles per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @return a page of {@link OutputUserProfileDto} and HTTP status 200 (OK)
     */
    @GetMapping(value = "/profiles", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputPageDto<OutputUserProfileDto> findAll(
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "id", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) SortDirection direction
    );

    /**
     * activate the user profile.
     */
    @PostMapping(value = "/profile/activate", produces = MediaType.APPLICATION_JSON_VALUE)
    void activate();

    /**
     * deactivate the user profile.
     */
    @DeleteMapping(value = "/profile/activate", produces = MediaType.APPLICATION_JSON_VALUE)
    void deactivate();

    /**
     * Retrieves the list of user profile ids whose interests match at least one the provided list of strings.
     *
     * @param dto the input to match interests against
     * @return the matching {@link List} of user profile ids
     */
    @PostMapping(value = "/profile-ids-by-interests", consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    List<UUID> findMatchingInterests(@Valid @RequestBody InputFindUserProfilesByInterestsDto dto);

    /**
     * Endorses a specific expertise of a user profile.
     *
     * @param profileId   the ID of the user profile receiving the endorsement
     * @param expertiseId the ID of the expertise being endorsed
     * @return a {@link OutputEndorsementAcknowledgementDto} representing the endorsement acknowledgement
     */
    @PostMapping(value = "/profiles/{profileId}/expertise/{expertiseId}/confirm", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputEndorsementAcknowledgementDto addEndorsement(@PathVariable("profileId") UUID profileId, @PathVariable("expertiseId") UUID expertiseId);

    /**
     * Delete an existing endorsement.
     *
     * @param profileId   the ID of the user profile receiving the endorsement
     * @param expertiseId the ID of the expertise being endorsed
     */
    @DeleteMapping(value = "/profiles/{profileId}/expertise/{expertiseId}/confirm", produces = MediaType.APPLICATION_JSON_VALUE)
    void removeEndorsement(@PathVariable("profileId") UUID profileId, @PathVariable("expertiseId") UUID expertiseId);

}
