package com.finconsgroup.itserr.marketplace.search.dm.api;

import com.finconsgroup.itserr.marketplace.core.web.utils.OpenApiUtils;
import com.finconsgroup.itserr.marketplace.core.web.validation.annotation.ValidPutIdParameters;
import com.finconsgroup.itserr.marketplace.core.web.validation.annotation.ValidQueryFilter;
import com.finconsgroup.itserr.marketplace.search.dm.constant.FilterConfigKeys;
import com.finconsgroup.itserr.marketplace.search.dm.dto.InputEventDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputEventDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputEventLocalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotNull;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import jakarta.validation.constraints.Size;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.List;

/**
 * APIs for managing and querying the documents in event search index.
 */
@Tag(name = "Event", description = "Operations related to event search")
@Validated
@SecurityRequirement(name = "BearerAuth")
@ApiResponses(value = {
        @ApiResponse(responseCode = "500", ref = OpenApiUtils.RESPONSE_INTERNAL_SERVER_ERROR)
})
public interface EventApi {

    /**
     * Creates a new document or updates an existing document entry in the event search index.
     *
     * @param dto the input data transfer object containing index entry details
     * @return the created/updated {@link OutputEventDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Create/Update document in Event Search index",
            description = "Create a new document or update an existing document entry in the Event Search index using " +
                    "the provided dto. It also performs validations and returns errors in case of validation failures"
    )
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "OK"),
            @ApiResponse(responseCode = "400", ref = OpenApiUtils.RESPONSE_BAD_REQUEST)
    })
    @ResponseStatus(HttpStatus.OK)
    @PutMapping(value = "/event/{id}", produces = MediaType.APPLICATION_JSON_VALUE,
            consumes = MediaType.APPLICATION_JSON_VALUE)
    @ValidPutIdParameters
    OutputEventDto upsertEvent(@PathVariable("id") String id, @Valid @RequestBody InputEventDto dto);

    /**
     * Fetch document entry from the event search index.
     *
     * @param id the id of the document to fetch
     * @return the found {@link OutputEventDto} and HTTP status 200 (Ok)
     */
    @Operation(
            summary = "Fetches document from Event Search index",
            description = "Fetches a document entry in the Event Search index using the provided id."
                    + "It returns not found response in case if the id does not exist"
    )
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "OK"),
            @ApiResponse(responseCode = "404", ref = OpenApiUtils.RESPONSE_NOT_FOUND)
    })
    @GetMapping(value = "/event/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputEventDto getEvent(@PathVariable("id") String id);

    /**
     * Delete document entry from the event search index.
     *
     * @param id the id of the document to delete
     *           Returns HTTP status 204 (No Content), if the document is deleted successfully
     */
    @Operation(
            summary = "Deletes document from Event Search index",
            description = "Deletes a document entry in the Event Search index using the provided id."
                    + "It returns not found response in case if the id does not exist"
    )
    @ApiResponses(value = {
            @ApiResponse(responseCode = "204", description = "No Content"),
            @ApiResponse(responseCode = "404", ref = OpenApiUtils.RESPONSE_NOT_FOUND)
    })
    @DeleteMapping("/event/{id}")
    @ResponseStatus(HttpStatus.NO_CONTENT)
    void deleteEvent(@PathVariable("id") String id);

    /**
     * Deletes all the documents from the event search index.
     * Returns HTTP status 204 (No Content), if the documents are deleted successfully
     */
    @Operation(
            summary = "Deletes all the documents from Event Search index",
            description = "Deletes all the documents in the Event Search index.",
            // Hide from Swagger, as this API should not be normally used by any other service
            hidden = true
    )
    @ApiResponses(value = {
            @ApiResponse(responseCode = "204", description = "No Content")
    })
    // Kept the endpoint url purposefully different from the delete by id url, to avoid accidental execution if any
    // item has the id = all
    @DeleteMapping("/event-all")
    @ResponseStatus(HttpStatus.NO_CONTENT)
    void deleteAll();

    /**
     * Retrieves a limited list of autocompletion results matching the provided terms from Event Search index.
     *
     * @param terms the terms to search for
     * @return the list of {@link OutputGlobalSearchAutoCompleteDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Perform event search for autocompletion and retrieve the results",
            description = "Perform event search for autocompletion based on the terms provided and retrieve results"
    )
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "OK"),
            @ApiResponse(responseCode = "400", ref = OpenApiUtils.RESPONSE_BAD_REQUEST)
    })
    @ResponseStatus(HttpStatus.OK)
    @GetMapping(value = "/event/auto-complete", produces = MediaType.APPLICATION_JSON_VALUE)
    List<OutputGlobalSearchAutoCompleteDto> getAutoCompletions(
            @NotNull @Size(min = 3) @RequestParam("terms") String terms
    );

    /**
     * Retrieves a limited list of local search results matching the provided terms and filters from Event Search index.
     *
     * @param terms      the terms to search for
     * @param filters    the additional filters to apply in the format key1=value1,value2&key2=value3,value4
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of event documents per page (default is 10)
     * @param sort       the field to sort by (default is "relevance")
     * @param direction  the direction of sorting (default is ascending)
     * @param multiSort  the multiple fields to sort by, takes preference over sort (default is "")
     * @return a page of {@link OutputEventLocalSearchDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Perform event local search and retrieve the results",
            description = "Perform the event local search based on the terms and filters provided and retrieve results",
            parameters = {
                    @Parameter(name = "terms", description = "The terms to search for"),
                    @Parameter(
                            name = "filters",
                            description = "The additional filters to apply in the format key1=value1,value2&key2=value3,value4. " +
                                    "In order to view only the events created by user, pass the filter mineOnly=true along with" +
                                    "any additional filters as applicable."
                    ),
                    @Parameter(name = "pageNumber", description = "The page index, starting from 0"),
                    @Parameter(name = "pageSize", description = "The number of items per page"),
                    @Parameter(name = "sort", description = "The field by which to sort the results"),
                    @Parameter(name = "direction", description = "The sorting direction: ASC or DESC"),
                    @Parameter(name = "multiSort", description = "The multiple fields to sort by, takes preference over sort in" +
                            " the format field1,field2:desc,field3")
            }
    )
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "OK"),
            @ApiResponse(responseCode = "400", ref = OpenApiUtils.RESPONSE_BAD_REQUEST)
    })
    @ResponseStatus(HttpStatus.OK)
    @GetMapping(value = "/event/search", produces = MediaType.APPLICATION_JSON_VALUE)
    Page<OutputEventLocalSearchDto> getLocalSearch(
            @Size(min = 3) @RequestParam(value = "terms", required = false) String terms,
            @RequestParam(value = "filters", required = false) @ValidQueryFilter(configKey = FilterConfigKeys.EVENT) String filters,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "relevance", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "DESC", required = false) Sort.Direction direction,
            @RequestParam(name = "multiSort", defaultValue = "", required = false) String multiSort
    );

}
