package com.finconsgroup.itserr.marketplace.search.dm.controller;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2AuthorizationException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ExecutionException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.search.dm.api.DiagnosticsApi;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputIngestDataDto;
import com.finconsgroup.itserr.marketplace.search.dm.enums.Category;
import com.finconsgroup.itserr.marketplace.search.dm.service.CatalogService;
import com.finconsgroup.itserr.marketplace.search.dm.service.DiscussionService;
import com.finconsgroup.itserr.marketplace.search.dm.service.EventService;
import com.finconsgroup.itserr.marketplace.search.dm.service.InstitutionalPageService;
import com.finconsgroup.itserr.marketplace.search.dm.service.NewsService;
import com.finconsgroup.itserr.marketplace.search.dm.service.UserProfileService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.appender.RollingFileAppender;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.lang.NonNull;
import org.springframework.web.bind.annotation.RestController;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Function;

/**
 * REST controller class for handling debugging/analysis related API requests.
 *
 * <p>Implements the {@link DiagnosticsApi} interface.</p>
 */
@Slf4j
@RestController
@RequiredArgsConstructor
public class DiagnosticsController implements DiagnosticsApi {

    private final CatalogService catalogService;
    private final UserProfileService userProfileService;
    private final InstitutionalPageService institutionalPageService;
    private final EventService eventService;
    private final NewsService newsService;
    private final DiscussionService discussionService;
    private final ObjectMapper objectMapper;

    // The name of the RollingFile Log4j2 component
    // It is not the name of the log file
    // We use this to dynamically retrieve the name of the log file.
    @Value("${log.log4j2-rolling-file-name}")
    private String log4j2RollingFileName;

    @Value("${search.dm.diagnostics.ingest-data-enabled}")
    private boolean ingestDataEnabled;

    @Override
    public ResponseEntity<Resource> downloadLogs() {
        log.debug("call to DiagnosticsController - downloadLogs");
        Path filePath = Paths.get(getLogFilePathFromLog4j2());
        log.debug("Trying to retrieve log file from: {}", filePath);
        File logFile = filePath.toFile();
        if (!logFile.exists() || !logFile.isFile()) {
            throw new WP2ResourceNotFoundException("Log file not found: %s".formatted(logFile.getAbsolutePath()));
        }
        Resource resource = new FileSystemResource(logFile);
        return ResponseEntity.ok()
                .header(HttpHeaders.CONTENT_DISPOSITION, getContentDispositionHeaderValue(logFile.getName()))
                .contentType(MediaType.APPLICATION_OCTET_STREAM)
                .body(resource);
    }

    @Override
    public ResponseEntity<OutputIngestDataDto> ingestData(Category category, String body) {
        if (!ingestDataEnabled) {
            throw new WP2AuthorizationException("Forbidden");
        }

        var outputIngestData = switch (category) {
            case CATALOG -> ingestData(body, catalogService::upsertDocument, new TypeReference<>() {
            });
            case PEOPLE -> ingestData(body, userProfileService::upsertDocument, new TypeReference<>() {
            });
            case INSTITUTIONAL_PAGE ->
                    ingestData(body, institutionalPageService::upsertDocument, new TypeReference<>() {
                    });
            case EVENT -> ingestData(body, eventService::upsertDocument, new TypeReference<>() {
            });
            case NEWS -> ingestData(body, newsService::upsertDocument, new TypeReference<>() {
            });
            case DISCUSSION -> ingestData(body, discussionService::upsertDocument, new TypeReference<>() {
            });
        };
        return ResponseEntity.ok(outputIngestData);
    }

    // private

    public String getLogFilePathFromLog4j2() {
        LoggerContext context = (LoggerContext) LogManager.getContext(false);
        RollingFileAppender appender = context.getConfiguration().getAppender(log4j2RollingFileName);
        return appender.getFileName();
    }

    private static String getContentDispositionHeaderValue(String fileName) {
        return "attachment; filename=\"%s\"".formatted(fileName);
    }

    private <T, R> OutputIngestDataDto ingestData(@NonNull final String body,
                                                  @NonNull final Function<T, R> upsertFunction,
                                                  @NonNull final TypeReference<List<T>> typeReference) {
        try {
            final List<T> dtoList = objectMapper.readValue(body, typeReference);
            int totalCount = 0;
            int failedCount = 0;
            final int size = dtoList.size();
            final List<String> errors = new LinkedList<>();
            for (int i = 0; i < size; i++) {
                try {
                    upsertFunction.apply(dtoList.get(i));
                } catch (Exception e) {
                    failedCount++;
                    log.error("Failed to process record at index [{}] - {}", i, e.getMessage(), e);
                    errors.add(String.format("Failed to process record at index [%d] - %s",
                            i, ExceptionUtils.getRootCause(e)));
                }
                totalCount++;
            }
            return OutputIngestDataDto.builder()
                    .totalCount(totalCount)
                    .failedCount(failedCount)
                    .successfulCount(totalCount - failedCount)
                    .errors(errors)
                    .build();
        } catch (JsonProcessingException e) {
            throw new WP2ExecutionException(e);
        }
    }

}