package com.finconsgroup.itserr.marketplace.search.bs.messaging;

import com.finconsgroup.itserr.marketplace.search.bs.client.dm.SearchDmClient;
import com.finconsgroup.itserr.marketplace.search.bs.config.properties.MessagingUserProfileBsProperties;
import com.finconsgroup.itserr.marketplace.search.bs.config.properties.SearchBsConfigurationProperties;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Objects;

/**
 * Concrete implementation of {@link AbstractResourceSubscriber} for handling user-profile-related resource events.
 */
@Component("userProfileResourceSubscriber")
@Slf4j
public class UserProfileResourceSubscriber extends AbstractResourceSubscriber {

    /**
     * Messaging configuration properties for user-profile-bs events.
     */
    private final MessagingUserProfileBsProperties messagingUserProfileBsProperties;

    /**
     * The client used to interact with the search dm.
     */
    private final SearchDmClient searchDmClient;

    /**
     * Constructs the UserProfileResourceSubscriber
     *
     * @param searchBsConfigurationProperties the application configuration properties
     * @param searchDmClient                  http client for interacting with the search DM service
     */
    public UserProfileResourceSubscriber(
            SearchBsConfigurationProperties searchBsConfigurationProperties,
            SearchDmClient searchDmClient
    ) {
        this.messagingUserProfileBsProperties = searchBsConfigurationProperties.getMessaging().getUserProfileBs();
        this.searchDmClient = searchDmClient;
    }

    /**
     * Processes a user profile resource event by invoking the appropriate Search DM operation.
     *
     * @param resourceId      the unique resource identifier
     * @param resourcePayload the resource data payload
     * @param eventType       the type of the event
     */
    @Override
    protected void handleResource(String resourceId, Object resourcePayload, String eventType) {
        log.debug("Processing user profile event: id: {}, payload: {}, type: {}", eventType, resourceId, resourcePayload);
        if (Objects.equals(eventType, messagingUserProfileBsProperties.getResourceCreatedEventType()) ||
                Objects.equals(eventType, messagingUserProfileBsProperties.getResourceUpdatedEventType())) {
            searchDmClient.upsertUserProfile(resourceId, resourcePayload);
        } else if (Objects.equals(eventType, messagingUserProfileBsProperties.getResourceDeletedEventType())) {
            searchDmClient.deleteUserProfile(resourceId);
        } else {
            log.error("Unsupported eventType: {} for resourceId: {}", eventType, resourceId);
        }
    }
}
